/**
 * *************************************************************************
 * Copyright (C) 2011 by Paul Lutus * lutusp@arachnoid.com * * This program is
 * free software; you can redistribute it and/or modify * it under the terms of
 * the GNU General Public License as published by * the Free Software
 * Foundation; either version 2 of the License, or * (at your option) any later
 * version. * * This program is distributed in the hope that it will be useful,
 * * but WITHOUT ANY WARRANTY; without even the implied warranty of *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the * GNU General
 * Public License for more details. * * You should have received a copy of the
 * GNU General Public License * along with this program; if not, write to the *
 * Free Software Foundation, Inc., * 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA. *
 * *************************************************************************
 */

/*
 * BiquadDesigner.java
 *
 * Created on Mar 21, 2011, 1:52:25 PM
 */
package biquaddesigner;

import java.awt.*;
import java.util.*;
import java.util.Timer;
import java.util.TimerTask;
import javax.swing.*;
import java.awt.datatransfer.Clipboard;
import java.awt.datatransfer.StringSelection;
import java.awt.datatransfer.Transferable;

/**
 *
 * @author lutusp
 */
final public class BiQuadDesignerPanel extends javax.swing.JPanel {

    boolean debug = false;
    java.util.List<String> filter_names = Arrays.asList("Lowpass", "Highpass", "Bandpass", "Notch", "Peak", "Lowshelf", "Highshelf");
    java.util.List<String> filter_tips = Arrays.asList(
            "Gain has no effect, Q works",
            "Default Q setting = 70.7, gain has no effect",
            "Default Q setting = 70.7, gain has no effect",
            "Gain has no effect, Q works",
            "Gain and Q work",
            "Gain works, Q has no effect",
            "Gain works, Q has no effect");
    java.util.List<String> gain_vals = Arrays.asList("1", "2", "5", "10", "20", "50", "100");
    HashMap<String, Integer> filter_hash;
    HashMap<Integer, String> name_hash;
    HashMap<String, String> tips_hash;
    Object parent;
    GraphPanel gpanel;
    TextFieldController q, filter_gain, center_frequency, graph_start_frequency, graph_end_frequency, sample_rate;
    double y_axis_gain = 1;
    ComboBoxTextController filter_type;
    ToggleButtonController db_mode;
    TabPanelController selected_tab;
    Timer periodic_timer = null;
    int timer_period_ms = 250;
    BiQuadraticFilter filter = null;
    int type = -1;
    double v_q, v_filter_gain, v_y_axis_gain, v_cf, v_sr, v_start, v_end;
    double v_sr_old = -1;
    boolean v_dbmode = false;
    double[] constants;

    /**
     * Creates new form BiquadDesigner
     */
    public BiQuadDesignerPanel(Object p, String[] args) {
        parent = p;
        if (args.length > 0 && args[0].equals("-d")) {
            debug = true;
        }
        initComponents();
        quit_button.setVisible(parent instanceof JFrame);
        help_button.setVisible(parent instanceof JFrame);
        filter_hash = new HashMap<String, Integer>() {
            // this actually works

            {
                put("Lowpass", BiQuadraticFilter.LOWPASS);
                put("Highpass", BiQuadraticFilter.HIGHPASS);
                put("Bandpass", BiQuadraticFilter.BANDPASS);
                put("Peak", BiQuadraticFilter.PEAK);
                put("Notch", BiQuadraticFilter.NOTCH);
                put("Highshelf", BiQuadraticFilter.HIGHSHELF);
                put("Lowshelf", BiQuadraticFilter.LOWSHELF);
            }
        };
        name_hash = new HashMap<Integer, String>();
        for (String s : filter_hash.keySet()) {
            name_hash.put(filter_hash.get(s), s);
        }
        tips_hash = new HashMap<String, String>();
        int i = 0;
        for (String s : filter_names) {
            tips_hash.put(s, filter_tips.get(i++));
        }
        filter = new BiQuadraticFilter();
        gpanel = new GraphPanel(this);
        graph_panel_tab.add(gpanel, BorderLayout.CENTER);
        setup_controls();
        periodic_timer = new java.util.Timer();
        periodic_timer.scheduleAtFixedRate(new PeriodicEvents(), 500, timer_period_ms);
    }

    class PeriodicEvents extends TimerTask {

        public void run() {
            update_controls();
            update_description();
        }
    }

    private void update_description() {
        if (filter != null) {
            StringBuilder sb = new StringBuilder();
            sb.append("Filter Description\n");
            String name = name_hash.get(filter.type);
            sb.append(String.format("   %-14s\n", "Type"));
            format_const(sb, "", name);
            sb.append(String.format("   %-14s\n", "Notes"));
            format_const(sb, "", tips_hash.get(name));
            sb.append(String.format("   %-14s\n", "Constants"));
            double[] constants = filter.constants();
            for (int i = 1; i < 6; i++) {
                double v = constants[i - 1];
                String s = String.format("%s%d", (i < 3) ? "a" : "b", i % 3);
                format_const(sb, s, v);
            }

            sb.append(String.format("   %-14s\n", "User Settings"));
            format_const(sb, "Q", q.get_dvalue());
            format_const(sb, "Filter Gain", filter_gain.get_dvalue());
            format_const(sb, "Center Freq. Hz", center_frequency.get_dvalue());
            format_const(sb, "Sample Rate", sample_rate.get_dvalue());
            sb.append(String.format("   %-14s\n", "Graph Settings"));
            format_const(sb, "Start Hz", graph_start_frequency.get_dvalue());
            format_const(sb, "End Hz", graph_end_frequency.get_dvalue());
            //format_const(sb, "Graph Vert. Gain", y_axis_gain.get_dvalue());

            format_const(sb, "Display Mode", db_mode.get_value() ? "DB" : "Linear");
            sb.append("\nHome page: http://arachnoid.com/BiQuadDesigner");
            sb.append("\nCopyright (C) 2011, P. Lutus — http://arachnoid.com");
            description_text.setText(sb.toString());

        }
    }

    private void format_const(StringBuilder sb, String name, double v) {
        String s = String.format("%g", v);
        int p = s.indexOf(".");
        p = (p == -1) ? s.length() : p;
        while (p++ < 6) {
            s = " " + s;
        }
        sb.append(String.format("      %-16s = %s\n", name, s));
    }

    private void format_const(StringBuilder sb, String name, String v) {
        if (name.length() > 0) {
            sb.append(String.format("      %-16s = %s\n", name, v));
        } else {
            sb.append(String.format("      %4s\n", v));
        }
    }

    private void update_controls() {
        if (type == -1 || need_new_filter()) {
            new_filter();
        }
    }

    private void new_filter() {
        type = filter_hash.get(filter_type.get_value());
        v_q = q.get_dvalue();
        v_filter_gain = filter_gain.get_dvalue();
        v_y_axis_gain = 1;//y_axis_gain.get_dvalue();
        v_cf = center_frequency.get_dvalue();
        v_sr = sample_rate.get_dvalue();
        // force end freq to be rate / 2
        if (v_sr != 0 && v_sr != v_sr_old) {
            graph_start_frequency.set_value(0);
            graph_end_frequency.set_value(v_sr / 2);
            v_sr_old = v_sr;
        }
        if (v_cf > v_sr / 2) {
            v_cf = v_sr / 2;
            center_frequency.set_value(v_cf);
        }
        filter.configure(type, v_cf, v_sr, v_q, v_filter_gain);
        constants = filter.constants();
        v_dbmode = db_mode.get_value();
        v_start = graph_start_frequency.get_dvalue();
        v_end = graph_end_frequency.get_dvalue();
        repaint();
    }

    private boolean need_new_filter() {
        boolean need = false;
        need |= type != filter_hash.get(filter_type.get_value());
        need |= v_q != q.get_pct_dvalue();
        need |= v_filter_gain != filter_gain.get_dvalue();
        //need |= v_y_axis_gain != y_axis_gain.get_dvalue();
        need |= v_cf != center_frequency.get_dvalue();
        need |= v_sr != sample_rate.get_dvalue();
        need |= v_dbmode != db_mode.get_value();
        need |= v_start != graph_start_frequency.get_dvalue();
        need |= v_end != graph_end_frequency.get_dvalue();
        return need;
    }

    private void setup_controls() {
        int max = 100000000;
        q = new TextFieldController(q_textfield, "0.707", max, 0);
        filter_gain = new TextFieldController(gain_textfield, "1", max, -max);
        center_frequency = new TextFieldController(cf_textfield, "1000", max, 0);
        graph_start_frequency = new TextFieldController(startf_textfield, "500", max, 0);
        graph_end_frequency = new TextFieldController(endf_textfield, "1500", max, 0);
        sample_rate = new TextFieldController(rate_textfield, "40000", max, 0);
        filter_type = new ComboBoxTextController(filter_type_combobox, filter_names, "Bandpass", filter_tips, "Select filter type");
        //y_axis_gain = new ComboBoxController(y_axis_gain_combobox, gain_vals, "1");
        db_mode = new ToggleButtonController(db_radiobutton, false);
        selected_tab = new TabPanelController(tabbed_pane, 0);
        v_dbmode = db_mode.get_value();
        db_radiobutton.setText("<html>20 log<sub>10</sub></html>");
    }

    private void set_defaults() {
        if (ask_user(this, "Okay to reset to defaults?", "Set Program Defaults")) {
            q.set_value("0.707");
            filter_gain.set_value("1");
            center_frequency.set_value("10000");
            graph_start_frequency.set_value("0");
            graph_end_frequency.set_value("20000");
            sample_rate.set_value("40000");
            filter_type.set_value("Bandpass");
            //y_axis_gain.set_value("1");
            db_mode.set_value(false);
            selected_tab.set_value("0");
        }
    }

    private void launch_help() {
        try {
            java.awt.Desktop.getDesktop().browse(java.net.URI.create("http://arachnoid.com/BiQuadDesigner"));
        } catch (Exception e) {
            p(e);
        }
    }

    public static boolean ask_user(JComponent src, String query, String title, Object[] options) {
        if (options != null) {
            return (JOptionPane.showOptionDialog(src, query, title,
                    JOptionPane.YES_NO_OPTION, JOptionPane.QUESTION_MESSAGE, null, options, options[0]) == 0);
        } else {
            return (JOptionPane.showConfirmDialog(src, query, title, JOptionPane.YES_NO_OPTION) == JOptionPane.YES_OPTION);
        }

    }

    public static boolean ask_user(JComponent src, String query, String title) {
        return ask_user(src, query, title, null);
    }

    public static void tell_user(JFrame src, String message, String title) {
        JOptionPane.showMessageDialog(src, message, title, JOptionPane.INFORMATION_MESSAGE);
    }

    public <T> void p(T s) {
        System.out.println(s);
    }

    protected void copy_to_clipboard() {
        String s = description_text.getText();
        Clipboard clipboard = Toolkit.getDefaultToolkit().getSystemClipboard();
        Transferable transferable = new StringSelection(s);
        clipboard.setContents(transferable, null);
    }

    private void close() {
        if (parent instanceof BiQuadDesigner) {
            ((BiQuadDesigner) parent).close();
        }
    }

    /**
     * This method is called from within the constructor to initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is always
     * regenerated by the Form Editor.
     */
    @SuppressWarnings("unchecked")
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {
        java.awt.GridBagConstraints gridBagConstraints;

        jLabel8 = new javax.swing.JLabel();
        buttonGroup1 = new javax.swing.ButtonGroup();
        tabbed_pane = new javax.swing.JTabbedPane();
        graph_panel_tab = new javax.swing.JPanel();
        graph_control_panel = new javax.swing.JPanel();
        mouse_pos_label = new javax.swing.JLabel();
        description_container = new javax.swing.JPanel();
        jScrollPane1 = new javax.swing.JScrollPane();
        description_text = new javax.swing.JTextArea();
        control_panel = new javax.swing.JPanel();
        panela = new javax.swing.JPanel();
        jLabel1 = new javax.swing.JLabel();
        filter_type_combobox = new javax.swing.JComboBox();
        jLabel2 = new javax.swing.JLabel();
        q_textfield = new javax.swing.JTextField();
        jLabel6 = new javax.swing.JLabel();
        gain_textfield = new javax.swing.JTextField();
        jLabel3 = new javax.swing.JLabel();
        cf_textfield = new javax.swing.JTextField();
        jLabel7 = new javax.swing.JLabel();
        rate_textfield = new javax.swing.JTextField();
        defaults_button = new javax.swing.JButton();
        panelb = new javax.swing.JPanel();
        jLabel4 = new javax.swing.JLabel();
        startf_textfield = new javax.swing.JTextField();
        jLabel5 = new javax.swing.JLabel();
        endf_textfield = new javax.swing.JTextField();
        jLabel9 = new javax.swing.JLabel();
        linear_radiobutton = new javax.swing.JRadioButton();
        db_radiobutton = new javax.swing.JRadioButton();
        copy_button = new javax.swing.JButton();
        help_button = new javax.swing.JButton();
        quit_button = new javax.swing.JButton();

        jLabel8.setText("jLabel7");

        setLayout(new java.awt.BorderLayout());

        tabbed_pane.setTabPlacement(javax.swing.JTabbedPane.BOTTOM);

        graph_panel_tab.setLayout(new java.awt.BorderLayout());

        graph_control_panel.setLayout(new java.awt.GridBagLayout());

        mouse_pos_label.setFont(new java.awt.Font("Monospaced", 0, 12)); // NOI18N
        mouse_pos_label.setText("                 ");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(0, 4, 0, 4);
        graph_control_panel.add(mouse_pos_label, gridBagConstraints);

        graph_panel_tab.add(graph_control_panel, java.awt.BorderLayout.SOUTH);

        tabbed_pane.addTab("Filter Graph", graph_panel_tab);

        description_container.setLayout(new java.awt.BorderLayout());

        description_text.setColumns(20);
        description_text.setFont(new java.awt.Font("Monospaced", 0, 12)); // NOI18N
        description_text.setRows(5);
        description_text.setMargin(new java.awt.Insets(4, 4, 4, 4));
        jScrollPane1.setViewportView(description_text);

        description_container.add(jScrollPane1, java.awt.BorderLayout.CENTER);

        tabbed_pane.addTab("Filter Description", description_container);

        add(tabbed_pane, java.awt.BorderLayout.CENTER);

        control_panel.setLayout(new java.awt.GridBagLayout());

        panela.setLayout(new java.awt.GridBagLayout());

        jLabel1.setHorizontalAlignment(javax.swing.SwingConstants.RIGHT);
        jLabel1.setText("Type:");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.insets = new java.awt.Insets(0, 4, 0, 4);
        panela.add(jLabel1, gridBagConstraints);

        filter_type_combobox.setModel(new javax.swing.DefaultComboBoxModel(new String[] { "Item 1", "Item 2", "Item 3", "Item 4" }));
        filter_type_combobox.setToolTipText("Select a filter type");
        panela.add(filter_type_combobox, new java.awt.GridBagConstraints());

        jLabel2.setHorizontalAlignment(javax.swing.SwingConstants.RIGHT);
        jLabel2.setText("Q:");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.insets = new java.awt.Insets(0, 4, 0, 4);
        panela.add(jLabel2, gridBagConstraints);

        q_textfield.setHorizontalAlignment(javax.swing.JTextField.RIGHT);
        q_textfield.setText("0.00");
        q_textfield.setToolTipText("Actual Q value is 1/100 displayed value<br/>\nNo effect on highshelf, lowshelf");
        q_textfield.setMaximumSize(new java.awt.Dimension(60, 2147483647));
        q_textfield.setMinimumSize(new java.awt.Dimension(60, 19));
        q_textfield.setPreferredSize(new java.awt.Dimension(60, 19));
        panela.add(q_textfield, new java.awt.GridBagConstraints());

        jLabel6.setHorizontalAlignment(javax.swing.SwingConstants.RIGHT);
        jLabel6.setText("Filter Gain DB:");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.insets = new java.awt.Insets(0, 4, 0, 4);
        panela.add(jLabel6, gridBagConstraints);

        gain_textfield.setHorizontalAlignment(javax.swing.JTextField.RIGHT);
        gain_textfield.setText("0.00");
        gain_textfield.setToolTipText("Only affects peak, highshelf and lowshelf filters");
        gain_textfield.setMaximumSize(new java.awt.Dimension(60, 2147483647));
        gain_textfield.setMinimumSize(new java.awt.Dimension(60, 19));
        gain_textfield.setPreferredSize(new java.awt.Dimension(60, 19));
        panela.add(gain_textfield, new java.awt.GridBagConstraints());

        jLabel3.setHorizontalAlignment(javax.swing.SwingConstants.RIGHT);
        jLabel3.setText("Center Freq.:");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.insets = new java.awt.Insets(0, 4, 0, 4);
        panela.add(jLabel3, gridBagConstraints);

        cf_textfield.setHorizontalAlignment(javax.swing.JTextField.RIGHT);
        cf_textfield.setText("0.00");
        cf_textfield.setToolTipText("Filter center frequency Hz");
        cf_textfield.setMaximumSize(new java.awt.Dimension(80, 2147483647));
        cf_textfield.setMinimumSize(new java.awt.Dimension(60, 19));
        cf_textfield.setPreferredSize(new java.awt.Dimension(80, 19));
        panela.add(cf_textfield, new java.awt.GridBagConstraints());

        jLabel7.setHorizontalAlignment(javax.swing.SwingConstants.RIGHT);
        jLabel7.setText("Rate:");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.insets = new java.awt.Insets(0, 4, 0, 4);
        panela.add(jLabel7, gridBagConstraints);

        rate_textfield.setHorizontalAlignment(javax.swing.JTextField.RIGHT);
        rate_textfield.setText("0.00");
        rate_textfield.setToolTipText("Sample Rate samples/second");
        rate_textfield.setMaximumSize(new java.awt.Dimension(80, 2147483647));
        rate_textfield.setMinimumSize(new java.awt.Dimension(60, 19));
        rate_textfield.setPreferredSize(new java.awt.Dimension(80, 19));
        panela.add(rate_textfield, new java.awt.GridBagConstraints());

        defaults_button.setText("Defaults");
        defaults_button.setToolTipText("Set all default values");
        defaults_button.setMargin(new java.awt.Insets(2, 4, 2, 4));
        defaults_button.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                defaults_buttonMouseClicked(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.insets = new java.awt.Insets(0, 4, 0, 4);
        panela.add(defaults_button, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.SOUTH;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(2, 2, 2, 2);
        control_panel.add(panela, gridBagConstraints);

        panelb.setLayout(new java.awt.GridBagLayout());

        jLabel4.setHorizontalAlignment(javax.swing.SwingConstants.RIGHT);
        jLabel4.setText("Start:");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.insets = new java.awt.Insets(0, 4, 0, 4);
        panelb.add(jLabel4, gridBagConstraints);

        startf_textfield.setHorizontalAlignment(javax.swing.JTextField.RIGHT);
        startf_textfield.setText("0.00");
        startf_textfield.setToolTipText("Graph start frequency Hz");
        startf_textfield.setMaximumSize(new java.awt.Dimension(60, 2147483647));
        startf_textfield.setMinimumSize(new java.awt.Dimension(60, 19));
        startf_textfield.setPreferredSize(new java.awt.Dimension(80, 19));
        panelb.add(startf_textfield, new java.awt.GridBagConstraints());

        jLabel5.setHorizontalAlignment(javax.swing.SwingConstants.RIGHT);
        jLabel5.setText("End:");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.insets = new java.awt.Insets(0, 4, 0, 4);
        panelb.add(jLabel5, gridBagConstraints);

        endf_textfield.setHorizontalAlignment(javax.swing.JTextField.RIGHT);
        endf_textfield.setText("0.00");
        endf_textfield.setToolTipText("Graph end frequency Hz");
        endf_textfield.setMaximumSize(new java.awt.Dimension(60, 2147483647));
        endf_textfield.setMinimumSize(new java.awt.Dimension(60, 19));
        endf_textfield.setPreferredSize(new java.awt.Dimension(80, 19));
        panelb.add(endf_textfield, new java.awt.GridBagConstraints());

        jLabel9.setHorizontalAlignment(javax.swing.SwingConstants.RIGHT);
        jLabel9.setText("Vert. Units:");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.insets = new java.awt.Insets(0, 4, 0, 4);
        panelb.add(jLabel9, gridBagConstraints);

        buttonGroup1.add(linear_radiobutton);
        linear_radiobutton.setSelected(true);
        linear_radiobutton.setText("Linear");
        linear_radiobutton.setToolTipText("Choose linear vertical axis scaling");
        panelb.add(linear_radiobutton, new java.awt.GridBagConstraints());

        buttonGroup1.add(db_radiobutton);
        db_radiobutton.setText("20 Log10");
        db_radiobutton.setToolTipText("Choose decibel vertical axis scaling");
        db_radiobutton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                db_radiobuttonActionPerformed(evt);
            }
        });
        panelb.add(db_radiobutton, new java.awt.GridBagConstraints());

        copy_button.setText("Copy");
        copy_button.setToolTipText("Copy filter description to system clipboard");
        copy_button.setMargin(new java.awt.Insets(2, 4, 2, 4));
        copy_button.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                copy_buttonMouseClicked(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.insets = new java.awt.Insets(0, 4, 0, 4);
        panelb.add(copy_button, gridBagConstraints);

        help_button.setText("Help");
        help_button.setToolTipText("Visit the BiQuadDesigner Home Page");
        help_button.setMargin(new java.awt.Insets(2, 4, 2, 4));
        help_button.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                help_buttonMouseClicked(evt);
            }
        });
        panelb.add(help_button, new java.awt.GridBagConstraints());

        quit_button.setText("Quit");
        quit_button.setToolTipText("Exit BiQuadDesigner");
        quit_button.setMargin(new java.awt.Insets(2, 4, 2, 4));
        quit_button.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                quit_buttonMouseClicked(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.insets = new java.awt.Insets(0, 4, 0, 4);
        panelb.add(quit_button, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.SOUTH;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(2, 2, 2, 2);
        control_panel.add(panelb, gridBagConstraints);

        add(control_panel, java.awt.BorderLayout.SOUTH);
    }// </editor-fold>//GEN-END:initComponents

    private void quit_buttonMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_quit_buttonMouseClicked
        close();
    }//GEN-LAST:event_quit_buttonMouseClicked

    private void defaults_buttonMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_defaults_buttonMouseClicked
        set_defaults();
    }//GEN-LAST:event_defaults_buttonMouseClicked

    private void help_buttonMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_help_buttonMouseClicked
        launch_help();
    }//GEN-LAST:event_help_buttonMouseClicked

    private void copy_buttonMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_copy_buttonMouseClicked
        // TODO add your handling code here:
        copy_to_clipboard();
    }//GEN-LAST:event_copy_buttonMouseClicked

    private void db_radiobuttonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_db_radiobuttonActionPerformed
        // TODO add your handling code here:
    }//GEN-LAST:event_db_radiobuttonActionPerformed

    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.ButtonGroup buttonGroup1;
    private javax.swing.JTextField cf_textfield;
    private javax.swing.JPanel control_panel;
    protected javax.swing.JButton copy_button;
    private javax.swing.JRadioButton db_radiobutton;
    private javax.swing.JButton defaults_button;
    private javax.swing.JPanel description_container;
    private javax.swing.JTextArea description_text;
    private javax.swing.JTextField endf_textfield;
    private javax.swing.JComboBox filter_type_combobox;
    private javax.swing.JTextField gain_textfield;
    private javax.swing.JPanel graph_control_panel;
    private javax.swing.JPanel graph_panel_tab;
    private javax.swing.JButton help_button;
    private javax.swing.JLabel jLabel1;
    private javax.swing.JLabel jLabel2;
    private javax.swing.JLabel jLabel3;
    private javax.swing.JLabel jLabel4;
    private javax.swing.JLabel jLabel5;
    private javax.swing.JLabel jLabel6;
    private javax.swing.JLabel jLabel7;
    private javax.swing.JLabel jLabel8;
    private javax.swing.JLabel jLabel9;
    private javax.swing.JScrollPane jScrollPane1;
    private javax.swing.JRadioButton linear_radiobutton;
    protected javax.swing.JLabel mouse_pos_label;
    private javax.swing.JPanel panela;
    private javax.swing.JPanel panelb;
    private javax.swing.JTextField q_textfield;
    private javax.swing.JButton quit_button;
    private javax.swing.JTextField rate_textfield;
    private javax.swing.JTextField startf_textfield;
    private javax.swing.JTabbedPane tabbed_pane;
    // End of variables declaration//GEN-END:variables
}
