/**
 * *************************************************************************
 * Copyright (C) 2011 by Paul Lutus * lutusp@arachnoid.com * * This program is
 * free software; you can redistribute it and/or modify * it under the terms of
 * the GNU General Public License as published by * the Free Software
 * Foundation; either version 2 of the License, or * (at your option) any later
 * version. * * This program is distributed in the hope that it will be useful,
 * * but WITHOUT ANY WARRANTY; without even the implied warranty of *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the * GNU General
 * Public License for more details. * * You should have received a copy of the
 * GNU General Public License * along with this program; if not, write to the *
 * Free Software Foundation, Inc., * 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA. *
 * *************************************************************************
 */
/*
 * GraphPanel.java
 *
 * Created on Mar 21, 2011, 2:05:32 PM
 */
package biquaddesigner;

import java.awt.*;
import java.awt.geom.GeneralPath;
import java.io.*;
import javax.imageio.*;
import java.awt.image.*;
import java.util.ArrayList;

/**
 *
 * @author lutusp
 */
final public class GraphPanel extends javax.swing.JPanel {

    BiQuadDesignerPanel parent;
    int gsize = 2000;
    double xstart, xend, ystart, yend;
    boolean db_mode;
    double gain_offset;
    double width_factor = 1;
    int width = -1, height = -1;
    int[] offsets;
    int font_factor;
    boolean show_scale_numbers = true;
    // size of produced graphic image
    int graphicw = 200;
    int graphich = 150;
    double maxval, minval;
    ArrayList<Double> ydata;

    /**
     * Creates new form GraphPanel
     */
    public GraphPanel(BiQuadDesignerPanel p) {
        parent = p;
        initComponents();
    }

    private double ntrp(double x, double xa, double xb, double ya, double yb) {
        return (x - xa) * (yb - ya) / (xb - xa) + ya;
    }

    private double get_filter_result(double x) {
        return (db_mode) ? parent.filter.log_result(x) : parent.filter.result(x);
    }

    // capture images for documentation
    private void handle_mouse_click(java.awt.event.MouseEvent evt) {
        if (parent.debug && evt.getButton() == 3) {
            try {
                BufferedImage bi = new BufferedImage(graphicw, graphich, BufferedImage.TYPE_INT_ARGB);
                Graphics2D g = bi.createGraphics();
                draw_graph(graphicw, graphich, g);
                int image_num = 0;
                File f = null;
                do {
                    f = new File("BiQuadDesigner_image_" + image_num + ".png");
                    image_num++;
                } while (f.exists());
                ImageIO.write(bi, "PNG", f);
                image_num++;
            } catch (Exception e) {
                System.out.println(e);
                e.printStackTrace();
            }
        }
    }

    @Override
    public void paintComponent(Graphics g) {
        draw_graph(getWidth(), getHeight(), g);
    }

    private void draw_graph(int width, int height, Graphics gg) {
        this.width = width;
        this.height = height;
        width_factor = width / 800.0;
        show_scale_numbers = width_factor > 0.85;
        if (show_scale_numbers) {
            offsets = new int[]{
                (int) (70 * width_factor),
                (int) (10 * width_factor),
                (int) (-36 * width_factor),
                (int) (-25 * width_factor)
            };
        } else {
            offsets = new int[]{
                (int) (10 * width_factor),
                (int) (10 * width_factor),
                (int) (-10 * width_factor),
                (int) (-10 * width_factor)
            };
        }
        font_factor = (int) (5 * width_factor);
        Graphics2D g = (Graphics2D) gg;
        g.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
                RenderingHints.VALUE_ANTIALIAS_ON);
        xstart = parent.graph_start_frequency.get_dvalue();
        xend = parent.graph_end_frequency.get_dvalue();
        ystart = 0;
        yend = 1;
        db_mode = parent.db_mode.get_value();
        if (db_mode) {
            ystart = -100;
            yend = 0;
        }
        if (parent.filter != null) {
            double x, y;
            ydata = new ArrayList();
            maxval = -1e30;
            minval = 1e30;
            for (int i = 0; i < gsize; i++) {
                x = ntrp(i, 0, gsize, xstart, xend);
                acquire_value(x, ydata, true);
            }
            // also acquire this important data point
            acquire_value(parent.v_cf, ydata, false);
            ystart = minval;//(ystart > minval) ? minval : ystart;
            yend = maxval;//(yend < maxval) ? maxval : yend;
        }
        double gain_offset = (db_mode) ? 0 : 1;
        ystart = (ystart - gain_offset) / parent.y_axis_gain + gain_offset;
        yend = (yend - gain_offset) / parent.y_axis_gain + gain_offset;
        g.setColor(Color.white);
        g.fillRect(0, 0, width, height);
        draw_grid(offsets[0], width + offsets[2], height + offsets[3], offsets[1], g);
        if (parent.filter != null) {
            draw_trace(offsets[0], width + offsets[2], height + offsets[3], offsets[1], g);
        }
    }

    private void acquire_value(double x, ArrayList<Double> ydata, boolean collect) {
        double y = get_filter_result(x);
        if (collect) {
            ydata.add(y);
        }
        maxval = Math.max(y, maxval);
        minval = Math.min(minval, y);
    }

    private void draw_trace(int xa, int xb, int ya, int yb, Graphics2D g) {
        int px, py, ox = 0, oy = 0;
        g.setColor(new Color(0f, .25f, 1f));
        double x, y;
        GeneralPath gp = new GeneralPath();
        boolean first = true;
        for (int i = 0; i < gsize; i++) {
            // use the data acquired during scaling
            y = ydata.get(i);
            px = (int) ntrp(i, 0, gsize, xa, xb);
            py = (int) ntrp(y, ystart, yend, ya, yb);
            if (first) {
                gp.moveTo(px, py);
                first = false;
            } else {
                gp.lineTo(px, py);
            }
        }
        g.draw(gp);
    }

    private void draw_grid(int xa, int xb, int ya, int yb, Graphics2D g) {
        int hs = 10;
        int vs = 5;
        double v = 0;
        String st;
        // axis lines
        g.setColor(new Color(.7f, .7f, .7f));
        for (int y = 0; y <= vs; y++) {
            int pv = (int) ntrp(y, 0, vs, ya, yb);
            g.drawLine(xa, pv, xb, pv);
            //pv = (int) ntrp(y, 0, vs, xa, xb);
            //g.drawLine(pv, ya, pv, yb);
        }
        for (int x = 0; x <= hs; x++) {
            int pv = (int) ntrp(x, 0, hs, xa, xb);
            g.drawLine(pv, ya, pv, yb);
        }
        if (show_scale_numbers) {
            // axis numbers
            g.setColor(Color.black);
            Font f = new Font(Font.MONOSPACED, Font.PLAIN, (int) (width_factor * 12));
            g.setFont(f);
            int h_off = font_factor * 7;
            int v_off = font_factor * 4;
            for (int y = 0; y <= vs; y++) {
                int pv = (int) ntrp(y, 0, vs, ya, yb);
                v = ntrp(y, 0, vs, ystart, yend);
                st = String.format("%+.1f", v);
                int len = st.length();
                g.drawString(st, 64 - len * 8, pv + font_factor);
            }
            for (int x = 0; x <= hs; x++) {
                int pv = (int) ntrp(x, 0, hs, xa, xb);
                v = ntrp(x, 0, hs, xstart, xend);
                st = String.format("%+.1f", v);
                g.drawString(st, pv - h_off, ya + v_off);
            }
        }
        //g.setStroke(old_stroke);
    }

    private void mouse_move(java.awt.event.MouseEvent evt) {
        double x = ntrp(evt.getX(), offsets[0], width + offsets[2], xstart, xend);
        double y = ntrp(evt.getY(), offsets[1], height + offsets[3], yend, ystart);
        double logy, liny;
        if (db_mode) {
            logy = y;
            liny = Math.pow(10, y / 20.0);
        } else {
            logy = Math.log10(y) * 20;
            liny = y;
        }
        String s = String.format("Mouse cursor: x = %8.4g Hz, y = %8.4g Units / %8.4g db", x, liny, logy);
        parent.mouse_pos_label.setText(s);
    }

    private void mouse_exit() {
        parent.mouse_pos_label.setText("     ");
    }

    /**
     * This method is called from within the constructor to initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is always
     * regenerated by the Form Editor.
     */
    @SuppressWarnings("unchecked")
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {

        setPreferredSize(new java.awt.Dimension(600, 400));
        addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                formMouseClicked(evt);
            }
            public void mouseExited(java.awt.event.MouseEvent evt) {
                formMouseExited(evt);
            }
        });
        addMouseMotionListener(new java.awt.event.MouseMotionAdapter() {
            public void mouseMoved(java.awt.event.MouseEvent evt) {
                formMouseMoved(evt);
            }
        });
        setLayout(new java.awt.GridBagLayout());
    }// </editor-fold>//GEN-END:initComponents

    private void formMouseMoved(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_formMouseMoved
        mouse_move(evt);
    }//GEN-LAST:event_formMouseMoved

    private void formMouseExited(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_formMouseExited
        // TODO add your handling code here:
        mouse_exit();
    }//GEN-LAST:event_formMouseExited

    private void formMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_formMouseClicked
        // TODO add your handling code here:
        handle_mouse_click(evt);
    }//GEN-LAST:event_formMouseClicked
    // Variables declaration - do not modify//GEN-BEGIN:variables
    // End of variables declaration//GEN-END:variables
}
