#!/usr/bin/env python
# -*- coding: utf-8 -*-

import cadquery as cq
import os

# all dimensions in mm

pitch = 1.5 # interval between thread peaks

# gap to allow threading to bolt
# this value is shared between bolt and nut
clearance = 0.25

thread_a = 3.845 + clearance # "valley" of thread
thread_b = 5.160 + clearance # "peak" of thread

bolt_head_dia = 18 # mm
bolt_head_height = 6.5
shaft_radius = thread_b + 0.001
#shaft_height = 31

bolt_shaft = (
  cq.Workplane("XY")
  .circle(shaft_radius)
  .extrude(bolt_head_height)
)

bolt_hex_head = (
  cq.Workplane("XY")
  .polygon(6,bolt_head_dia, circumscribed=False)
  .extrude(bolt_head_height)  
)


thread_vals = [
  (thread_b,0),
  (thread_a,pitch/2),
  (thread_b,pitch),
  (thread_b,0)
]

r = thread_a # helix radius (base dimension)
h = bolt_head_height+pitch

# create helix
wire = cq.Wire.makeHelix(pitch=pitch, height=h, radius=r)
helix = cq.Workplane(obj=wire)

thread = (
    cq.Workplane('XZ')
    .polyline(thread_vals)
    .close()
    .sweep(helix, isFrenet=True)
    .translate((0,0,-pitch))
)

bolt = bolt_hex_head.cut(bolt_shaft) + thread
#show(assembly)

v1 = bolt_head_dia/2

r1 = bolt_head_dia / 12 # radius of bolt head curve

top_slope = 3

enclosure = ( cq.Workplane("XZ")
    .lineTo(-v1+r1,-.01)
    .radiusArc((-v1,r1),r1)
    .lineTo(-v1,bolt_head_height-r1)
    .radiusArc((-v1+r1,bolt_head_height),r1)
    .lineTo(0,bolt_head_height)
    .close()
    .revolve(360,(0,0),(0,1))
)

#show(enclosure)

assembled = enclosure.intersect(bolt, clean=False)

if 'JPY_PARENT_PID' in os.environ:
    assembled
else:
    cq.exporters.export(assembled,'cadquery_nut.stl')
    ibh = enclosure.intersect(bolt_hex_head, clean=False)
    ibs = enclosure.intersect(bolt_shaft, clean=False)
    ibt = enclosure.intersect(thread, clean=False)
    assy = cq.Assembly()
    assy.add(ibh, color=cq.Color(1,0,0), name = "Hex head")
    assy.add(ibs, color=cq.Color(0,1,0), name = "Shaft")
    assy.add(ibt, color=cq.Color(0,.5,.5), name = "Thread")
    assy.save("cadquery_nut.step")

