// ***************************************************************************
// *   Copyright (C) 2012 by Paul Lutus                                      *
// *   lutusp@arachnoid.com                                                  *
// *                                                                         *
// *   This program is free software; you can redistribute it and/or modify  *
// *   it under the terms of the GNU General Public License as published by  *
// *   the Free Software Foundation; either version 2 of the License, or     *
// *   (at your option) any later version.                                   *
// *                                                                         *
// *   This program is distributed in the hope that it will be useful,       *
// *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
// *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
// *   GNU General Public License for more details.                          *
// *                                                                         *
// *   You should have received a copy of the GNU General Public License     *
// *   along with this program; if not, write to the                         *
// *   Free Software Foundation, Inc.,                                       *
// *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
// ***************************************************************************

package jdbclient;

import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.swing.JTable;

/**
 *
 * @author lutusp
 */
final public class ExportMethods {

    static String utf8Block = "<?xml version=\"1.0\" encoding=\"UTF-8\"?>";
    static String[] headBlock = {
        "<meta http-equiv=\"content-type\" content=\"application/xhtml+xml; charset=UTF-8\" />",
        "    <style type='text/css'>",
        "    /* for DBClient-generated tables */",
        "     body * {",
        "       font-family: Verdana, Tahoma, Helvetica, Arial;",
        "       font-size: 14px;",
        "     }",
        "     table.dbclient {",
        "       border-collapse:collapse;",
        "     }",
        "     .dbclient tr td {",
        "        border:1px solid #808080;",
        "        padding-right:4px;",
        "        padding-left:4px;",
        "        vertical-align:top;",
        "     }",
        "     .dbclient tr td.ra { text-align:right }",
        "     .dbclient tr th {",
        "        padding-right:4px;",
        "        padding-left:4px;",
        "        border:1px solid #808080;",
        "        background:#96b183;",
        "        font-weight: bold;",
        "        text-align: center;",
        "     }",
        "     .dbclient tr.row0 { background:#bfdeb9; }",
        "     .dbclient tr.row1 { background:#f6ffda; }",
        "     </style>"
    };
    static String noWrapStyle = "<style type='text/css'>table {white-space:nowrap}</style>";

    public static String tableToDelimText(JTable table, String lineSep, String delim) {
        StringBuilder sb = new StringBuilder();
        int cols = table.getColumnCount();
        int rows = table.getRowCount();
        StringBuilder row = new StringBuilder();
        for (int x = 0; x < cols; x++) {
            if (x > 0) {
                row.append(delim);
            }
            row.append(table.getColumnName(x));
        }
        row.append(lineSep);
        sb.append(row);
        for (int y = 0; y < rows; y++) {
            row = new StringBuilder();
            for (int x = 0; x < cols; x++) {
                if (x > 0) {
                    row.append(delim);
                }
                row.append(table.getValueAt(y, x));
            }
            row.append(lineSep);
            sb.append(row);
        }
        return sb.toString();
    }

    private static String wrapTag(String tag, String data, String mod) {
        if (mod.length() > 0) {
            mod = " " + mod;
        }
        return String.format("<%s%s>%s</%s>\n", tag, mod, data, tag);
    }

    private static String makeHeadBlock() {
        StringBuilder sb = new StringBuilder();
        for (String s : headBlock) {
            sb.append(s);
            sb.append("\n");
        }
        return sb.toString();
    }

    private static int countMatches(String data, String search) {
        Pattern pattern = Pattern.compile(search);
        Matcher matcher = pattern.matcher(data);
        int n = 0;
        while (matcher.find()) {
            n += 1;
        }
        return n;
    }

    private static String makeTab(int n, String tabStr) {
        StringBuilder sb = new StringBuilder();
        while (n-- > 0) {
            sb.append(tabStr);
        }
        return sb.toString();
    }

    private static String beautifyXHTML(String data, String lineSep, boolean lf) {
        if (lf) {
            // each tag on a separate line
            data = data.replaceAll("<", "\n<");
            data = data.replaceAll(">", ">\n");
            data = data.replaceAll("\n+", "\n");
        }
        int tab = 0;
        int outc, inc, net;
        String[] array = data.split("\n+");
        StringBuilder xml = new StringBuilder();
        for (String record : array) {
            record = record.trim();
            // no blank lines
            if (record.length() > 0) {
                outc = countMatches(record, "</|/>");
                inc = countMatches(record, "<\\w");
                net = inc - outc;
                tab += Math.min(net,0);
                xml.append(makeTab(tab, "  "));
                xml.append(record);
                xml.append(lineSep);
                tab += Math.max(net,0);
            }
        }
        if (tab != 0) {
            System.out.println("In beautifyXHTML, tag mismatch: " + tab);
        }
        return xml.toString();
    }

    public static String tableToHTML(String tableTitle,JTable table, String lineSep, boolean wrapLines) {
        StringBuilder sb = new StringBuilder();
        int cols = table.getColumnCount();
        int rows = table.getRowCount();
        StringBuilder row = new StringBuilder();
        for (int x = 0; x < cols; x++) {
            row.append(wrapTag("th", table.getColumnName(x), ""));
        }
        sb.append(wrapTag("tr", lineSep + row.toString(), ""));
        for (int y = 0; y < rows; y++) {
            row = new StringBuilder();
            for (int x = 0; x < cols; x++) {
                Object obj = table.getValueAt(y, x);
                String s = obj.toString();
                s = s.replaceAll("<","&lt;");
                s = s.replaceAll(">","&gt;");
                // format bulleted lists within fields
                // based on presence of embedded tabs and linefeeds
                if(s.matches("(?s).*(\n|\t).*")) {
                    s = s.replaceAll("\n","<br/>\n");
                    s = s.replaceAll("(?s)\t(.*?)\n","\n<li>$1</li>");
                    s = s.replaceAll("(?s)(<li>.*</li>)","\n<ul>\n$1\n</ul>\n");
                }
                // create page links for URLs
                s = s.replaceAll("((ftp|https?)://\\S+)","<a href=\"$1\">$1</a>");
                // right-align non-strings
                String ra = (obj instanceof java.lang.String) ? "" : "class=\"ra\"";
                row.append(wrapTag("td", s, ra));
            }
            String mod = String.format("class=\"row%d\"", (y % 2));
            sb.append(wrapTag("tr", lineSep + row.toString(), mod));
        }
        
        String banner = wrapTag("h2",tableTitle,"");
        String title = lineSep + wrapTag("title",tableTitle,"");
        String htable = lineSep + banner + lineSep + wrapTag("table", lineSep + sb.toString(), "class=\"dbclient\"");
        String tail = lineSep + String.format("<!-- Generated by JDBClient (http://arachnoid.com/JDBClient), copyright © 2012, P.lutus -->\n");
        String cdiv = wrapTag("div",lineSep + htable + tail,"align=\"center\"");
        String headContent = lineSep + makeHeadBlock();
        if (!wrapLines) {
            headContent += noWrapStyle;
        }
        String page = lineSep + wrapTag("head", title + headContent, "");
        
        page += lineSep + wrapTag("body", cdiv, "");
        page = wrapTag("html", page, "");
        return beautifyXHTML(page, lineSep, false);
    }
    
    private static void p(String s) {
        System.out.println(s);
    }
}
