// ***************************************************************************
// *   Copyright (C) 2012 by Paul Lutus                                      *
// *   lutusp@arachnoid.com                                                  *
// *                                                                         *
// *   This program is free software; you can redistribute it and/or modify  *
// *   it under the terms of the GNU General Public License as published by  *
// *   the Free Software Foundation; either version 2 of the License, or     *
// *   (at your option) any later version.                                   *
// *                                                                         *
// *   This program is distributed in the hope that it will be useful,       *
// *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
// *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
// *   GNU General Public License for more details.                          *
// *                                                                         *
// *   You should have received a copy of the GNU General Public License     *
// *   along with this program; if not, write to the                         *
// *   Free Software Foundation, Inc.,                                       *
// *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
// ***************************************************************************

/*
 * Main.java
 *
 * Created on Oct 23, 2012, 11:00:48 AM
 */
package jdbclient;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Desktop;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Image;
import java.awt.Insets;
import java.awt.Point;
import java.awt.Toolkit;
import java.awt.datatransfer.Clipboard;
import java.awt.datatransfer.StringSelection;
import java.awt.datatransfer.Transferable;
import java.awt.event.MouseEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.text.SimpleDateFormat;
import java.util.AbstractCollection;
import java.util.ArrayList;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.Iterator;
import javax.swing.ImageIcon;
import javax.swing.InputMap;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JDialog;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JPasswordField;
import javax.swing.JTable;
import javax.swing.JTextField;
import javax.swing.KeyStroke;
import javax.swing.SwingUtilities;
import javax.swing.border.LineBorder;

/**
 *
 * @author lutusp
 */
final public class JDBClient extends javax.swing.JFrame implements Executable {

    boolean debug = false;
    final String VERSION = "2.7";
    static final long serialVersionUID = 1128;
    final String appName;
    final String programName;
    String appDir;
    String appPath;
    String fileSep;
    String lineSep;
    String userDir;
    String userPath;
    String initPath;
    String webPageDir = "web_pages";
    String webPagePath;
    final String[] queryTableToolTips;
    ConfigManager config;
    String serverName = null;
    String userName = null;
    String password = null;
    String serverPort = null;
    boolean inhibit = true;
    ArrayList<String> dbNames;
    ArrayList<String> tableNames;
    QueryProcessor tableDescriptionQuery = null;
    QueryProcessor sv_currentQuery = null;
    QueryProcessor logQuery;
    //ArrayList<QueryControlRow> parentQueryRows;
    int charWidth;
    int charHeight;
    int[] logColumnWidths;
    int tablePaddingConstant = 4;
    Color tableBorderColor;
    Color tableHeaderColor;
    Color selectedColor;
    Color hoverColor;
    Color oddeven[];
    JFrame sv_mainFrame;
    ParseComLine parseComs;
    Font baseFont;
    int maxFontSize = 32;
    String baseFontName = Font.MONOSPACED;
    EditFunctions editFunctions;
    Component currentTab = null;
    int oldDatabaseIndex = -1;
    int oldTableIndex = -1;
    Color changedColor;
    Color changedBColor;
    TerminalFunctions terminal;
    String unicodeSpec = "useUnicode=true&characterEncoding=utf8";
    double sv_dividerLocation = 0.5;
    MyHelpPane sv_helpPane;
    boolean allFieldsSelected;
    Image appIcon;
    final String[][] escapeChars;
    HistoryEngine sv_termEngine, sv_prefixEngine, sv_postfixEngine;
    MyTableCellRenderer mtcr = null;

    /**
     * Creates new form Main
     */
    //@SuppressWarnings("LeakingThisInConstructor")
    public JDBClient(String[] args) {
        parseComs = new ParseComLine(this, args);
        inhibit = true;
        queryTableToolTips = new String[]{
            "The table field to which this row applies",
            "Logical \"And\"",
            "Logical \"Or\"",
            "<html>Enter first query argument for \"%s\" field<br/>May be blank, \"Enter\" executes, arrow keys move vertically",
            "Logical \"And\"",
            "Logical \"Or\"",
            "<html>Enter second query argument for \"%s\" field<br/>May be blank, \"Enter\" executes, arrow keys move vertically",
            "Include \"%s\" field in query result table"
        };
        escapeChars = new String[][]{{"\\\\b", "\b"}, {"\\\\t", "\t"}, {"\\\\n", "\n"}, {"\\\\f", "\f"}, {"\\\\r", "\r"}};
        appName = getClass().getSimpleName();
        URL url = getClass().getResource(appName + ".class");
        appPath = url.getPath().replaceFirst("(.*?)!.*", "$1");
        appPath = appPath.replaceFirst("file:", "");
        appPath = new File(appPath).getPath();
        appDir = new File(appPath).getParent();
        programName = appName + " Version " + VERSION;
        setTitle(programName);
        lineSep = System.getProperty("line.separator");
        fileSep = System.getProperty("file.separator");
        userDir = System.getProperty("user.home");
        userPath = userDir + fileSep + "." + appName;
        initPath = userPath + fileSep + appName + ".ini";
        new File(userPath).mkdirs();
        webPagePath = userDir + fileSep + "." + appName + fileSep + webPageDir;
        new File(webPagePath).mkdirs();
        initComponents();
        mtcr = new MyTableCellRenderer(this, true, false);
        //logTable.setDefaultRenderer(Object.class, new MyTableCellRenderer(this, false, true));
        setTabToolTips();
        this.setSize(800, 400);
        //tableBorderColor = new Color(192, 192, 192);
        tableBorderColor = new Color(0,0,0);
        tableHeaderColor = new Color(0x96,0xb1,0x83);
        hoverColor = new Color(0xff,0xef,0x98);
        selectedColor = new Color(192, 240, 255);
        changedColor = new Color(160, 0, 0);
        changedBColor = new Color(255,240,240);
        oddeven = new Color[]{new Color(0xbf,0xde,0xb9),new Color(0xf6,0xff,0xda)};
        appIcon = new ImageIcon(getClass().getResource("icons/" + appName + ".png")).getImage();
        setIconImage(appIcon);
        preConfigInitialize();
        sv_helpPane = new MyHelpPane(this);
        helpPanel.add(sv_helpPane);
        config = new ConfigManager(this);
        config.read();
        setDefaultTableContent(queryResultTable, "Select a database");
        setDefaultTableContent(tableDescriptionTable, "Select a table");
        instantiateJDBC();
        inhibit = false;
        parseComs.process1();
        resetFontSize(false);
        setDefaultLogTableContent();
        logTable.setDefaultRenderer(Object.class, new MyTableCellRenderer(this,false,true));
        tableDescriptionTable.setDefaultRenderer(Object.class, new MyTableCellRenderer(this,false,true));
        editFunctions = new EditFunctions(this);
        

        // don't render content until the app has become visible
        SwingUtilities.invokeLater(
                new Runnable() {
                    @Override
                    public void run() {
                        readDatabaseList(true);
                        parseComs.process2();
                        performTabActions(null);
                    }
                });

    }

    @Override
    public void execute(String text) {
        formatRunQuery();
    }

    protected String makeDBUrl(String sdb) {
        String result;
        if (sdb == null) {
            result = String.format("jdbc:mysql://%s:%s?%s", serverName, serverPort, unicodeSpec);
        } else {
            result = String.format("jdbc:mysql://%s:%s/%s?%s", serverName, serverPort, sdb, unicodeSpec);
        }
        if (debug) {
            p("connection url: " + result);
        }
        return result;
    }

    private void resetFontSize(boolean commit) {
        if (!inhibit) {
            baseFont = new Font(baseFontName, Font.PLAIN, (int) sv_fontSizeComboBox.getSelectedItem());
            setFont(baseFont);
            FontMetrics fontMetrics = getFontMetrics(getFont());
            charWidth = fontMetrics.charWidth('X');
            charHeight = fontMetrics.getHeight();
            if (commit && tableDescriptionQuery != null && logQuery != null) {
                sv_currentQuery.populateResultTable(false);
                tableDescriptionQuery.populateResultTable(false);
                logQuery.populateResultTable(true);
                sv_currentQuery.populateQueryPanel(tableDescriptionQuery);
            }
            terminal.setFont();
            this.sv_helpPane.setFont();
        }
    }

    private void setTabToolTips() {
        String[] tips = new String[]{
            "Setup server, user, password and read system log",
            "View query result table",
            "Edit and enter records",
            "Perform queries",
            "Examine table descriptions",
            "Execute SQL commands in a terminal",
            "Read help file"
        };
        int i = 0;
        for (String s : tips) {
            this.sv_mainTabbedPane.setToolTipTextAt(i, s);
            i += 1;
        }
    }

    private void instantiateJDBC() {
        try {
            Class.forName("com.mysql.jdbc.Driver").newInstance();
        } catch (ClassNotFoundException | InstantiationException | IllegalAccessException e) {
            if (debug) {
                e.printStackTrace(System.out);
            }
        }
    }

    private int testState() {
        int state = 0;
        serverName = getTextFieldContent(this.sv_serverTextField);
        userName = getTextFieldContent(this.sv_userTextField);
        password = getTextFieldContent(this.passwordTextField);
        serverPort = getTextFieldContent(this.sv_portTextField);
        if (stringIsValid(serverName)) {
            state += 1;
        }
        if (stringIsValid(userName)) {
            state += 1;
        }
        if (stringIsValid(password)) {
            state += 1;
        }
        return state;
    }

    protected boolean readDatabaseList(boolean proceed) {
        if (!inhibit) {
            if (testState() < 3) {
                this.sv_mainTabbedPane.setSelectedIndex(0);
            } else {
                try {
                    reset();
                    QueryProcessor qr = new QueryProcessor(this, null, false);
                    getQueryResult(qr, "", "", "show databases", null, false);
                    dbNames = new ArrayList<>();
                    populateComboBox(qr, sv_databaseComboBox, dbNames);
                    this.setDefaultTableContent(queryResultTable, "Select a database");
                    if (proceed) {
                        readTableList();
                    }
                    return true;
                } catch (Exception e) {
                    e.printStackTrace(System.out);
                }
            }
        }
        return false;
    }

    protected boolean askAbandonEdit() {
        return (editFunctions == null || editFunctions.askAbandonEdit());
    }

    protected void readTableList() {
        if (!inhibit) {
            if (askAbandonEdit()) {
                reset();
                oldDatabaseIndex = sv_databaseComboBox.getSelectedIndex();
                String db = (String) sv_databaseComboBox.getSelectedItem();
                QueryProcessor qr = new QueryProcessor(this, null, false);
                getQueryResult(qr, db, "", "show tables", null, false);
                tableNames = new ArrayList<>();
                populateComboBox(qr, sv_tableComboBox, tableNames);
                this.setDefaultTableContent(queryResultTable, "Select a table");
            } else {
                inhibit = true;
                sv_databaseComboBox.setSelectedIndex(oldDatabaseIndex);
                inhibit = false;
            }
        }
    }

    protected void tableSelected() {
        if (!inhibit) {
            if (askAbandonEdit()) {
                reset();
                oldTableIndex = sv_tableComboBox.getSelectedIndex();
                getTableDescription(true);
                setDefaultTableContent(queryResultTable, "Run query");

            } else {
                inhibit = true;
                sv_tableComboBox.setSelectedIndex(oldTableIndex);
                inhibit = false;
            }
        }
    }

    protected void reset() {
        sv_currentQuery.reset();
        sv_currentQuery.populateQueryPanel(null);
        if (editFunctions != null) {
            editFunctions.reset();
        }
    }

    protected void getTableDescription(boolean populateQuery) {
        String db = (String) sv_databaseComboBox.getSelectedItem();
        String table = (String) sv_tableComboBox.getSelectedItem();
        String query = String.format("describe `%s`.`%s`", db, table);
        tableDescriptionQuery = new QueryProcessor(this, this.sv_tableDescriptionEllipsizeCheckBox, false);
        getQueryResult(tableDescriptionQuery, db, table, query, tableDescriptionTable, false);
        tableDescriptionQuery.populateResultTable(false);
        tableDescriptionQuery.tableDescSetup();
        addKeyButton.setEnabled(tableDescriptionQuery.primaryKeyFields.isEmpty());
        tableDescriptionLabel.setText(String.format("Table: `%s`.`%s`", db, table));
        if (populateQuery || this.sv_currentQuery.sqlColNames.isEmpty()) {
            sv_currentQuery.populateQueryPanel(tableDescriptionQuery);
        }
    }
    
    // return column data type as a string
    
    protected String dataType(String key) {
        try {
            return tableDescriptionQuery.getRecordByFieldName(key).get(1).toString();
        } catch (Exception e) {
            return "";
        }
    }

    protected boolean validString(String s) {
        return (s != null && s.length() > 0);
    }

    protected void backupTable() {
        String db = (String) sv_databaseComboBox.getSelectedItem();
        String table = (String) sv_tableComboBox.getSelectedItem();
        if (validString(db) && validString(table)) {
            if (askUser(String.format("Okay to make a backup copy of \"%s.%s\"?", db, table))) {
                {
                    String newTable = String.format("%s_backup", table);
                    QueryProcessor qr = new QueryProcessor(this, null, false);
                    qr.setup(serverName, userName, password, serverPort, db, null, null);
                    String query = String.format("CREATE TABLE `%s`.`%s` LIKE `%s`.`%s`", db, newTable, db, table);
                    qr.execMySQLUpdate(query);
                    query = String.format("INSERT `%s`.`%s` SELECT * FROM `%s`.`%s`", db, newTable, db, table);
                    qr.execMySQLUpdate(query);
                    readTableList();
                }
            }
        }
    }

    protected void queryDisp(String query) {
        queryShowLabel.setText(String.format("Query: %s", query));
    }

    private QueryProcessor readTable(String arg, String spec) {
        spec = (spec == null) ? "*" : spec;
        if (!inhibit) {
            getTableDescription(false);
            String db = (String) sv_databaseComboBox.getSelectedItem();
            String table = (String) this.sv_tableComboBox.getSelectedItem();
            String query = String.format("SELECT %s FROM `%s`.`%s` %s", spec, db, table, arg);
            queryDisp(query);
            getQueryResult(sv_currentQuery, db, table, query, queryResultTable, true);
            sv_currentQuery.populateResultTable(false);
            allFieldsSelected = sv_currentQuery.sqlColNames.size() == tableDescriptionQuery.sqlData.size();
            if (editFunctions != null) {
                editFunctions.activateEdit(-1);
            }
        }
        return sv_currentQuery;
    }

    

    private void populateComboBox(QueryProcessor qr, JComboBox<String> box, ArrayList<String> array) {
        boolean old_inhibit = inhibit;
        inhibit = true;
        int index = box.getSelectedIndex();
        box.removeAllItems();
        for (ArrayList<Object> obj : qr.getData()) {
            String s = obj.get(0).toString();
            box.addItem(s);
            array.add(s);
        }
        int count = box.getItemCount();
        if (count > 0) {
            index = Math.min(index, count - 1);
            index = Math.max(0, index);
            box.setSelectedIndex(index);
        }
        inhibit = old_inhibit;
    }

    private void clearHistories() {
        sv_prefixEngine.clearHistory();
        sv_postfixEngine.clearHistory();
    }

    private void preConfigInitialize() {
        sv_mainFrame = this;
        sv_currentQuery = new QueryProcessor(this, this.sv_tableEllipsizeCheckBox, true);
        terminal = new TerminalFunctions(this);
        sv_prefixEngine = new HistoryEngine(this, sv_currentQuery.prefix, false);
        sv_postfixEngine = new HistoryEngine(this, sv_currentQuery.postfix, false);
        ArrayList<JComboBox<String>> boxen = new ArrayList<>();
        boxen.add(sv_databaseComboBox);
        boxen.add(sv_tableComboBox);
        for (JComboBox<String> box : boxen) {
            setDefaultComboBoxContent(box);
        }
        sv_fontSizeComboBox.removeAllItems();
        for (int i = 8; i <= maxFontSize; i++) {
            sv_fontSizeComboBox.addItem(i);
        }
        sv_fontSizeComboBox.setSelectedIndex(6);
        // prevent query scroll pane from reacting to the keyboard
        InputMap im = queryScrollPane.getInputMap(JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT);
        im.put(KeyStroke.getKeyStroke("UP"), "none");
        im.put(KeyStroke.getKeyStroke("DOWN"), "none");
        im.put(KeyStroke.getKeyStroke("LEFT"), "none");
        im.put(KeyStroke.getKeyStroke("RIGHT"), "none");

    }

    private void setDefaultComboBoxContent(JComboBox<String> box) {
        int index = box.getSelectedIndex();
        box.removeAllItems();
        for (int i = 1; i <= 50; i++) {
            String s = String.format("-- n/a %d --", i);
            box.addItem(s);
        }
        box.setSelectedIndex(index);
    }

    private boolean stringIsValid(String s) {
        return (s != null && s.length() > 0);
    }

    private String getTextFieldContent(JTextField tf) {
        String result = null;
        if (tf != null && tf.getText().length() > 0) {
            result = tf.getText();
        }
        return result;
    }

    private void getQueryResult(QueryProcessor qr, String dbName, String tableName, String query, JTable table, boolean primary) {
        qr.setup(serverName, userName, password, serverPort, dbName, tableName, table);
        qr.execMySQLQuery(query);
        // if this is a data table query
        if (primary) {
            if (dbName == null) {
                dbName = "";
            } else {
                dbName = String.format("from `%s`.", dbName);
            }
            if (tableName == null) {
                tableName = "";
            }
            String et = this.gcFromTimeMS(qr.endTime - qr.startTime);
            setStatus(String.format("Read %d records %s`%s` (%s)", qr.getRecordCount(), dbName, tableName, et));
        }
    }

    protected String joinStringCollection(AbstractCollection<String> array, String token) {
        String result = null;
        Iterator<String> iter = array.iterator();
        if (iter.hasNext()) {
            StringBuilder sb = new StringBuilder(iter.next());
            while (iter.hasNext()) {
                sb.append(token).append(iter.next());
            }
            result = sb.toString();
        }
        return result;
    }

    protected void formatRunQuery() {
        formatRunQuery2();
        // the first query execution sets up 
        // the query form fields and sizes
        // at which point a restore may work
        if (sv_currentQuery.canRestore()) {
            sv_currentQuery.restore();
            formatRunQuery2();
        }
    }

    // allow user to enter escaped characters
    protected String escapeQueryArg(String s) {
        for (String[] pair : escapeChars) {
            s = s.replaceAll(pair[0], pair[1]);
        }
        return s;
    }

    protected void formatRunQuery2() {
        ArrayList<String> incFields = new ArrayList<>();
        ArrayList<String> args = new ArrayList<>();
        for (QueryControlRow row : sv_currentQuery.queryRows) {
            args = row.buildQuery(args);
            if (row.getInclude()) {
                incFields.add(String.format("`%s`", row.getFieldName()));
            }
        }
        String fields = "*";
        if (incFields.size() != sv_currentQuery.queryRows.size()) {
            fields = this.joinStringCollection(incFields, ",");
        }
        StringBuilder sb = new StringBuilder();
        for (String s : args) {
            sb.append(" ");
            sb.append(s);
        }
        String arg = sb.toString().trim();
        arg = (arg.length() > 0) ? "WHERE " + arg : arg;
        if (sv_currentQuery.postfix != null) {
            String free2 = sv_currentQuery.postfix.getText().trim();
            if (free2.length() > 0) {
                arg += " " + free2;
            }
        }
        String result = fields;
        if (sv_currentQuery.prefix != null) {
            String s = sv_currentQuery.prefix.getText().trim();
            if (s.length() > 0) {
                result = s;
            }
        }
        readTable(arg, result);
    }

    private void setEllipsize(QueryProcessor qr, MouseEvent evt) {
        if (qr != null) {
            qr.setEllipsize((JCheckBox) evt.getSource());
        }
    }

    private void setDefaultTableContent(JTable table, String prompt) {
        //prompt = String.format("<html><h2>%s</h2></html>",prompt);
        //MyTableCellRenderer mr = new MyTableCellRenderer(this, true, false);
        table.setDefaultRenderer(Object.class, mtcr);
        table.setRowHeight(64);
        table.setBorder(new LineBorder(Color.red, 0));
        table.setModel(new javax.swing.table.DefaultTableModel(
                new Object[][]{
                    {prompt}
                },
                new String[]{
                    ""
                }));
        table.getParent().setBackground(Color.white);
        table.setAutoResizeMode(JTable.AUTO_RESIZE_NEXT_COLUMN);
        
        //setEllipsize(table, sv_tableEllipsizeCheckBox);
    }

    protected int tableRowHeight(int n) {
        return ((charHeight == 0) ? 12 : charHeight) * n + tablePaddingConstant * 2;
    }

    private void clearLog() {
        if (askUser("Okay to clear system log?")) {
            setDefaultLogTableContent();
        }
    }

    private void setDefaultLogTableContent() {
        String[] headers = new String[]{"DateTime", "Command", "Elapsed", "Errors"};
        logQuery = new QueryProcessor(this, this.sv_logEllipsizeCheckBox, false);
        logQuery.setup(serverName, userName, password, serverPort, null, null, logTable);
        logQuery.setup(headers);
        logQuery.populateResultTable(true);
        logEntriesLabel.setText(String.format("%d log entries", 0));
    }

    protected void logEventAction(String query, long st, long et, String errorText) {
        if (logQuery != null) {
            boolean error = (errorText != null && errorText.length() > 0);
            queryShowLabel.setForeground((error) ? changedColor : Color.black);
            if (error) {
                Beep.beep();
            }
            if (query == null) {
                query = "(none)";
            }
            String elt = this.gcFromTimeMS(et - st);
            //query = (query == null) ? "(none)" : query;
            errorText = (errorText == null) ? "(none)" : errorText;
            SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss.SSS z");
            String timeStamp = sdf.format(new Date());
            String[] logEntry = new String[]{timeStamp, query, elt, errorText};
            logQuery.addRecord(logEntry);
            final int rows = logTable.getRowCount();
            logTable.scrollRectToVisible(logTable.getCellRect(rows - 1, 0, true));
            logEntriesLabel.setText(String.format("%d log entries", rows));
        }
    }

    protected void copyQueryTableTSV() {
        if (queryResultTable != null) {
            String result = ExportMethods.tableToDelimText(queryResultTable, lineSep, "\t");
            writeToClipboard(result);
        }
    }

    protected String createHTML() {
        String result = null;
        if (queryResultTable != null) {
            String db = (String) sv_databaseComboBox.getSelectedItem();
            String table = (String) sv_tableComboBox.getSelectedItem();
            String title = db + "." + table;
            result = ExportMethods.tableToHTML(title, queryResultTable, lineSep, sv_tableEllipsizeCheckBox.isSelected());
        }
        return result;
    }

    protected void copyQueryTableHTML() {
        if (queryResultTable != null) {
            writeToClipboard(createHTML());
        }
    }

    protected void createTableWebPage() {
        if (queryResultTable != null) {
            String page = createHTML();
            String db = (String) sv_databaseComboBox.getSelectedItem();
            String table = (String) sv_tableComboBox.getSelectedItem();
            String path = String.format("%s/%s.%s.html", webPagePath, db, table);
            if (writeToFile(path, page)) {
                launchBrowser("file://" + path);
            }
        }
    }

    protected void copyQueryString() {
        if (sv_currentQuery != null) {
            writeToClipboard(sv_currentQuery.query);
        }
    }

    protected void copyTableDescription() {
        if (tableDescriptionQuery != null) {
            String result = ExportMethods.tableToDelimText(tableDescriptionQuery.displayTable, lineSep, "\t");
            writeToClipboard(result);
        }
    }

    protected void copyLogTable() {
        if (logQuery != null) {
            String result = ExportMethods.tableToDelimText(logQuery.displayTable, lineSep, "\t");
            writeToClipboard(result);
        }
    }

    protected void writeToClipboard(String s) {
        Clipboard clipboard = Toolkit.getDefaultToolkit().getSystemClipboard();
        Transferable transferable = new StringSelection(s);
        clipboard.setContents(transferable, null);
    }

    protected boolean writeToFile(String path, String data) {
        try (FileWriter fw = new FileWriter(new File(path))) {
            fw.write(data);
            return true;
        } catch (Exception e) {
            e.printStackTrace(System.out);
            return false;
        }
    }

    protected void setStatus(String s) {
        statusLabel.setText(s);
    }

    public void launchBrowser(String url) {
        try {
            Desktop desktop = Desktop.getDesktop();
            desktop.browse(new URI(url));
        } catch (URISyntaxException | IOException e) {
            e.printStackTrace(System.out);
        }
    }

    protected java.awt.GridBagConstraints createConstraints(int x, int y, int width, float weight, Insets insets) {
        java.awt.GridBagConstraints gbc = new java.awt.GridBagConstraints();
        gbc.gridx = x;
        gbc.gridy = y;
        gbc.gridwidth = width;
        gbc.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gbc.anchor = java.awt.GridBagConstraints.NORTH;
        gbc.weightx = weight;
        gbc.weighty = 0;
        if (insets != null) {
            gbc.insets = insets;
        } else {
            gbc.insets = new Insets(0, 2, 0, 2);
        }
        return gbc;
    }

    protected boolean askUser(String prompt) {
        Beep.beep();
        int choice = JOptionPane.showConfirmDialog(this, prompt);
        return choice == JOptionPane.OK_OPTION;
    }

    protected void tellUser(String prompt) {
        Beep.beep();
        JOptionPane.showMessageDialog(this, prompt);
    }

    protected String getUserInput(String prompt) {
        Beep.beep();
        return JOptionPane.showInputDialog(this, prompt);
    }

    protected String getUserPassword(String prompt) {
        Beep.beep();
        String pw = null;
        JLabel lbl = new JLabel(prompt);
        final JPasswordField jpf = new JPasswordField(20);
        JPanel panel = new JPanel(new BorderLayout());
        panel.add(lbl, BorderLayout.NORTH);
        panel.add(jpf, BorderLayout.SOUTH);
        JOptionPane pane = new JOptionPane(panel,
                JOptionPane.QUESTION_MESSAGE,
                JOptionPane.OK_CANCEL_OPTION);
        JDialog dialog = pane.createDialog(programName);
        dialog.setIconImage(appIcon);
        // all this just to get focus on the entry field
        dialog.addWindowListener(new WindowAdapter() {
            @Override
            public void windowActivated(WindowEvent e) {
                SwingUtilities.invokeLater(new Runnable() {
                    @Override
                    public void run() {
                        jpf.requestFocusInWindow();
                    }
                });
            }
        });
        dialog.setVisible(true);
        if (pane != null) {
            int result = (int) pane.getValue();
            dialog.dispose();
            if (result == JOptionPane.OK_OPTION) {
                pw = new String(jpf.getPassword());
            }
        }
        return pw;
    }

    protected void restoreTabs() {
        tabbedPanel.setVisible(true);
    }

    protected void resizeEvent() {
        performTabActions(null);
    }

    protected int getDividerMaxLocation() {
        try {
            tableTabPanel.setVisible(true);
            Point hp = mainSplitPane.getLocationOnScreen();
            Point tp = tableTabPanel.getLocationOnScreen();
            int th = tableTabPanel.getHeight();
            // the divider is 10 pixels wide
            int delta = tp.y + th - 10 - hp.y;
            return delta;
        } catch (Exception e) {
            return 0;
        }
    }

    protected double ntrp(double x, double xa, double xb, double ya, double yb) {
        return (x - xa) * (yb - ya) / (xb - xa) + ya;
    }

    protected void adjustUserDividerPosition() {
        if (!inhibit) {
            int max = mainSplitPane.getMaximumDividerLocation();
            int min = mainSplitPane.getMinimumDividerLocation();
            int p = mainSplitPane.getDividerLocation();
            double v = (max - min == 0) ? 0.5 : ntrp(p, min, max, 0, 1);
            if (v >= .1 && v <= .9) {
                sv_dividerLocation = v;
            }
        }
    }

    protected int getUserDividerPosition() {
        int max = mainSplitPane.getMaximumDividerLocation();
        int min = mainSplitPane.getMinimumDividerLocation();
        return (int) ntrp(sv_dividerLocation, 0, 1, min, max);
    }

    protected boolean performTabActions(Component comp) {
        if (!inhibit) {
            inhibit = true;
            Component ctab;
            if (comp == null) {
                ctab = sv_mainTabbedPane.getSelectedComponent();
            } else {
                ctab = comp;
            }
            if (currentTab == editingPanel && ctab != editingPanel && !askAbandonEdit()) {
                sv_mainTabbedPane.setSelectedComponent(currentTab);
                inhibit = false;
                return false;
            }
            if (ctab == editingPanel && !allFieldsSelected) {
                if (currentTab != null) {
                    tellUser("Cannot edit records unless all fields\nare present in the query result.");
                    sv_mainTabbedPane.setSelectedComponent(currentTab);
                } else {
                    sv_mainTabbedPane.setSelectedComponent(queryPanel);
                }
                inhibit = false;
                return false;
            }
            sv_mainTabbedPane.setSelectedComponent(ctab);
            String tabName = ctab.getName();
            switch (tabName) {
                case "Setup":
                    mainSplitPane.setDividerLocation(0);
                    break;
                case "Edit":
                    mainSplitPane.setDividerLocation(getUserDividerPosition());
                    break;
                case "Query":
                case "TableDesc":
                    mainSplitPane.setDividerLocation(getUserDividerPosition());
                    break;
                case "Table":
                    mainSplitPane.setDividerLocation(getDividerMaxLocation());
                    break;
                case "Terminal":
                    mainSplitPane.setDividerLocation(0);
                    terminalCommandTextField.grabFocus();
                    break;
                case "Help":
                    mainSplitPane.setDividerLocation(0);
                    break;
            }
            currentTab = ctab;
            inhibit = false;
        }
        return true;
    }

    protected void manageWordWrap() {
        if (editFunctions != null && askAbandonEdit()) {
            editFunctions.populateEditPane();
        }
    }

    protected void toResizeEvent() {
        if (editFunctions != null) {
            editFunctions.resizeEvent();
        }
    }

    protected String gcFromTimeMS(long timeMS) {
        GregorianCalendar gc = new GregorianCalendar();
        gc.setTimeInMillis(timeMS);
        SimpleDateFormat sdf = new SimpleDateFormat((timeMS < 3600000) ? "mm:ss.SSS" : "HH:mm:ss.SSS");
        return sdf.format(gc.getTime());
    }

    private void closeApp() {
        if (askAbandonEdit()) {
            config.write();
            System.exit(0);
        }
    }

    public void p(String s) {
        System.out.println(s);
    }

    /**
     * This method is called from within the constructor to initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is always
     * regenerated by the Form Editor.
     */
    @SuppressWarnings("unchecked")
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {
        java.awt.GridBagConstraints gridBagConstraints;

        mainSplitPane = new MySplitPane(this);
        tablePanel = new javax.swing.JPanel();
        topControlPanel = new javax.swing.JPanel();
        jLabel1 = new javax.swing.JLabel();
        sv_databaseComboBox = new javax.swing.JComboBox<>();
        jLabel2 = new javax.swing.JLabel();
        sv_tableComboBox = new javax.swing.JComboBox<>();
        sv_tableEllipsizeCheckBox = new javax.swing.JCheckBox();
        tableScrollPane = new javax.swing.JScrollPane();
        queryResultTable = new MyJTable(this);
        tabbedPanel = new javax.swing.JPanel();
        sv_mainTabbedPane = new javax.swing.JTabbedPane();
        setupLogPanel = new javax.swing.JPanel();
        setupPanelTop = new javax.swing.JPanel();
        serverLabel = new javax.swing.JLabel();
        sv_serverTextField = new javax.swing.JTextField();
        userLabel = new javax.swing.JLabel();
        sv_userTextField = new javax.swing.JTextField();
        passwordLabel = new javax.swing.JLabel();
        passwordTextField = new JPasswordField();
        sv_portTextField = new javax.swing.JTextField();
        portLabel = new javax.swing.JLabel();
        setupSubPanel = new javax.swing.JPanel();
        jLabel6 = new javax.swing.JLabel();
        sv_fontSizeComboBox = new javax.swing.JComboBox<>();
        restartButton = new javax.swing.JButton();
        logEntriesLabel = new javax.swing.JLabel();
        sv_logEllipsizeCheckBox = new javax.swing.JCheckBox();
        logCopyButton = new javax.swing.JButton();
        logCclearButton = new javax.swing.JButton();
        logTableScrollPane = new javax.swing.JScrollPane();
        logTable = new MyJTable(this);
        tableTabPanel = new javax.swing.JPanel();
        editingPanel = new javax.swing.JPanel();
        editScrollPane = new javax.swing.JScrollPane();
        editControlPanel = new CustomJPanel();
        editButtonPanel = new javax.swing.JPanel();
        selectedRecordLabel = new javax.swing.JLabel();
        changedRecordsLabel = new javax.swing.JLabel();
        editDeleteButton = new javax.swing.JButton();
        editNewButton = new javax.swing.JButton();
        editCopyButton = new javax.swing.JButton();
        editCommitButton = new javax.swing.JButton();
        editCancelButton = new javax.swing.JButton();
        sv_editAllowCheckBox = new javax.swing.JCheckBox();
        jLabel4 = new javax.swing.JLabel();
        sv_wordWrapCheckBox = new javax.swing.JCheckBox();
        priorButton = new javax.swing.JButton();
        nextButton = new javax.swing.JButton();
        queryPanel = new javax.swing.JPanel();
        queryMiddlePanel = new javax.swing.JPanel();
        queryTopPanel = new javax.swing.JPanel();
        queryShowLabel = new javax.swing.JLabel();
        copyQueryButton = new javax.swing.JButton();
        queryClearButton = new javax.swing.JButton();
        clearHistoriesButton = new javax.swing.JButton();
        copyTSVButton = new javax.swing.JButton();
        copyHTMLButton = new javax.swing.JButton();
        launchBrowserButton = new javax.swing.JButton();
        queryScrollPane = new javax.swing.JScrollPane();
        queryLayoutPanel = new javax.swing.JPanel();
        queryBottomPanel = new javax.swing.JPanel();
        tableDescriptionPanel = new javax.swing.JPanel();
        tableDescriptionMainPanel = new javax.swing.JPanel();
        tableDescriptionScrollPane = new javax.swing.JScrollPane();
        tableDescriptionTable = new MyJTable(this);
        tableDescriptionBottomPanel = new javax.swing.JPanel();
        tableDescriptionLabel = new javax.swing.JLabel();
        sv_tableDescriptionEllipsizeCheckBox = new javax.swing.JCheckBox();
        addKeyButton = new javax.swing.JButton();
        tableBackupButton = new javax.swing.JButton();
        tableDescriptionCopyButton = new javax.swing.JButton();
        terminalPanel = new javax.swing.JPanel();
        terminalMainPanel = new javax.swing.JPanel();
        jScrollPane1 = new javax.swing.JScrollPane();
        terminalTextDisplay = new javax.swing.JTextArea();
        terminalControlPanel = new javax.swing.JPanel();
        jLabel5 = new javax.swing.JLabel();
        terminalCommandTextField = new javax.swing.JTextField();
        sv_termWrapCheckBox = new javax.swing.JCheckBox();
        terminalClearButton = new javax.swing.JButton();
        termCopyButton = new javax.swing.JButton();
        connectButton = new javax.swing.JButton();
        clearHistoryButton = new javax.swing.JButton();
        helpPanel = new javax.swing.JPanel();
        statusMainPanel = new javax.swing.JPanel();
        statusLabel = new javax.swing.JLabel();
        queryButton = new javax.swing.JButton();
        helpButton = new javax.swing.JButton();
        quitButton = new javax.swing.JButton();

        setDefaultCloseOperation(javax.swing.WindowConstants.DO_NOTHING_ON_CLOSE);
        addWindowListener(new java.awt.event.WindowAdapter() {
            public void windowClosing(java.awt.event.WindowEvent evt) {
                formWindowClosing(evt);
            }
        });
        addComponentListener(new java.awt.event.ComponentAdapter() {
            public void componentResized(java.awt.event.ComponentEvent evt) {
                formComponentResized(evt);
            }
        });

        mainSplitPane.setOrientation(javax.swing.JSplitPane.VERTICAL_SPLIT);
        mainSplitPane.setToolTipText("Adjust this divider location to meet your needs");
        mainSplitPane.setMaximumSize(new java.awt.Dimension(800, 800));

        tablePanel.setLayout(new java.awt.BorderLayout());

        topControlPanel.setLayout(new java.awt.GridBagLayout());

        jLabel1.setText("Database:");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.insets = new java.awt.Insets(0, 4, 0, 4);
        topControlPanel.add(jLabel1, gridBagConstraints);

        sv_databaseComboBox.setModel(new javax.swing.DefaultComboBoxModel<>(new String[] { "Item 1", "Item 2", "Item 3", "Item 4" }));
        sv_databaseComboBox.setToolTipText("Choose a database");
        sv_databaseComboBox.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                sv_databaseComboBoxActionPerformed(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        topControlPanel.add(sv_databaseComboBox, gridBagConstraints);

        jLabel2.setText("Table:");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.insets = new java.awt.Insets(0, 4, 0, 4);
        topControlPanel.add(jLabel2, gridBagConstraints);

        sv_tableComboBox.setModel(new javax.swing.DefaultComboBoxModel<>(new String[] { "Item 1", "Item 2", "Item 3", "Item 4" }));
        sv_tableComboBox.setToolTipText("Choose a table");
        sv_tableComboBox.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                sv_tableComboBoxActionPerformed(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        topControlPanel.add(sv_tableComboBox, gridBagConstraints);

        sv_tableEllipsizeCheckBox.setSelected(true);
        sv_tableEllipsizeCheckBox.setText("Ellipsize");
        sv_tableEllipsizeCheckBox.setToolTipText("Truncate table fields with ellipsis (...)");
        sv_tableEllipsizeCheckBox.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                sv_tableEllipsizeCheckBoxMouseClicked(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.insets = new java.awt.Insets(0, 4, 0, 4);
        topControlPanel.add(sv_tableEllipsizeCheckBox, gridBagConstraints);

        tablePanel.add(topControlPanel, java.awt.BorderLayout.NORTH);

        queryResultTable.setAutoCreateRowSorter(true);
        queryResultTable.setModel(new javax.swing.table.DefaultTableModel(
            new Object [][] {
                {null}
            },
            new String [] {
                "Title 1"
            }
        ));
        queryResultTable.setToolTipText("");
        queryResultTable.setAutoResizeMode(javax.swing.JTable.AUTO_RESIZE_ALL_COLUMNS);
        tableScrollPane.setViewportView(queryResultTable);

        tablePanel.add(tableScrollPane, java.awt.BorderLayout.CENTER);

        mainSplitPane.setTopComponent(tablePanel);

        tabbedPanel.setLayout(new java.awt.BorderLayout());

        sv_mainTabbedPane.setTabPlacement(javax.swing.JTabbedPane.BOTTOM);
        sv_mainTabbedPane.setName("sv_mainTabbedPane"); // NOI18N
        sv_mainTabbedPane.addChangeListener(new javax.swing.event.ChangeListener() {
            public void stateChanged(javax.swing.event.ChangeEvent evt) {
                sv_mainTabbedPaneStateChanged(evt);
            }
        });

        setupLogPanel.setName("Setup"); // NOI18N
        setupLogPanel.setLayout(new java.awt.BorderLayout());

        setupPanelTop.setLayout(new java.awt.GridBagLayout());

        serverLabel.setText("Server:");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.insets = new java.awt.Insets(0, 4, 0, 4);
        setupPanelTop.add(serverLabel, gridBagConstraints);

        sv_serverTextField.setText("localhost");
        sv_serverTextField.setToolTipText("Name of MySQL server");
        sv_serverTextField.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                sv_serverTextFieldActionPerformed(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        setupPanelTop.add(sv_serverTextField, gridBagConstraints);

        userLabel.setText("User:");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 4;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.insets = new java.awt.Insets(0, 2, 0, 2);
        setupPanelTop.add(userLabel, gridBagConstraints);

        sv_userTextField.setText("guest");
        sv_userTextField.setToolTipText("Name of user known to MySQL");
        sv_userTextField.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                sv_userTextFieldActionPerformed(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 5;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        setupPanelTop.add(sv_userTextField, gridBagConstraints);

        passwordLabel.setText("Password:");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 6;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.insets = new java.awt.Insets(0, 2, 0, 2);
        setupPanelTop.add(passwordLabel, gridBagConstraints);

        passwordTextField.setToolTipText("User's MySQL password");
        passwordTextField.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                passwordTextFieldActionPerformed(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 7;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        setupPanelTop.add(passwordTextField, gridBagConstraints);

        sv_portTextField.setHorizontalAlignment(javax.swing.JTextField.RIGHT);
        sv_portTextField.setText("3306");
        sv_portTextField.setToolTipText("MySQL server access port");
        sv_portTextField.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                sv_portTextFieldActionPerformed(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 3;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        setupPanelTop.add(sv_portTextField, gridBagConstraints);

        portLabel.setText("Port:");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.insets = new java.awt.Insets(0, 2, 0, 2);
        setupPanelTop.add(portLabel, gridBagConstraints);

        setupSubPanel.setLayout(new java.awt.GridBagLayout());

        jLabel6.setText("Font size:");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(0, 4, 0, 4);
        setupSubPanel.add(jLabel6, gridBagConstraints);

        sv_fontSizeComboBox.setToolTipText("Table display font size");
        sv_fontSizeComboBox.setMinimumSize(new java.awt.Dimension(60, 24));
        sv_fontSizeComboBox.setPreferredSize(new java.awt.Dimension(60, 24));
        sv_fontSizeComboBox.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                sv_fontSizeComboBoxActionPerformed(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
        gridBagConstraints.insets = new java.awt.Insets(0, 4, 0, 4);
        setupSubPanel.add(sv_fontSizeComboBox, gridBagConstraints);

        restartButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/jdbclient/icons/start.png"))); // NOI18N
        restartButton.setText("Start/Restart");
        restartButton.setToolTipText("Refresh system state with new entries");
        restartButton.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                restartButtonMouseClicked(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 6;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
        gridBagConstraints.insets = new java.awt.Insets(0, 2, 0, 2);
        setupSubPanel.add(restartButton, gridBagConstraints);

        logEntriesLabel.setText("Log Entries");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(0, 4, 0, 2);
        setupSubPanel.add(logEntriesLabel, gridBagConstraints);

        sv_logEllipsizeCheckBox.setSelected(true);
        sv_logEllipsizeCheckBox.setText("Ellipsize");
        sv_logEllipsizeCheckBox.setToolTipText("Truncate table fields with ellipsis (...)");
        sv_logEllipsizeCheckBox.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                sv_logEllipsizeCheckBoxMouseClicked(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 3;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
        gridBagConstraints.insets = new java.awt.Insets(0, 4, 0, 4);
        setupSubPanel.add(sv_logEllipsizeCheckBox, gridBagConstraints);

        logCopyButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/jdbclient/icons/record_copy.png"))); // NOI18N
        logCopyButton.setText("Copy Log");
        logCopyButton.setToolTipText("Copy log entries to system clipboard");
        logCopyButton.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                logCopyButtonMouseClicked(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 4;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
        gridBagConstraints.insets = new java.awt.Insets(0, 2, 0, 2);
        setupSubPanel.add(logCopyButton, gridBagConstraints);

        logCclearButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/jdbclient/icons/mail-replied.png"))); // NOI18N
        logCclearButton.setText("Clear Log");
        logCclearButton.setToolTipText("Clear log entry table");
        logCclearButton.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                logCclearButtonMouseClicked(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 5;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
        gridBagConstraints.insets = new java.awt.Insets(0, 2, 0, 2);
        setupSubPanel.add(logCclearButton, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.gridwidth = 8;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        setupPanelTop.add(setupSubPanel, gridBagConstraints);

        setupLogPanel.add(setupPanelTop, java.awt.BorderLayout.NORTH);

        logTable.setModel(new javax.swing.table.DefaultTableModel(
            new Object [][] {
                {null}
            },
            new String [] {
                "Title 1"
            }
        ));
        logTable.setName("Log Table"); // NOI18N
        logTable.setRowSelectionAllowed(false);
        logTableScrollPane.setViewportView(logTable);

        setupLogPanel.add(logTableScrollPane, java.awt.BorderLayout.CENTER);

        sv_mainTabbedPane.addTab("Setup/Log", new javax.swing.ImageIcon(getClass().getResource("/jdbclient/icons/preferences-system.png")), setupLogPanel); // NOI18N

        tableTabPanel.setName("Table"); // NOI18N
        sv_mainTabbedPane.addTab("Table", new javax.swing.ImageIcon(getClass().getResource("/jdbclient/icons/document-edit-verify.png")), tableTabPanel, "View query result table"); // NOI18N

        editingPanel.setName("Edit"); // NOI18N
        editingPanel.setLayout(new java.awt.BorderLayout());

        editScrollPane.addComponentListener(new java.awt.event.ComponentAdapter() {
            public void componentResized(java.awt.event.ComponentEvent evt) {
                editScrollPaneComponentResized(evt);
            }
        });

        editControlPanel.setBorder(javax.swing.BorderFactory.createEmptyBorder(4, 4, 4, 4));
        editControlPanel.setLayout(new java.awt.GridBagLayout());
        editScrollPane.setViewportView(editControlPanel);

        editingPanel.add(editScrollPane, java.awt.BorderLayout.CENTER);

        editButtonPanel.setLayout(new java.awt.GridBagLayout());

        selectedRecordLabel.setText("Selected record: --");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(0, 4, 0, 0);
        editButtonPanel.add(selectedRecordLabel, gridBagConstraints);

        changedRecordsLabel.setText("Changed records: --");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        editButtonPanel.add(changedRecordsLabel, gridBagConstraints);

        editDeleteButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/jdbclient/icons/delete.png"))); // NOI18N
        editDeleteButton.setText("Delete ...");
        editDeleteButton.setToolTipText("Delete the selected record");
        editDeleteButton.setIconTextGap(2);
        editDeleteButton.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                editDeleteButtonMouseClicked(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 3;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.insets = new java.awt.Insets(0, 1, 0, 1);
        editButtonPanel.add(editDeleteButton, gridBagConstraints);

        editNewButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/jdbclient/icons/new.png"))); // NOI18N
        editNewButton.setText("New");
        editNewButton.setToolTipText("Create a new, empty record");
        editNewButton.setIconTextGap(2);
        editNewButton.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                editNewButtonMouseClicked(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 4;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.insets = new java.awt.Insets(0, 1, 0, 1);
        editButtonPanel.add(editNewButton, gridBagConstraints);

        editCopyButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/jdbclient/icons/record_copy.png"))); // NOI18N
        editCopyButton.setText("Copy");
        editCopyButton.setToolTipText("<html>Make a copy of a record within the table<br/>\n(Only enabled for tables with a primary key)");
        editCopyButton.setIconTextGap(2);
        editCopyButton.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                editCopyButtonMouseClicked(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 5;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.insets = new java.awt.Insets(0, 1, 0, 1);
        editButtonPanel.add(editCopyButton, gridBagConstraints);

        editCommitButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/jdbclient/icons/checkmark.png"))); // NOI18N
        editCommitButton.setText("Commit");
        editCommitButton.setToolTipText("Commit edits to the database");
        editCommitButton.setIconTextGap(2);
        editCommitButton.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                editCommitButtonMouseClicked(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 6;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.insets = new java.awt.Insets(0, 1, 0, 1);
        editButtonPanel.add(editCommitButton, gridBagConstraints);

        editCancelButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/jdbclient/icons/cancel.png"))); // NOI18N
        editCancelButton.setText("Cancel ...");
        editCancelButton.setToolTipText("Cancel edit, restore original data");
        editCancelButton.setIconTextGap(2);
        editCancelButton.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                editCancelButtonMouseClicked(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 7;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.insets = new java.awt.Insets(0, 1, 0, 1);
        editButtonPanel.add(editCancelButton, gridBagConstraints);

        sv_editAllowCheckBox.setToolTipText("This prevents inadvertent edits");
        sv_editAllowCheckBox.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                sv_editAllowCheckBoxActionPerformed(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 8;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.insets = new java.awt.Insets(0, 8, 0, 0);
        editButtonPanel.add(sv_editAllowCheckBox, gridBagConstraints);

        jLabel4.setText("Allow Edits");
        jLabel4.setToolTipText("This prevents inadvertent edits");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 10;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.insets = new java.awt.Insets(0, 2, 0, 4);
        editButtonPanel.add(jLabel4, gridBagConstraints);

        sv_wordWrapCheckBox.setText("Wrap");
        sv_wordWrapCheckBox.setToolTipText("Wrap lines on long records");
        sv_wordWrapCheckBox.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                sv_wordWrapCheckBoxActionPerformed(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.insets = new java.awt.Insets(0, 2, 0, 2);
        editButtonPanel.add(sv_wordWrapCheckBox, gridBagConstraints);

        priorButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/jdbclient/icons/go-previous.png"))); // NOI18N
        priorButton.setToolTipText("Prior record");
        priorButton.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                priorButtonMouseClicked(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 11;
        gridBagConstraints.insets = new java.awt.Insets(0, 1, 0, 1);
        editButtonPanel.add(priorButton, gridBagConstraints);

        nextButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/jdbclient/icons/go-next.png"))); // NOI18N
        nextButton.setToolTipText("Next record");
        nextButton.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                nextButtonMouseClicked(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 12;
        gridBagConstraints.insets = new java.awt.Insets(0, 1, 0, 1);
        editButtonPanel.add(nextButton, gridBagConstraints);

        editingPanel.add(editButtonPanel, java.awt.BorderLayout.SOUTH);

        sv_mainTabbedPane.addTab("Edit/Enter", new javax.swing.ImageIcon(getClass().getResource("/jdbclient/icons/allow_edit.png")), editingPanel); // NOI18N

        queryPanel.setName("Query"); // NOI18N
        queryPanel.setLayout(new java.awt.GridBagLayout());

        queryMiddlePanel.setLayout(new java.awt.BorderLayout());

        queryTopPanel.setLayout(new java.awt.GridBagLayout());

        queryShowLabel.setText("Query:");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(0, 4, 0, 4);
        queryTopPanel.add(queryShowLabel, gridBagConstraints);

        copyQueryButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/jdbclient/icons/record_copy.png"))); // NOI18N
        copyQueryButton.setText("Copy Q");
        copyQueryButton.setToolTipText("<html>Copy the text of the query to the system clipboard<br/>(also see the log on the setup/log tab)");
        copyQueryButton.setIconTextGap(2);
        copyQueryButton.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                copyQueryButtonMouseClicked(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
        gridBagConstraints.insets = new java.awt.Insets(0, 1, 0, 1);
        queryTopPanel.add(copyQueryButton, gridBagConstraints);

        queryClearButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/jdbclient/icons/mail-replied.png"))); // NOI18N
        queryClearButton.setText("Clear Q");
        queryClearButton.setToolTipText("Reset query panel to defaults");
        queryClearButton.setIconTextGap(2);
        queryClearButton.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                queryClearButtonMouseClicked(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 5;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
        gridBagConstraints.insets = new java.awt.Insets(0, 1, 0, 1);
        queryTopPanel.add(queryClearButton, gridBagConstraints);

        clearHistoriesButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/jdbclient/icons/delete.png"))); // NOI18N
        clearHistoriesButton.setText("Clear H");
        clearHistoriesButton.setToolTipText("Clear \"Prefix\" and \"Postfix\" entry histories");
        clearHistoriesButton.setIconTextGap(2);
        clearHistoriesButton.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                clearHistoriesButtonMouseClicked(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 6;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
        gridBagConstraints.insets = new java.awt.Insets(0, 1, 0, 1);
        queryTopPanel.add(clearHistoriesButton, gridBagConstraints);

        copyTSVButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/jdbclient/icons/text-csv.png"))); // NOI18N
        copyTSVButton.setText("Copy TSV");
        copyTSVButton.setToolTipText("Copy Tab-Separated-Values (TSV) version of result table to system clipboard");
        copyTSVButton.setIconTextGap(2);
        copyTSVButton.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                copyTSVButtonMouseClicked(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
        gridBagConstraints.insets = new java.awt.Insets(0, 1, 0, 1);
        queryTopPanel.add(copyTSVButton, gridBagConstraints);

        copyHTMLButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/jdbclient/icons/text-html.png"))); // NOI18N
        copyHTMLButton.setText("Copy HTML");
        copyHTMLButton.setToolTipText("Copy HTML version of result table to system clipboard");
        copyHTMLButton.setIconTextGap(2);
        copyHTMLButton.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                copyHTMLButtonMouseClicked(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 3;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
        gridBagConstraints.insets = new java.awt.Insets(0, 1, 0, 1);
        queryTopPanel.add(copyHTMLButton, gridBagConstraints);

        launchBrowserButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/jdbclient/icons/browser.png"))); // NOI18N
        launchBrowserButton.setText("Browser");
        launchBrowserButton.setToolTipText("Launch system browser showing HTML result table");
        launchBrowserButton.setIconTextGap(2);
        launchBrowserButton.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                launchBrowserButtonMouseClicked(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 4;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
        gridBagConstraints.insets = new java.awt.Insets(0, 1, 0, 1);
        queryTopPanel.add(launchBrowserButton, gridBagConstraints);

        queryMiddlePanel.add(queryTopPanel, java.awt.BorderLayout.PAGE_START);

        queryLayoutPanel.setBorder(javax.swing.BorderFactory.createEmptyBorder(4, 4, 4, 4));
        queryLayoutPanel.setLayout(new java.awt.GridBagLayout());
        queryScrollPane.setViewportView(queryLayoutPanel);

        queryMiddlePanel.add(queryScrollPane, java.awt.BorderLayout.CENTER);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTH;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.weighty = 1.0;
        queryPanel.add(queryMiddlePanel, gridBagConstraints);

        queryBottomPanel.setLayout(new java.awt.GridBagLayout());
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.SOUTH;
        gridBagConstraints.weightx = 1.0;
        queryPanel.add(queryBottomPanel, gridBagConstraints);

        sv_mainTabbedPane.addTab("Query", new javax.swing.ImageIcon(getClass().getResource("/jdbclient/icons/edit-find.png")), queryPanel); // NOI18N

        tableDescriptionPanel.setName("TableDesc"); // NOI18N
        tableDescriptionPanel.setLayout(new java.awt.BorderLayout());

        tableDescriptionMainPanel.setName("tableDescriptionMainPanel"); // NOI18N
        tableDescriptionMainPanel.setLayout(new java.awt.BorderLayout());

        tableDescriptionTable.setModel(new javax.swing.table.DefaultTableModel(
            new Object [][] {
                {null}
            },
            new String [] {
                "Title 1"
            }
        ));
        tableDescriptionTable.setToolTipText("This panel shows the current table's technical definition");
        tableDescriptionTable.setIntercellSpacing(new java.awt.Dimension(0, 0));
        tableDescriptionTable.setRowSelectionAllowed(false);
        tableDescriptionScrollPane.setViewportView(tableDescriptionTable);

        tableDescriptionMainPanel.add(tableDescriptionScrollPane, java.awt.BorderLayout.CENTER);

        tableDescriptionPanel.add(tableDescriptionMainPanel, java.awt.BorderLayout.CENTER);

        tableDescriptionBottomPanel.setLayout(new java.awt.GridBagLayout());

        tableDescriptionLabel.setText("Table:");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(0, 4, 0, 2);
        tableDescriptionBottomPanel.add(tableDescriptionLabel, gridBagConstraints);

        sv_tableDescriptionEllipsizeCheckBox.setSelected(true);
        sv_tableDescriptionEllipsizeCheckBox.setText("Ellipsize");
        sv_tableDescriptionEllipsizeCheckBox.setToolTipText("Truncate table fields with ellipsis (...)");
        sv_tableDescriptionEllipsizeCheckBox.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                sv_tableDescriptionEllipsizeCheckBoxMouseClicked(evt);
            }
        });
        tableDescriptionBottomPanel.add(sv_tableDescriptionEllipsizeCheckBox, new java.awt.GridBagConstraints());

        addKeyButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/jdbclient/icons/insert-link.png"))); // NOI18N
        addKeyButton.setText("Add Key");
        addKeyButton.setToolTipText("Add primary key to this table");
        addKeyButton.setEnabled(false);
        addKeyButton.setIconTextGap(2);
        addKeyButton.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                addKeyButtonMouseClicked(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.insets = new java.awt.Insets(0, 1, 0, 1);
        tableDescriptionBottomPanel.add(addKeyButton, gridBagConstraints);

        tableBackupButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/jdbclient/icons/document-save-as.png"))); // NOI18N
        tableBackupButton.setText("Backup ...");
        tableBackupButton.setToolTipText("Make a backup copy of this table");
        tableBackupButton.setIconTextGap(2);
        tableBackupButton.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                tableBackupButtonMouseClicked(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.insets = new java.awt.Insets(0, 1, 0, 1);
        tableDescriptionBottomPanel.add(tableBackupButton, gridBagConstraints);

        tableDescriptionCopyButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/jdbclient/icons/record_copy.png"))); // NOI18N
        tableDescriptionCopyButton.setText("Copy");
        tableDescriptionCopyButton.setToolTipText("Copy table description to system clipboard");
        tableDescriptionCopyButton.setIconTextGap(2);
        tableDescriptionCopyButton.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                tableDescriptionCopyButtonMouseClicked(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.insets = new java.awt.Insets(0, 1, 0, 1);
        tableDescriptionBottomPanel.add(tableDescriptionCopyButton, gridBagConstraints);

        tableDescriptionPanel.add(tableDescriptionBottomPanel, java.awt.BorderLayout.SOUTH);

        sv_mainTabbedPane.addTab("Table Description", new javax.swing.ImageIcon(getClass().getResource("/jdbclient/icons/code-class.png")), tableDescriptionPanel); // NOI18N

        terminalPanel.setName("Terminal"); // NOI18N
        terminalPanel.setLayout(new java.awt.BorderLayout());

        terminalMainPanel.setLayout(new java.awt.BorderLayout());

        terminalTextDisplay.setColumns(20);
        terminalTextDisplay.setRows(5);
        terminalTextDisplay.setToolTipText("<html>To copy content from this display, click the \"Copy\" button below,<br/>\nor drag the mouse cursor across an area of interest and type Ctrl+C (Copy).");
        jScrollPane1.setViewportView(terminalTextDisplay);

        terminalMainPanel.add(jScrollPane1, java.awt.BorderLayout.CENTER);

        terminalPanel.add(terminalMainPanel, java.awt.BorderLayout.CENTER);

        terminalControlPanel.setLayout(new java.awt.GridBagLayout());

        jLabel5.setText("Command:");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.insets = new java.awt.Insets(0, 4, 0, 4);
        terminalControlPanel.add(jLabel5, gridBagConstraints);

        terminalCommandTextField.setToolTipText("<html>Type SQL commands<br/>\"Enter\" executes, arrow keys browse history");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridwidth = 6;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        terminalControlPanel.add(terminalCommandTextField, gridBagConstraints);

        sv_termWrapCheckBox.setText("Wrap lines");
        sv_termWrapCheckBox.setToolTipText("Enable line wrapping in terminal window");
        sv_termWrapCheckBox.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                sv_termWrapCheckBoxMouseClicked(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
        gridBagConstraints.weightx = 1.0;
        terminalControlPanel.add(sv_termWrapCheckBox, gridBagConstraints);

        terminalClearButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/jdbclient/icons/mail-replied.png"))); // NOI18N
        terminalClearButton.setText("Clear Display");
        terminalClearButton.setToolTipText("Erase the terminal windows's contents");
        terminalClearButton.setIconTextGap(2);
        terminalClearButton.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                terminalClearButtonMouseClicked(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 3;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
        gridBagConstraints.insets = new java.awt.Insets(0, 1, 0, 1);
        terminalControlPanel.add(terminalClearButton, gridBagConstraints);

        termCopyButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/jdbclient/icons/record_copy.png"))); // NOI18N
        termCopyButton.setText("Copy");
        termCopyButton.setToolTipText("Copy terminal output to system clipboard");
        termCopyButton.setIconTextGap(2);
        termCopyButton.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                termCopyButtonMouseClicked(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 5;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
        gridBagConstraints.insets = new java.awt.Insets(0, 1, 0, 1);
        terminalControlPanel.add(termCopyButton, gridBagConstraints);

        connectButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/jdbclient/icons/start.png"))); // NOI18N
        connectButton.setText("Connect");
        connectButton.setToolTipText("<html>Connect / disconnect using the information<br/>entered on the Setup/Log pane");
        connectButton.setIconTextGap(2);
        connectButton.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                connectButtonMouseClicked(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 6;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
        gridBagConstraints.insets = new java.awt.Insets(0, 1, 0, 1);
        terminalControlPanel.add(connectButton, gridBagConstraints);

        clearHistoryButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/jdbclient/icons/delete.png"))); // NOI18N
        clearHistoryButton.setText("Clear History");
        clearHistoryButton.setToolTipText("Clear command history");
        clearHistoryButton.setIconTextGap(2);
        clearHistoryButton.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                clearHistoryButtonMouseClicked(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 4;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.insets = new java.awt.Insets(0, 1, 0, 1);
        terminalControlPanel.add(clearHistoryButton, gridBagConstraints);

        terminalPanel.add(terminalControlPanel, java.awt.BorderLayout.SOUTH);

        sv_mainTabbedPane.addTab("SQL Terminal", new javax.swing.ImageIcon(getClass().getResource("/jdbclient/icons/kig.png")), terminalPanel); // NOI18N

        helpPanel.setName("Help"); // NOI18N
        helpPanel.setLayout(new java.awt.BorderLayout());
        sv_mainTabbedPane.addTab("Help", new javax.swing.ImageIcon(getClass().getResource("/jdbclient/icons/help.png")), helpPanel); // NOI18N

        tabbedPanel.add(sv_mainTabbedPane, java.awt.BorderLayout.CENTER);

        mainSplitPane.setBottomComponent(tabbedPanel);

        getContentPane().add(mainSplitPane, java.awt.BorderLayout.CENTER);

        statusMainPanel.setLayout(new java.awt.GridBagLayout());

        statusLabel.setText("Status");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(0, 4, 0, 2);
        statusMainPanel.add(statusLabel, gridBagConstraints);

        queryButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/jdbclient/icons/checkmark.png"))); // NOI18N
        queryButton.setText("Query");
        queryButton.setToolTipText("Execute the current query");
        queryButton.setIconTextGap(2);
        queryButton.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                queryButtonMouseClicked(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
        statusMainPanel.add(queryButton, gridBagConstraints);

        helpButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/jdbclient/icons/help.png"))); // NOI18N
        helpButton.setText("Help");
        helpButton.setToolTipText("Visit the JDBClient home page");
        helpButton.setIconTextGap(2);
        helpButton.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                helpButtonMouseClicked(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 7;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
        statusMainPanel.add(helpButton, gridBagConstraints);

        quitButton.setIcon(new javax.swing.ImageIcon(getClass().getResource("/jdbclient/icons/quitIcon.png"))); // NOI18N
        quitButton.setText("Quit");
        quitButton.setToolTipText("EXit JDBClient");
        quitButton.setIconTextGap(2);
        quitButton.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                quitButtonMouseClicked(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 8;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.EAST;
        statusMainPanel.add(quitButton, gridBagConstraints);

        getContentPane().add(statusMainPanel, java.awt.BorderLayout.SOUTH);

        pack();
    }// </editor-fold>//GEN-END:initComponents

    private void formWindowClosing(java.awt.event.WindowEvent evt) {//GEN-FIRST:event_formWindowClosing
        // TODO add your handling code here:
        closeApp();
    }//GEN-LAST:event_formWindowClosing

    private void sv_tableEllipsizeCheckBoxMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_sv_tableEllipsizeCheckBoxMouseClicked
        // TODO add your handling code here:
        setEllipsize(sv_currentQuery, evt);
    }//GEN-LAST:event_sv_tableEllipsizeCheckBoxMouseClicked

    private void restartButtonMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_restartButtonMouseClicked
        // TODO add your handling code here:
        readDatabaseList(true);
    }//GEN-LAST:event_restartButtonMouseClicked

    private void sv_serverTextFieldActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_sv_serverTextFieldActionPerformed
        // TODO add your handling code here:
        readDatabaseList(true);
    }//GEN-LAST:event_sv_serverTextFieldActionPerformed

    private void sv_userTextFieldActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_sv_userTextFieldActionPerformed
        // TODO add your handling code here:
        readDatabaseList(true);
    }//GEN-LAST:event_sv_userTextFieldActionPerformed

    private void passwordTextFieldActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_passwordTextFieldActionPerformed
        // TODO add your handling code here:
        readDatabaseList(true);
    }//GEN-LAST:event_passwordTextFieldActionPerformed

    private void sv_databaseComboBoxActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_sv_databaseComboBoxActionPerformed
        // TODO add your handling code here:
        readTableList();
    }//GEN-LAST:event_sv_databaseComboBoxActionPerformed

    private void sv_tableComboBoxActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_sv_tableComboBoxActionPerformed
        // TODO add your handling code here:
        tableSelected();
    }//GEN-LAST:event_sv_tableComboBoxActionPerformed

    private void sv_logEllipsizeCheckBoxMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_sv_logEllipsizeCheckBoxMouseClicked
        // TODO add your handling code here:
        setEllipsize(logQuery, evt);
    }//GEN-LAST:event_sv_logEllipsizeCheckBoxMouseClicked

    private void queryButtonMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_queryButtonMouseClicked
        // TODO add your handling code here:
        formatRunQuery();
    }//GEN-LAST:event_queryButtonMouseClicked

    private void quitButtonMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_quitButtonMouseClicked
        // TODO add your handling code here:
        closeApp();
    }//GEN-LAST:event_quitButtonMouseClicked

    private void helpButtonMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_helpButtonMouseClicked
        // TODO add your handling code here:
        launchBrowser("http://arachnoid.com/JDBClient");
    }//GEN-LAST:event_helpButtonMouseClicked

    private void sv_tableDescriptionEllipsizeCheckBoxMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_sv_tableDescriptionEllipsizeCheckBoxMouseClicked
        // TODO add your handling code here:
        setEllipsize(tableDescriptionQuery, evt);
    }//GEN-LAST:event_sv_tableDescriptionEllipsizeCheckBoxMouseClicked

    private void copyTSVButtonMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_copyTSVButtonMouseClicked
        // TODO add your handling code here:
        copyQueryTableTSV();
    }//GEN-LAST:event_copyTSVButtonMouseClicked

    private void copyHTMLButtonMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_copyHTMLButtonMouseClicked
        // TODO add your handling code here:
        copyQueryTableHTML();
    }//GEN-LAST:event_copyHTMLButtonMouseClicked

    private void launchBrowserButtonMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_launchBrowserButtonMouseClicked
        // TODO add your handling code here:
        createTableWebPage();
    }//GEN-LAST:event_launchBrowserButtonMouseClicked

    private void sv_fontSizeComboBoxActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_sv_fontSizeComboBoxActionPerformed
        // TODO add your handling code here:
        resetFontSize(true);
    }//GEN-LAST:event_sv_fontSizeComboBoxActionPerformed

    private void copyQueryButtonMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_copyQueryButtonMouseClicked
        // TODO add your handling code here:
        copyQueryString();
    }//GEN-LAST:event_copyQueryButtonMouseClicked

    private void tableDescriptionCopyButtonMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_tableDescriptionCopyButtonMouseClicked
        // TODO add your handling code here:
        copyTableDescription();
    }//GEN-LAST:event_tableDescriptionCopyButtonMouseClicked

    private void addKeyButtonMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_addKeyButtonMouseClicked
        // TODO add your handling code here:
        editFunctions.addPrimaryKey();
    }//GEN-LAST:event_addKeyButtonMouseClicked

    private void logCopyButtonMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_logCopyButtonMouseClicked
        // TODO add your handling code here:
        copyLogTable();
    }//GEN-LAST:event_logCopyButtonMouseClicked

    private void sv_mainTabbedPaneStateChanged(javax.swing.event.ChangeEvent evt) {//GEN-FIRST:event_sv_mainTabbedPaneStateChanged
        // TODO add your handling code here:
        performTabActions(null);
    }//GEN-LAST:event_sv_mainTabbedPaneStateChanged

    private void sv_editAllowCheckBoxActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_sv_editAllowCheckBoxActionPerformed
        // TODO add your handling code here:
        editFunctions.setAllowEditState();
    }//GEN-LAST:event_sv_editAllowCheckBoxActionPerformed

    private void editCancelButtonMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_editCancelButtonMouseClicked
        // TODO add your handling code here:
        editFunctions.cancelEdit();
    }//GEN-LAST:event_editCancelButtonMouseClicked

    private void editCopyButtonMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_editCopyButtonMouseClicked
        // TODO add your handling code here:
        editFunctions.copyRecord();
    }//GEN-LAST:event_editCopyButtonMouseClicked

    private void editDeleteButtonMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_editDeleteButtonMouseClicked
        // TODO add your handling code here:
        editFunctions.deleteRecord();
    }//GEN-LAST:event_editDeleteButtonMouseClicked

    private void editNewButtonMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_editNewButtonMouseClicked
        // TODO add your handling code here:
        editFunctions.newRecord();
    }//GEN-LAST:event_editNewButtonMouseClicked

    private void editCommitButtonMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_editCommitButtonMouseClicked
        // TODO add your handling code here:
        editFunctions.commitEdit();
    }//GEN-LAST:event_editCommitButtonMouseClicked

    private void connectButtonMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_connectButtonMouseClicked
        // TODO add your handling code here:
        terminal.connectDisconnect();
    }//GEN-LAST:event_connectButtonMouseClicked

    private void terminalClearButtonMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_terminalClearButtonMouseClicked
        // TODO add your handling code here:
        terminal.clear();
    }//GEN-LAST:event_terminalClearButtonMouseClicked

    private void sv_termWrapCheckBoxMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_sv_termWrapCheckBoxMouseClicked
        // TODO add your handling code here:
        terminal.wrapControl();
    }//GEN-LAST:event_sv_termWrapCheckBoxMouseClicked

    private void termCopyButtonMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_termCopyButtonMouseClicked
        // TODO add your handling code here:
        terminal.copyToClipboard();
    }//GEN-LAST:event_termCopyButtonMouseClicked

    private void queryClearButtonMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_queryClearButtonMouseClicked
        // TODO add your handling code here:
        sv_currentQuery.populateQueryPanel(tableDescriptionQuery);
    }//GEN-LAST:event_queryClearButtonMouseClicked

    private void clearHistoryButtonMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_clearHistoryButtonMouseClicked
        // TODO add your handling code here:
        this.sv_termEngine.clearHistory();
    }//GEN-LAST:event_clearHistoryButtonMouseClicked

    private void formComponentResized(java.awt.event.ComponentEvent evt) {//GEN-FIRST:event_formComponentResized
        // TODO add your handling code here:
        resizeEvent();
    }//GEN-LAST:event_formComponentResized

    private void tableBackupButtonMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_tableBackupButtonMouseClicked
        // TODO add your handling code here:
        backupTable();
    }//GEN-LAST:event_tableBackupButtonMouseClicked

    private void clearHistoriesButtonMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_clearHistoriesButtonMouseClicked
        // TODO add your handling code here:
        clearHistories();
    }//GEN-LAST:event_clearHistoriesButtonMouseClicked

    private void sv_portTextFieldActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_sv_portTextFieldActionPerformed
        // TODO add your handling code here:
        readDatabaseList(true);
    }//GEN-LAST:event_sv_portTextFieldActionPerformed

    private void logCclearButtonMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_logCclearButtonMouseClicked
        // TODO add your handling code here:
        clearLog();
    }//GEN-LAST:event_logCclearButtonMouseClicked

    private void sv_wordWrapCheckBoxActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_sv_wordWrapCheckBoxActionPerformed
        // TODO add your handling code here:
        manageWordWrap();
    }//GEN-LAST:event_sv_wordWrapCheckBoxActionPerformed

    private void editScrollPaneComponentResized(java.awt.event.ComponentEvent evt) {//GEN-FIRST:event_editScrollPaneComponentResized
        // TODO add your handling code here:

        toResizeEvent();
    }//GEN-LAST:event_editScrollPaneComponentResized

    private void priorButtonMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_priorButtonMouseClicked
        // TODO add your handling code here:
        this.editFunctions.moveToTableRowDelta(-1);
    }//GEN-LAST:event_priorButtonMouseClicked

    private void nextButtonMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_nextButtonMouseClicked
        // TODO add your handling code here:
        this.editFunctions.moveToTableRowDelta(1);
    }//GEN-LAST:event_nextButtonMouseClicked

    /**
     * @param args the command line arguments
     */
    public static void main(final String args[]) {
        /* Set the Nimbus look and feel */
        //<editor-fold defaultstate="collapsed" desc=" Look and feel setting code (optional) ">
        /* If Nimbus (introduced in Java SE 6) is not available, stay with the default look and feel.
         * For details see http://download.oracle.com/javase/tutorial/uiswing/lookandfeel/plaf.html 
         */
        try {
            for (javax.swing.UIManager.LookAndFeelInfo info : javax.swing.UIManager.getInstalledLookAndFeels()) {
                if ("Nimbus".equals(info.getName())) {
                    javax.swing.UIManager.setLookAndFeel(info.getClassName());
                    break;
                }
            }
        } catch (ClassNotFoundException | InstantiationException | IllegalAccessException | javax.swing.UnsupportedLookAndFeelException ex) {
            java.util.logging.Logger.getLogger(JDBClient.class.getName()).log(java.util.logging.Level.SEVERE, null, ex);
        }
        //</editor-fold>

        /* Create and display the form */
        java.awt.EventQueue.invokeLater(new Runnable() {
            @Override
            public void run() {
                new JDBClient(args).setVisible(true);
            }
        });
    }
    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JButton addKeyButton;
    protected javax.swing.JLabel changedRecordsLabel;
    private javax.swing.JButton clearHistoriesButton;
    private javax.swing.JButton clearHistoryButton;
    protected javax.swing.JButton connectButton;
    private javax.swing.JButton copyHTMLButton;
    private javax.swing.JButton copyQueryButton;
    private javax.swing.JButton copyTSVButton;
    private javax.swing.JPanel editButtonPanel;
    protected javax.swing.JButton editCancelButton;
    protected javax.swing.JButton editCommitButton;
    protected javax.swing.JPanel editControlPanel;
    protected javax.swing.JButton editCopyButton;
    protected javax.swing.JButton editDeleteButton;
    protected javax.swing.JButton editNewButton;
    protected javax.swing.JScrollPane editScrollPane;
    protected javax.swing.JPanel editingPanel;
    private javax.swing.JButton helpButton;
    private javax.swing.JPanel helpPanel;
    private javax.swing.JLabel jLabel1;
    private javax.swing.JLabel jLabel2;
    private javax.swing.JLabel jLabel4;
    private javax.swing.JLabel jLabel5;
    private javax.swing.JLabel jLabel6;
    private javax.swing.JScrollPane jScrollPane1;
    private javax.swing.JButton launchBrowserButton;
    private javax.swing.JButton logCclearButton;
    private javax.swing.JButton logCopyButton;
    private javax.swing.JLabel logEntriesLabel;
    private javax.swing.JTable logTable;
    private javax.swing.JScrollPane logTableScrollPane;
    private javax.swing.JSplitPane mainSplitPane;
    protected javax.swing.JButton nextButton;
    private javax.swing.JLabel passwordLabel;
    protected javax.swing.JTextField passwordTextField;
    private javax.swing.JLabel portLabel;
    protected javax.swing.JButton priorButton;
    private javax.swing.JPanel queryBottomPanel;
    private javax.swing.JButton queryButton;
    private javax.swing.JButton queryClearButton;
    protected javax.swing.JPanel queryLayoutPanel;
    protected javax.swing.JPanel queryMiddlePanel;
    private javax.swing.JPanel queryPanel;
    protected javax.swing.JTable queryResultTable;
    protected javax.swing.JScrollPane queryScrollPane;
    private javax.swing.JLabel queryShowLabel;
    private javax.swing.JPanel queryTopPanel;
    private javax.swing.JButton quitButton;
    private javax.swing.JButton restartButton;
    protected javax.swing.JLabel selectedRecordLabel;
    private javax.swing.JLabel serverLabel;
    private javax.swing.JPanel setupLogPanel;
    private javax.swing.JPanel setupPanelTop;
    private javax.swing.JPanel setupSubPanel;
    private javax.swing.JLabel statusLabel;
    private javax.swing.JPanel statusMainPanel;
    protected javax.swing.JComboBox<String> sv_databaseComboBox;
    protected javax.swing.JCheckBox sv_editAllowCheckBox;
    protected javax.swing.JComboBox<Integer> sv_fontSizeComboBox;
    protected javax.swing.JCheckBox sv_logEllipsizeCheckBox;
    protected javax.swing.JTabbedPane sv_mainTabbedPane;
    protected javax.swing.JTextField sv_portTextField;
    protected javax.swing.JTextField sv_serverTextField;
    protected javax.swing.JComboBox<String> sv_tableComboBox;
    protected javax.swing.JCheckBox sv_tableDescriptionEllipsizeCheckBox;
    protected javax.swing.JCheckBox sv_tableEllipsizeCheckBox;
    protected javax.swing.JCheckBox sv_termWrapCheckBox;
    protected javax.swing.JTextField sv_userTextField;
    protected javax.swing.JCheckBox sv_wordWrapCheckBox;
    private javax.swing.JPanel tabbedPanel;
    private javax.swing.JButton tableBackupButton;
    private javax.swing.JPanel tableDescriptionBottomPanel;
    private javax.swing.JButton tableDescriptionCopyButton;
    private javax.swing.JLabel tableDescriptionLabel;
    private javax.swing.JPanel tableDescriptionMainPanel;
    private javax.swing.JPanel tableDescriptionPanel;
    private javax.swing.JScrollPane tableDescriptionScrollPane;
    private javax.swing.JTable tableDescriptionTable;
    private javax.swing.JPanel tablePanel;
    private javax.swing.JScrollPane tableScrollPane;
    private javax.swing.JPanel tableTabPanel;
    private javax.swing.JButton termCopyButton;
    private javax.swing.JButton terminalClearButton;
    protected javax.swing.JTextField terminalCommandTextField;
    private javax.swing.JPanel terminalControlPanel;
    private javax.swing.JPanel terminalMainPanel;
    private javax.swing.JPanel terminalPanel;
    protected javax.swing.JTextArea terminalTextDisplay;
    private javax.swing.JPanel topControlPanel;
    private javax.swing.JLabel userLabel;
    // End of variables declaration//GEN-END:variables
}
