// ***************************************************************************
// *   Copyright (C) 2012 by Paul Lutus                                      *
// *   lutusp@arachnoid.com                                                  *
// *                                                                         *
// *   This program is free software; you can redistribute it and/or modify  *
// *   it under the terms of the GNU General Public License as published by  *
// *   the Free Software Foundation; either version 2 of the License, or     *
// *   (at your option) any later version.                                   *
// *                                                                         *
// *   This program is distributed in the hope that it will be useful,       *
// *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
// *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
// *   GNU General Public License for more details.                          *
// *                                                                         *
// *   You should have received a copy of the GNU General Public License     *
// *   along with this program; if not, write to the                         *
// *   Free Software Foundation, Inc.,                                       *
// *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
// ***************************************************************************
package jdbclient;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Insets;
import java.awt.Point;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.TreeMap;
import javax.swing.JCheckBox;
import javax.swing.JLabel;
import javax.swing.JScrollBar;
import javax.swing.JTable;
import javax.swing.JTextField;
import javax.swing.table.DefaultTableModel;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Statement;

/**
 *
 * @author lutusp
 */
final public class QueryProcessor implements Configurable {

    JDBClient parent;
    JCheckBox checkBox;
    JTextField prefix = null, postfix = null;
    Map<Object, Integer> colWidths;
    Map<Integer, Integer> rowHeights;
    ArrayList<ArrayList<Object>> sqlData;
    Map<String, ArrayList<Object>> fieldMap = null;
    ArrayList<Object> sqlColNames = null;
    ArrayList<QueryControlRow> queryRows;
    String serverName;
    String userName;
    String password;
    String mysqlPort;
    String db = null;
    String table = null;
    String query = null;
    JTable displayTable = null;
    DefaultTableModel tableModel = null;
    ArrayList<String> primaryKeyFields = null;
    String recDelimiter = "<\t>";
    ArrayList<String> restoreState = null;
    boolean columnSortable = false;
    long startTime, endTime;

    public QueryProcessor(JDBClient p, JCheckBox box, boolean cs) {
        parent = p;
        checkBox = box;
        columnSortable = cs;
        restoreState = new ArrayList<>();
        prefix = new JTextField();
        postfix = new JTextField();
        reset();
    }

    protected void reset() {
        queryRows = new ArrayList<>();
        sqlData = new ArrayList<>();
        sqlColNames = new ArrayList<>();
        colWidths = new HashMap<>();
        rowHeights = new HashMap<>();
        primaryKeyFields = new ArrayList<>();
    }

    public void setup(String sn, String user, String pw, String mport, String qdb, String qtable, JTable qt) {
        serverName = sn;
        userName = user;
        password = pw;
        mysqlPort = mport;
        db = qdb;
        table = qtable;
        displayTable = qt;
    }

    protected void addQueryRow(QueryControlRow qr) {
        queryRows.add(qr);
    }

    private void updateWidthHeight(ArrayList<Object> vo, int row, JTable localTable) {
        int col = 0;
        int rowHeight = 1;
        for (Object k : sqlColNames) {
            String key = k.toString();
            if (col >= vo.size()) {
                break;
            }
            Object obj = vo.get(col);
            int strLen;
            if (obj != null) {
                String s = obj.toString();
                // must check widths of multiline fields
                String[] array = s.split("\n", -1);
                int cellHeight = array.length;
                rowHeight = Math.max(rowHeight, cellHeight);
                int width = 0;
                if (colWidths.containsKey(key)) {
                    width = colWidths.get(key);
                }
                for (String v : array) {
                    strLen = v.length();
                    width = Math.max(strLen, width);
                }
                colWidths.put(key, width);
                if (localTable != null) {
                    width = width * parent.charWidth + parent.tablePaddingConstant * 4;
                    localTable.getColumn(key).setPreferredWidth(width);
                }
            }
            col += 1;
        }
        if (row >= 0) {
            rowHeights.put(row, rowHeight);
        }
    }

    public void addRecord(ArrayList<Object> vo, int row, JTable table) {
        // this the table's data store, so adding a record here
        // adds it to the displayed table's model
        sqlData.add(vo);
        //p("addrecord: " + vo);
        updateWidthHeight(vo, row, table);
    }

    public ArrayList<ArrayList<Object>> getData() {
        return sqlData;
    }

    public ArrayList<Object> getColumnNames() {
        return sqlColNames;
    }

    public Map<Object, Integer> getColumnWidths() {
        return colWidths;
    }

    public int getColumnWidth(String s) {
        if (colWidths.containsKey(s)) {
            return colWidths.get(s);
        }
        return 0;
    }

    public int getRecordCount() {
        return sqlData.size();
    }

    public void execMySQLQuery(String query) {
        startTime = System.currentTimeMillis();
        String dbUrl = parent.makeDBUrl(db);
        this.query = query;
        String queryText = null;
        String errorText = null;
        if (parent.validString(dbUrl)
                && parent.validString(userName)
                && parent.validString(password)
                && parent.validString(query)) {
            try {
                queryText = query;
                parent.queryDisp(query);
                Class.forName("com.mysql.jdbc.Driver").newInstance();
                Connection conn = DriverManager.getConnection(dbUrl, userName, password);
                Statement stmt = conn.createStatement();
                String escQuery = parent.escapeQueryArg(query);
                ResultSet rs = stmt.executeQuery(escQuery);
                ResultSetMetaData md = rs.getMetaData();
                int cols = md.getColumnCount();
                sqlColNames = new ArrayList<>();
                // get column names
                for (int i = 0; i < cols; i++) {
                    String cl = md.getColumnLabel(i + 1);
                    sqlColNames.add(cl);
                }
                updateWidthHeight(sqlColNames, -1, null);
                sqlData = new ArrayList<>();
                ArrayList<Object> dataRecord;
                int row = 0;
                // read all records
                while (rs.next()) {
                    dataRecord = new ArrayList<>();
                    // read fields from record
                    for (Object objkey : sqlColNames) {
                        String key = objkey.toString().trim();
                        Object obj = null;
                        try {
                            obj = rs.getObject(key);
                        } catch (Exception e) {
                            //e.printStackTrace();
                        }
                        dataRecord.add(obj);
                    }
                    addRecord(dataRecord, row, null);
                    row += 1;
                }
            } catch (Exception e) {
                if (parent.debug) {
                    e.printStackTrace(System.out);
                }
                errorText = e.getMessage();
            }
        }
        endTime = System.currentTimeMillis();
        parent.logEventAction(String.format("Q: %s", query), startTime, endTime, errorText);
    }

    protected int execMySQLUpdate(String query) {
        startTime = System.currentTimeMillis();
        String errorText = null;
        int changed = 0;
        try {
            String dbUrl = parent.makeDBUrl("");
            java.sql.Connection conn = DriverManager.getConnection(dbUrl, userName, password);
            Statement stmt = conn.createStatement();
            String escQuery = parent.escapeQueryArg(query);
            changed = stmt.executeUpdate(escQuery);
        } catch (Exception e) {
            if (parent.debug) {
                e.printStackTrace(System.out);
            }
            errorText = e.getMessage();
        }
        endTime = System.currentTimeMillis();
        parent.logEventAction(String.format("Q: %s", query), startTime, endTime, errorText);
        return changed;
    }

    protected void populateQueryPanel(QueryProcessor qr) {
        parent.queryLayoutPanel.removeAll();
        if (parent.editFunctions != null) {
            parent.editFunctions.activateEdit(-1);
        }
        if (qr != null && qr.getRecordCount() > 0) {
            prefix.setText("");
            postfix.setText("");
            queryRows = new ArrayList<>();
            // speed up vertical scrolling
            parent.queryScrollPane.getVerticalScrollBar().setUnitIncrement(16);
            int x = 0;
            int y = 0;
            Insets insets = new Insets(-1, 2, -1, 2);
            JLabel freeLabel1 = new JLabel("<html><b>Prefix arguments:</b></html>");
            freeLabel1.setFont(parent.baseFont);

            prefix.setToolTipText("<html>Enter SQL to precede any entries below such as a selection filter<br/>May be blank, \"Enter\" executes, arrow keys browse history");
            parent.queryLayoutPanel.add(freeLabel1, parent.createConstraints(0, y, 3, 0f, insets));
            parent.queryLayoutPanel.add(prefix, parent.createConstraints(3, y, 6, 1f, insets));
            y += 1;
            ArrayList<ArrayList<Object>> colNames = qr.getData();
            for (ArrayList<Object> obj : colNames) {
                //String fn = String.format("Name %d", y);
                addQueryRow(new QueryControlRow(parent, obj.get(0).toString(), y, this));
                y += 1;
            }
            JLabel freeLabel2 = new JLabel("<html><b>Postfix arguments:</b></html>");
            freeLabel2.setFont(parent.baseFont);
            postfix.setToolTipText("<html>Enter SQL to follow any entries above like a WHERE or GROUP BY clause<br/>May be blank, \"Enter\" executes, arrow keys browse history");
            parent.queryLayoutPanel.add(freeLabel2, parent.createConstraints(0, y, 3, 0f, insets));
            parent.queryLayoutPanel.add(postfix, parent.createConstraints(3, y, 6, 1f, insets));
            parent.repaint();
        }
    }

    protected void setWidthsHeights() {
        if (displayTable != null) {
            if (sqlColNames.size() > 0 && colWidths.size() > 0) {
                for (Object obj : sqlColNames) {
                    if (colWidths.containsKey(obj)) {
                        int w = colWidths.get(obj) * parent.charWidth + parent.tablePaddingConstant * 4;
                        displayTable.getColumn(obj).setPreferredWidth(w);
                    }
                }
                if (rowHeights.size() > 0) {
                    for (int n = 0; n < displayTable.getRowCount(); n++) {
                        if (rowHeights.containsKey(n)) {
                            displayTable.setRowHeight(n, parent.tableRowHeight(rowHeights.get(n)));
                        }
                    }
                } else {
                    // default single height setting for entire table
                    displayTable.setRowHeight(parent.tableRowHeight(1));
                }
            }
        }
    }

    protected Object getRecordField(ArrayList<Object> rec, String key) {
        Object result = null;
        if (sqlColNames != null && rec != null) {
            if (sqlColNames.contains(key)) {
                int col = sqlColNames.indexOf(key);
                result = rec.get(col);
            }
        }
        return result;
    }

    protected Object getRecordField(int row, String key) {
        Object obj = null;
        if (row >= 0 && row < sqlData.size()) {
            obj = getRecordField(sqlData.get(row), key);
        }
        return obj;
    }

    protected Object getRecordField(String fieldName, String key) {
        if (fieldMap != null) {
            return getRecordField(fieldMap.get(fieldName), key);
        } else {
            return null;
        }
    }

    protected ArrayList<Object> getRecordByFieldName(String fieldName) {
        if (fieldMap != null) {
            return fieldMap.get(fieldName);
        } else {
            return null;
        }
    }

    protected void tableDescSetup() {
        mapFieldNamesToDescriptions();
        mapPrimaryKeys();
    }

    // this only makes sense for a table description
    // where the field column values are unique
    protected void mapFieldNamesToDescriptions() {
        fieldMap = new TreeMap<>();
        for (ArrayList<Object> rec : sqlData) {
            String key = rec.get(0).toString();
            fieldMap.put(key, rec);
        }
    }

    // this is used by the table description routines
    // to determine whether a record can be safely copied
    protected boolean mapPrimaryKeys() {
        boolean result = false;
        for (ArrayList<Object> rec : sqlData) {
            String keyVal = (String) getRecordField(rec, "Key");
            if (keyVal != null && keyVal.equals("PRI")) {
                primaryKeyFields.add((String) getRecordField(rec, "Field"));
                result = true;
            }
        }
        return result;
    }

    // this is a secondary use for this class --
    // as a log entry accumulator
    public void setup(String[] columnNames) {
        sqlColNames.addAll(Arrays.asList(columnNames));
    }

    public void addRecord(String[] record) {
        ArrayList<Object> vs = new ArrayList<>();
        vs.addAll(Arrays.asList(record));
        //vs.add(record);
        int row = displayTable.getRowCount();
        // must do both these actions -- one updates the table's internal data store
        // and the other updates the local ArrayList so the
        // table can be restored after a font size change
        ((DefaultTableModel) displayTable.getModel()).addRow(vs.toArray());
        addRecord(vs, row, displayTable);
        displayTable.setRowHeight(parent.tableRowHeight(1));
    }

    protected int getOriginalTableRow(int row) {
        int result = -1;
        if (displayTable != null && row >= 0 && row < displayTable.getRowCount() && displayTable.getColumnCount() > 0) {
            result = displayTable.convertRowIndexToModel(row);
        }
        return result;
    }

    // create a table model that returns the 
    // true class of its columns rather than "Object"
    // to allow intelligent column sorting
    protected DefaultTableModel createSortableTableModel() {
        // this is how we avoid any reference to class "Vector"
        Object[][] data = new Object[getData().size()][];
        int row = 0;
        for (ArrayList<Object> ao : getData()) {
            data[row++] = ao.toArray();
        }
        return new DefaultTableModel(data, getColumnNames().toArray()) {
            static final long serialVersionUID = 23954;
            @Override
            public Class<?> getColumnClass(int column) {
                if (column >= 0 && column < getColumnCount() && getRowCount() > 0) {
                    Object obj = getValueAt(0, column);
                    return (obj == null) ? Object.class : obj.getClass();
                } else {
                    return Object.class;
                }
            }
        };
    }

    protected void populateResultTable(boolean errorDisp) {
        try {
            tableModel = createSortableTableModel();
            displayTable.setModel(tableModel);
            // allow/prevent sorting by clicking on column headers
            displayTable.setAutoCreateRowSorter(columnSortable);
            setWidthsHeights();
            // This may look strange, but it's how to create
            // a one-pixel line between adjacent cells
            // and no uncontrolled margon
            displayTable.setIntercellSpacing(new Dimension(-4, -4));
            displayTable.setGridColor(parent.tableBorderColor);
            displayTable.setShowGrid(true);
            displayTable.setDefaultRenderer(Object.class, new MyTableCellRenderer(parent, false, errorDisp));
            displayTable.getParent().setBackground(Color.white);
            setEllipsize(checkBox);
        } catch (Exception e) {
            e.printStackTrace(System.out);
        }
    }

    protected void setEllipsize(JCheckBox box) {
        if (box != null) {
            boolean ellipsize = box.isSelected();
            displayTable.setAutoResizeMode(ellipsize ? JTable.AUTO_RESIZE_NEXT_COLUMN : JTable.AUTO_RESIZE_OFF);
        }
    }

    protected void vtab(MyJTextField tf, int v) {
        int i = 0;
        int j = 0;
        int ip;
        for (QueryControlRow qr : queryRows) {
            if (tf == qr.leftEntryField) {
                j = 0;
                break;
            } else if (tf == qr.rightEntryField) {
                j = 1;
                break;
            }
            i += 1;
        }
        int sz = queryRows.size();
        ip = Math.abs((i + v + sz) % sz);
        if (ip >= 0 && ip < queryRows.size()) {
            QueryControlRow tr = queryRows.get(ip);
            MyJTextField targ = (j == 0) ? tr.leftEntryField : tr.rightEntryField;
            targ.grabFocus();
            Point p = targ.getLocation();
            JScrollBar vs = parent.queryScrollPane.getVerticalScrollBar();
            vs.setValue(p.y - vs.getHeight() / 2);
        }

    }

    protected boolean canRestore() {
        boolean restorable = (restoreState.size() == queryRows.size() + 2);
        return restorable;
    }

    protected void restore() {
        if (canRestore()) {
            prefix.setText(restoreState.remove(0));
            postfix.setText(restoreState.remove(restoreState.size() - 1));
            for (QueryControlRow qr : queryRows) {
                qr.fromString(restoreState.remove(0));
            }
        }
    }

    @Override
    public void fromString(String data) {
        // the "-1" preserves null fields in the split
        String[] array = data.split(recDelimiter, -1);
        restoreState.addAll(Arrays.asList(array));
    }

    @Override
    public String toString() {

        ArrayList<String> array = new ArrayList<>();
        array.add((prefix == null) ? "" : prefix.getText());
        for (QueryControlRow row : queryRows) {
            array.add(row.toString());
        }
        array.add((postfix == null) ? "" : postfix.getText());
        return parent.joinStringCollection(array, recDelimiter);
    }

    public void p(String s) {
        System.out.println(s);
    }
}
