// ***************************************************************************
// *   Copyright (C) 2017 by Paul Lutus                                      *
// *   lutusp@arachnoid.com                                                  *
// *                                                                         *
// *   This program is free software; you can redistribute it and/or modify  *
// *   it under the terms of the GNU General Public License as published by  *
// *   the Free Software Foundation; either version 2 of the License, or     *
// *   (at your option) any later version.                                   *
// *                                                                         *
// *   This program is distributed in the hope that it will be useful,       *
// *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
// *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
// *   GNU General Public License for more details.                          *
// *                                                                         *
// *   You should have received a copy of the GNU General Public License     *
// *   along with this program; if not, write to the                         *
// *   Free Software Foundation, Inc.,                                       *
// *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
// ***************************************************************************
package jsqliteclient;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Insets;
import java.awt.Point;
import java.awt.Rectangle;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.TreeMap;
import javax.swing.JLabel;
import javax.swing.JScrollBar;
import javax.swing.JScrollPane;
import javax.swing.SwingUtilities;
import javax.swing.text.DefaultCaret;

/**
 *
 * @author lutusp
 */
final public class EditFunctions {

    JSQLiteClient parent;
    int userSelectedRow = -1, realRow = -1;
    boolean isChanged = false;
    boolean allowEdits = false;
    Map<String, MyJTextArea> editControls;
    ArrayList<MyJTextArea> editList;
    private String focusedControlFieldName = null;
    JLabel chooseLabel;
    boolean commitInProgress = false;
    int currentFocusedRow = -1;

    public EditFunctions(JSQLiteClient p) {
        parent = p;
        chooseLabel = new JLabel();
        getAllowEditState();
        populateEditPane();
        parent.editScrollPane.getVerticalScrollBar().setUnitIncrement(16);
    }

    protected void setAllowEditState() {
        if (!askAbandonEdit()) {
            parent.sv_editAllowCheckBox.setSelected(allowEdits);
        }
        getAllowEditState();
    }

    protected boolean getAllowEditState() {
        allowEdits = parent.sv_editAllowCheckBox.isSelected();
        chooseLabel.setEnabled(allowEdits);
        boolean active = (allowEdits && userSelectedRow >= 0 && parent.sv_currentQuery != null && parent.sv_currentQuery.getRecordCount() > 0);
        if (editControls != null && editControls.size() > 0) {
            for (String key : editControls.keySet()) {
                editControls.get(key).setEnabled(active);
            }
        }
        parent.priorButton.setEnabled(active);
        parent.nextButton.setEnabled(active);
        parent.sv_wordWrapCheckBox.setEnabled(active);
        parent.editCancelButton.setEnabled(isChanged && active);
        parent.editCommitButton.setEnabled(isChanged && active);
        parent.editCopyButton.setEnabled(active && !isChanged && parent.tableDescriptionQuery.mapPrimaryKeys());
        parent.editNewButton.setEnabled(!isChanged && allowEdits && parent.sv_currentQuery != null);
        parent.editDeleteButton.setEnabled(active && !isChanged);
        return allowEdits;
    }

    protected boolean askAbandonEdit() {
        if (isChanged) {
            boolean agree = parent.askUser("Okay to abandon an incomplete edit?");
            if (agree) {
                doCancel();
            } else {
                refocus();
            }
            return agree;
        } else {
            return true;
        }
    }

    protected void activateEdit(int row) {
        if (commitInProgress) {
            return;
        }
        if (!askAbandonEdit()) {
            refocus();
            return;
        }
        // if row < 0, this signal is meant to cancel a selection
        // so don't do this:
        //System.out.println("current focused row: " + currentFocusedRow);
        //if(row < 0 && currentFocusedRow >= 0) {
            //row = currentFocusedRow;
        //}
        isChanged = false;
        userSelectedRow = row;
        realRow = -1;
        if (row >= 0) {
            setRealRow();
            parent.performTabActions(parent.editingPanel);
        }
        populateEditPane();
        getAllowEditState();
        currentFocusedRow = userSelectedRow;
        identifySelectedRow();
    }
    
    protected void moveToTableRowDelta(int delta) {
        int max = parent.queryResultTable.getRowCount();
        currentFocusedRow += delta;
        currentFocusedRow = (currentFocusedRow < 0)?0:currentFocusedRow;
        currentFocusedRow = (currentFocusedRow > max-1)?max-1:currentFocusedRow;
        focusRow(currentFocusedRow);
        activateEdit(currentFocusedRow);
    }
    
    protected void identifySelectedRow() {
        if (currentFocusedRow >= 0) {
            parent.selectedRecordLabel.setText(String.format("Selected record: %d", currentFocusedRow  + 1));
        } else {
            parent.selectedRecordLabel.setText("No selected record");
        }
    }

    protected void setChanged() {
        isChanged = true;
        getAllowEditState();
    }

    protected void setRealRow() {
        realRow = parent.sv_currentQuery.getOriginalTableRow(userSelectedRow);
    }

    protected void refocus() {
        if (focusedControlFieldName != null) {
            editControls.get(focusedControlFieldName).grabFocus();
        }
    }

    protected void saveFocus(String fn) {
        focusedControlFieldName = fn;
    }

    protected void cancelEdit() {
        if (isChanged) {
            if (!parent.askUser("Okay to cancel current edit?")) {
                refocus();
            } else {
                doCancel();
            }
        }
    }

    protected void doCancel() {
        isChanged = false;
        getAllowEditState();
        populateEditPane();
        refocus();
    }

    protected void focusRow(int row) {
        currentFocusedRow = -1;
        if (parent.queryResultTable != null) {
            //row = parent.sv_currentQuery.getRecordCount() - 1;
            if (row >= 0 && row < parent.queryResultTable.getRowCount()) {
                parent.queryResultTable.setRowSelectionInterval(row, row);
                Rectangle rect = new Rectangle((parent.queryResultTable.getCellRect(row, 0, true)));
                parent.queryResultTable.scrollRectToVisible(rect);
                currentFocusedRow = row;
                identifySelectedRow();
            }
        }
    }

    protected void focusBottomRow() {
        if (parent.queryResultTable != null) {
            focusRow(parent.queryResultTable.getRowCount() - 1);
        }
    }

    protected String dateTimeFormat(String format) {
        SimpleDateFormat sdf = new SimpleDateFormat(format);
        return "'" + sdf.format(new Date()) + "'";
    }
    
    

    protected void getFieldValue(String key, Object orig, ArrayList<String> fields, ArrayList<String> values) {
        // don't include primary keys
        if (!parent.tableDescriptionQuery.primaryKeyFields.contains(key)) {
            //System.out.println("key: " + key);
            LinkedHashMap<String,Object> desc = parent.tableDescriptionQuery.getRecordByFieldName(key);
            //System.out.println("Key: " + key + ", value: " + desc);
            fields.add(String.format("`%s`", key));
            String def = (String) desc.get("default");
            if (orig != null) {
                // this comes with "copy record"
                values.add(String.format("'%s'", escapeSQL(orig)));
            } else {
                // a new record gets default value
                values.add(def);
            }
        }
    }

    protected void newRecord() {
        if (parent.sv_currentQuery != null
                && askAbandonEdit()) {
            //String db = parent.sv_currentQuery.db;
            String table = parent.sv_currentQuery.table;
            //System.out.println("table: " + table);
            ArrayList<String> fields = new ArrayList<>();
            ArrayList<String> values = new ArrayList<>();
            for (String key : parent.sv_currentQuery.sqlColNames) {
                //String key =  obj;
                getFieldValue(key, null, fields, values);
            }
            String fs = String.join(",",fields);//parent.joinStringCollection(fields, ",");
            String vs = String.join(",",values);//parent.joinStringCollection(values, ",");
            String sql = String.format("INSERT INTO `%s` (%s) VALUES (%s)", table, fs, vs);
            int changed = parent.sv_currentQuery.execSQLiteUpdate(sql);
            updateChangedLabel(changed);
            parent.formatRunQuery();
            // assume new record goes to the end
            userSelectedRow = parent.queryResultTable.getRowCount() - 1;
            setRealRow();
            populateEditPane();
            focusRow(userSelectedRow);
            clearChanged();
            refocus();
        }
    }

    protected void deleteRecord() {
        if (userSelectedRow >= 0
                && parent.sv_currentQuery != null
                && parent.sv_currentQuery.getRecordCount() > 0
                && askAbandonEdit()
                && parent.askUser(String.format("Okay to delete record %d?", userSelectedRow + 1))) {
            // parent.p("A");
            Map<String, Object> original = getOriginalRecord();
            ArrayList<String> where = new ArrayList<>();
            for (Object obj : parent.sv_currentQuery.sqlColNames) {
                String key = (String) obj;
                Object orig = original.get(key);
                where.add(equalsPhrase(key, orig, false));
            }
            //String db = parent.sv_currentQuery.db;
            String table = parent.sv_currentQuery.table;
            String set = String.join(" AND",where);//, elements)parent.joinStringCollection(where, " AND ");
            String sql = String.format("DELETE FROM `%s` WHERE %s", table, set);
            //String sql = assembleDeleteString(original);
            int changed = parent.sv_currentQuery.execSQLiteUpdate(sql);
            updateChangedLabel(changed);
            parent.formatRunQuery();
            //int top = parent.sv_currentQuery.getRecordCount();
            userSelectedRow = -1;
            populateEditPane();
            setRealRow();
            clearChanged();
            refocus();
        }
    }

    protected void copyRecord() {
        if (realRow >= 0 && parent.sv_currentQuery != null && parent.tableDescriptionQuery != null && askAbandonEdit()) {
            Map<String, Object> original = getOriginalRecord();
            ArrayList<String> fields = new ArrayList<>();
            ArrayList<String> values = new ArrayList<>();
            for (Object obj : parent.sv_currentQuery.sqlColNames) {
                String key = (String) obj;
                Object orig = original.get(key);
                getFieldValue(key, orig, fields, values);
            }
            //String db = parent.sv_currentQuery.db;
            String table = parent.sv_currentQuery.table;
            String flds = String.join(",",fields);//, elements)parent.joinStringCollection(fields, ",");
            String vals = String.join(",",values);//parent.joinStringCollection(values, ",");
            String sql = String.format("INSERT INTO `%s` (%s) VALUES (%s)", table, flds, vals);
            //String sql = assembleCopyString(original);
            int changed = parent.sv_currentQuery.execSQLiteUpdate(sql);
            updateChangedLabel(changed);
            parent.formatRunQuery();
            // assume copy goes to the end
            userSelectedRow = parent.queryResultTable.getRowCount() - 1;
            setRealRow();
            populateEditPane();
            focusRow(userSelectedRow);
            clearChanged();
            refocus();
        }
    }

    protected void updateChangedLabel(int changed) {
        parent.changedRecordsLabel.setText(String.format("Changed records: %d", changed));
        parent.changedRecordsLabel.setForeground((changed == 0) ? parent.changedColor : Color.black);
    }

    protected void commitEdit() {
        if (realRow >= 0 && parent.sv_currentQuery != null) {
            Map<String, Object> original = getOriginalRecord();
            Map<String, MyJTextArea> edited = getEditedRecord();
            String sql = assembleEditString(original, edited);
            int changed = parent.sv_currentQuery.execSQLiteUpdate(sql);
            updateChangedLabel(changed);
            commitInProgress = true;
            parent.formatRunQuery();
            commitInProgress = false;
            populateEditPane();
            setRealRow();
            clearChanged();
            refocus();
            parent.sv_currentQuery.displayTable.setRowSelectionInterval(realRow, realRow);
            focusRow(realRow);
        }
    }

    protected String assembleEditString(Map<String, Object> original, Map<String, MyJTextArea> edited) {
        ArrayList<String> update = new ArrayList<>();
        ArrayList<String> where = new ArrayList<>();
        //String db = parent.sv_currentQuery.db;
        String table = parent.sv_currentQuery.table;
        for (Object obj : parent.sv_currentQuery.sqlColNames) {
            String key = (String) obj;
            Object orig = original.get(key);
            MyJTextArea edit = edited.get(key);
            if (edit.changed) {
                String se = edit.getText();
                if (se.equals("NULL")) {
                    se = null;
                }
                update.add(equalsPhrase(key, se, true));
            }
            where.add(equalsPhrase(key, orig, false));
        }
        String set = String.join(" , ",update);//, elements)parent.joinStringCollection(update, " , ");
        String whs = String.join(" AND ",where);//, elements)parent.joinStringCollection(where, " AND ");
        return String.format("UPDATE `%s` SET %s WHERE %s", table, set, whs);
    }

    protected String equalsPhrase(String key, Object value, boolean assign) {
        if (value == null) {
            if (assign) {
                return String.format("`%s` = NULL", key);
            } else {
                return String.format("`%s` is NULL", key);
            }
        } else {
            String s = escapeSQL(value);
            String dtype = parent.dataType(key);
            // must not quote boolean values
            String fmtstr = (dtype.equals("tinyint(1)")?"`%s` = %s":"`%s` = '%s'");
            return String.format(fmtstr, key, s);
        }
    }

    protected String escapeSQL(Object value) {
        String s;
        if (value == null) {
            s = "NULL";
        } else {
            s = value.toString();
            s = s.replaceAll("\\\\", "\\\\\\\\");
            s = s.replaceAll("'", "''");
        }
        return s;
    }

    protected Map<String, Object> getOriginalRecord() {
        Map<String, Object> map = new TreeMap<>();
        for (Object obj : parent.sv_currentQuery.sqlColNames) {
            String key = (String) obj;
            Object targ = parent.sv_currentQuery.getRecordField(realRow, key);
            map.put(key, targ);
        }
        return map;
    }

    protected Map<String, MyJTextArea> getEditedRecord() {
        Map<String, MyJTextArea> map = new TreeMap<>();
        for (Object obj : parent.sv_currentQuery.sqlColNames) {
            String key = (String) obj;
            map.put(key, editControls.get(key));
        }
        return map;
    }

    protected void clearChanged() {
        if (editControls != null) {
            for (String key : editControls.keySet()) {
                editControls.get(key).clearChanged();
            }
        }
        isChanged = false;
        getAllowEditState();
    }

    protected void reset() {
        populateEditPane();
    }

    protected void vtab(MyJTextArea ta, int v) {
        int pv = editList.indexOf(ta);
        int sz = editList.size();
        pv = Math.abs(pv + v + sz) % sz;
        if (pv >= 0 && pv < editList.size()) {
            MyJTextArea targ = editList.get(pv);
            targ.grabFocus();
            Point p = targ.getLocation();
            JScrollBar vs = parent.editScrollPane.getVerticalScrollBar();
            vs.setValue(p.y - vs.getHeight() / 2);
            //targ.scrollRectToVisible(ta.getVisibleRect());
        }
    }

    protected void populateEditPane() {
        parent.editControlPanel.removeAll();
        parent.editScrollPane.setHorizontalScrollBarPolicy(
                parent.sv_wordWrapCheckBox.isSelected()
                ? JScrollPane.HORIZONTAL_SCROLLBAR_NEVER
                : JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED);
        if (parent.sv_currentQuery != null
                && parent.sv_currentQuery.getRecordCount() > 0
                && realRow >= 0) {
            int scroll = parent.editScrollPane.getVerticalScrollBar().getValue();
            editControls = new HashMap<>();
            editList = new ArrayList<>();
            Insets insets = new Insets(-1, 2, -1, 2);
            int y = 0;
            for (Object obj : parent.sv_currentQuery.sqlColNames) {
                String key = (String) obj;
                JLabel lbl = new JLabel(String.format("<html><b>%s:", key));
                lbl.setFont(parent.baseFont);
                parent.editControlPanel.add(lbl, parent.createConstraints(0, y, 1, 0f, insets));
                MyJTextArea tf = new MyJTextArea(parent, this, key);
                tf.setLineWrap(parent.sv_wordWrapCheckBox.isSelected());
                tf.setWrapStyleWord(true);
                Object val = parent.sv_currentQuery.getRecordField(realRow, key);
                if (val == null) {
                    val = "NULL";
                }
                tf.setText(val.toString());
                tf.setToolTipText(String.format("<html>Compose or edit field \"%s\"<br/>Shift+Enter accepts,<br/>Shift+arrow key moves vertically", key));
                editControls.put(key, tf);
                editList.add(tf);
                DefaultCaret caret = (DefaultCaret) tf.getCaret();
                caret.setUpdatePolicy(DefaultCaret.ALWAYS_UPDATE);
                parent.editControlPanel.add(tf, parent.createConstraints(1, y, 1, 1f, insets));
                y += 1;
            }
            // must delay the rescroll
            scrollPaintLater(scroll);
        } else {
            String option = (this.allowEdits)?"":" select \"Allow Edits\" at the lower right, then";
            String s = String.format("To edit a record,%s click a row above.",option);
            int lm = (parent.getWidth() - s.length() * parent.charWidth) / 2;
            Insets insets = new Insets(16, lm, 0, 0);
            chooseLabel.setText("<html><h3>" + s);
            parent.editControlPanel.add(chooseLabel, parent.createConstraints(0, 0, 1, 0.5f, insets));
            scrollPaintLater(0);
        }
    }

    protected void scrollPaintLater(final int scroll) {
        SwingUtilities.invokeLater(new Runnable() {
            @Override
            public void run() {
                parent.editScrollPane.getVerticalScrollBar().setValue(scroll);
                parent.repaint();
            }
        });
    }

    protected void resizeEvent() {

        if (editList != null) {
            //resizeEvent2(3);
            //parent.editScrollPane.getViewport().revalidate();
            //parent.editControlPanel.setPreferredSize(new Dimension(10,10));
        }
    }

    protected void resizeEvent2(int n) {
        if (n > 0) {
            parent.editControlPanel.validate();
            int w = parent.editScrollPane.getViewport().getSize().width;
            int h = parent.editControlPanel.getSize().height;
            Dimension d = new Dimension(w, h);
            p("resize: " + d + "," + n);
            parent.editControlPanel.setPreferredSize(d);
            //for (MyJTextArea tf : editList) {
            //    tf.revalidate();
            //}
            resizeEvent2(n - 1);
        }
    }

    protected void addPrimaryKey() {
        if (parent.sv_currentQuery != null) {
            String prompt = String.format("Okay to add primary key to table \"%s\"?\n(nearly always a good idea)", parent.sv_currentQuery.table);
            if (parent.askUser(prompt)) {
                String pkname = "pk";
                int i = 0;
                // deal with possible duplicate field name
                while (parent.sv_currentQuery.sqlColNames.contains(pkname)) {
                    i += 1;
                    pkname = String.format("pk%d", i);
                }
                String sql = String.format("alter table `%s` add column %s integer not null auto_increment primary key", parent.sv_currentQuery.table, pkname);
                int changed = parent.sv_currentQuery.execSQLiteUpdate(sql);
                updateChangedLabel(changed);
                commitInProgress = true;
                parent.formatRunQuery();
                commitInProgress = false;
                populateEditPane();
                clearChanged();
                refocus();
            }
        }
    }

    public void p(String s) {
        System.out.println(s);
    }
}
