// ***************************************************************************
// *   Copyright (C) 2017 by Paul Lutus                                      *
// *   lutusp@arachnoid.com                                                  *
// *                                                                         *
// *   This program is free software; you can redistribute it and/or modify  *
// *   it under the terms of the GNU General Public License as published by  *
// *   the Free Software Foundation; either version 2 of the License, or     *
// *   (at your option) any later version.                                   *
// *                                                                         *
// *   This program is distributed in the hope that it will be useful,       *
// *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
// *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
// *   GNU General Public License for more details.                          *
// *                                                                         *
// *   You should have received a copy of the GNU General Public License     *
// *   along with this program; if not, write to the                         *
// *   Free Software Foundation, Inc.,                                       *
// *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
// ***************************************************************************
package jsqliteclient;

import java.awt.Component;
import java.awt.Font;
import java.util.HashMap;
import java.util.Map;
import java.util.regex.Pattern;
import javax.swing.JTable;
import javax.swing.JTextArea;
import javax.swing.border.Border;
import javax.swing.border.EmptyBorder;
import javax.swing.border.LineBorder;
import javax.swing.table.DefaultTableCellRenderer;

/**
 *
 * @author lutusp
 */
final public class MyTableCellRenderer extends DefaultTableCellRenderer {

    JSQLiteClient parent;
    static final long serialVersionUID = 32148;
    String rightJustify;
    int rjLen;
    boolean title;
    boolean errorDisp = false;
    Border cb = null;
    Font font;
    boolean needEllipse = false;
    EmptyBorder emptyBorder;
    LineBorder lineBorder;
    Map<Integer, Integer> decPlaces;

    public MyTableCellRenderer(JSQLiteClient p, boolean title, boolean errorDisp, Map<Integer, Integer> dp) {
        super();
        parent = p;
        decPlaces = dp;
        this.title = title;
        this.errorDisp = errorDisp;
        // this is used for a complicated right-justify method
        rightJustify = "                           ";
        rjLen = rightJustify.length();
        if (title) {
            font = new Font(Font.MONOSPACED, Font.BOLD, 16);
        } else {
            cb = new EmptyBorder(
                    parent.tablePaddingConstant + 2,
                    parent.tablePaddingConstant + 2,
                    parent.tablePaddingConstant,
                    parent.tablePaddingConstant);
            //cb = new LineBorder(Color.red,1);
            lineBorder = new LineBorder(parent.tableBorderColor);

            //cb = new CompoundBorder(lineBorder, emptyBorder);
            //cb = emptyBorder;
            //cb = new LineBorder(Color.black,0);
            font = parent.baseFont;
        }
    }

    private String ellipsizeString(String sv, int pixw) {
        int stringLen = sv.length();
        int sw = stringLen * parent.charWidth;
        boolean needFix = (sw >= pixw);
        if (needFix) {
            int ni = (pixw / parent.charWidth) - 3;
            ni = Math.max(0, ni);
            sv = sv.substring(0, ni) + "...";
            needEllipse |= needFix;
        }
        return sv;
    }

    private String ellipsizeTest(String sv, int pixw) {
        String[] array = sv.split("\n");
        if (array.length < 2) {
            return ellipsizeString(sv, pixw);
        }
        StringBuilder sb = new StringBuilder();
        for (String sa : array) {
            sb.append(ellipsizeString(sa, pixw));
            sb.append("\n");
        }
        return sb.toString();
    }

    @Override
    public Component getTableCellRendererComponent(
            JTable table, Object value,
            boolean isSelected, boolean hasFocus,
            int row, int column) {
        needEllipse = false;
        String sv = "";
        if (value == null) {
            sv = "NULL";
        } else {
            String cls = value.getClass().getName();
            // produce formatted number display
            // that SQLite can't create on its own
            if (decPlaces != null
                    && decPlaces.containsKey(column)
                    && cls.matches("(?i).*(double|float|integer|long).*")) {
                double dval = Double.parseDouble(value.toString());
                int dp = decPlaces.get(column);
                String form = String.format("%%.%df", dp);
                sv = String.format(form, dval);
            } else {
                if (cls.matches("(?i).*(integer|long).*")) {
                    sv = String.format("%d", value);
                } else if (cls.matches("(?i).*(float|double).*")) {
                    sv = String.format("%.8g", value);
                } else {
                    sv = value.toString();
                }
            }
        }
        sv = sv.replaceAll("\t", "        ");
        int stringLen = sv.length();
        int colw = table.getColumnModel().getColumn(column).getWidth();
        int pixw = colw - parent.tablePaddingConstant;
        sv = ellipsizeTest(sv, pixw);
        if (!needEllipse && !(value instanceof java.lang.String)) {
            StringBuilder sb = new StringBuilder();
            int chw = (colw / parent.charWidth) - 1;
            int delta = chw - stringLen;
            // pad string with spaces at the left
            if (delta > 0) {
                if (delta < rjLen) {
                    sb.append(rightJustify.substring(0, delta));
                } else {
                    sb.append(rightJustify);
                    delta -= rjLen;
                    while (delta > 0) {
                        sb.append(" ");
                        delta -= 1;
                    }
                }
            }
            sv = new String(sb.append(sv));
        }
        JTextArea ta = new JTextArea();
        ta.setEditable(false);
        ta.setLineWrap(false);
        //ta.setMargin(new Insets(0,0,0,0));

        if (font != null) {
            ta.setFont(font);
        }
        if (!title) {
            // highlight error text
            if (errorDisp && column == 3 && !sv.equals("(none)")) {
                ta.setForeground(parent.changedColor);
            }
            if (isSelected) {
                ta.setBackground(parent.selectedColor);
            } else {
                if (table instanceof MyJTable) {
                    if (row == ((MyJTable) table).hoverRow) {
                        ta.setBackground(parent.hoverColor);
                    } else {
                        ta.setBackground(parent.oddeven[row % 2]);
                    }
                }
            }
        } else {
            int lm = (table.getWidth() - sv.length() * parent.charWidth) / 2;
            cb = new EmptyBorder(16, lm, 0, 0);
        }
        ta.setBorder(cb);
        ta.setText(sv);
        return ta;
    }

    private void p(String s) {
        System.out.println(s);
    }
}
