/***************************************************************************
 *   Copyright (C) 2011 by Paul Lutus                                      *
 *   lutusp@arachnoid.com                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

package jsiggen;

import javax.sound.sampled.*;

/**
 *
 * @author lutusp
 */
public class SignalGenerator extends Thread {

    JSigGenPanel parent;
    boolean running = false;
    int word_size = 2;
    int mask = 0xff;
    int sbufsz = 2048;
    int bbufsz = sbufsz * 2;
    byte[] bbuffer;
    short[] sbuffer;
    int data_rate;
    double sample_interval;
    double scale_mult = Math.pow(2, 15);
    SourceDataLine audioOutputLine = null;
    AudioFormat audioFormat = null;
    SigFunction fsine = new SigFunction() {

        public double f(double f, double t) {
            return Math.sin(2.0 * Math.PI * f * t);
        }
    };
    SigFunction ftriangle = new SigFunction() {

        public double f(double f, double t) {
            double q = 4.0 * (t * f % 1.0);
            q = (q > 1.0) ? 2 - q : q;
            return (q < -1) ? -2 - q : q;
        }
    };
    SigFunction fsquare = new SigFunction() {

        public double f(double f, double t) {
            if (f == 0) {
                return 0;
            }
            double q = 0.5 - t * f % 1.0;
            return (q > 0) ? 1 : -1;
        }
    };
    SigFunction fsawtooth = new SigFunction() {

        public double f(double f, double t) {
            return 2.0 * (((t * f) + 0.5) % 1.0) - 1.0;
        }
    };
    SigFunction[] functs = new SigFunction[]{fsine, ftriangle, fsquare, fsawtooth};

    public SignalGenerator(JSigGenPanel p) {
        parent = p;
        bbuffer = new byte[bbufsz];
        sbuffer = new short[sbufsz];
    }

    AudioFormat createAudioFormat() {
        int sampleSizeInBits = 8 * word_size;
        int channels = 2;
        boolean signed = true;
        boolean bigEndian = true;
        float rate = data_rate;
        return new AudioFormat(
                rate,
                sampleSizeInBits,
                channels,
                signed,
                bigEndian);
    }

    @Override
    public void run() {
        data_rate = parent.data_rate.get_ivalue();
        try {
            audioFormat = createAudioFormat();
            DataLine.Info info = new DataLine.Info(SourceDataLine.class, audioFormat);
            Mixer m = parent.source_mixers.get(parent.output_channel.get_ivalue() - 1);
            audioOutputLine = (SourceDataLine) m.getLine(info);
            audioOutputLine.open(audioFormat, bbufsz * 4);
            audioOutputLine.start();
        } catch (Exception e) {
            System.out.println(e);
            return;
        }
        sample_interval = 1.0 / data_rate;
        running = true;
        double time_sec = 0;
        double osig, sig, noise, mod, fm_integral = 0;
        short ssig;
        SigFunction sf, mf;
        int n = 0;
        while (running) {
            sf = functs[parent.signal_waveform.get_index()];
            if (!parent.modulation_enabled.get_value()) {
                sig = sf.f(parent.frequency.get_dvalue(), time_sec);
            } else {
                mf = functs[parent.modulation_waveform.get_index()];
                mod = mf.f(parent.modulation_frequency.get_dvalue(), time_sec) * parent.modulation_level.get_pct_dvalue();
                if (parent.am_fm_mode.get_index() == 0) {
                    sig = 0.5 * sf.f(parent.frequency.get_dvalue(), time_sec) * (1.0 + mod);
                } else {
                    fm_integral += mod;
                    sig = sf.f(parent.frequency.get_dvalue(), (time_sec + fm_integral * sample_interval));
                }
            }
            osig = 0;
            if (parent.noise_enabled.get_value()) {
                noise = (Math.random() - 0.5) * 2;
                osig += noise * parent.noise_level.get_pct_dvalue();
            }
            if (parent.signal_enabled.get_value()) {
                osig += sig * parent.signal_level.get_pct_dvalue();
            }
            ssig = (short) (osig * scale_mult);
            sbuffer[n++] = (parent.left_channel_enabled.get_value()) ? ssig : 0;
            sbuffer[n++] = (parent.right_channel_enabled.get_value()) ? ssig : 0;
            if (n >= sbufsz) {
                n = 0;
                int j = 0;
                for (int i = 0; i < sbuffer.length; i++) {
                    ssig = sbuffer[i];
                    bbuffer[j++] = (byte) (ssig >> 8 & mask);
                    bbuffer[j++] = (byte) (ssig & mask);
                }
                try {
                    audioOutputLine.write(bbuffer, 0, bbufsz);
                } catch (Exception e) {
                    System.out.println(e);
                }
            }
            time_sec += sample_interval;
        }
        if (audioOutputLine != null) {
            audioOutputLine.flush();
            audioOutputLine.drain();
            audioOutputLine.close();
            audioOutputLine = null;
        }
    }
};

interface SigFunction {

    public double f(double f, double t);
};
