/***************************************************************************
 *   Copyright (C) 2004 by Paul Lutus                                      *
 *   lutusp@arachnoid.com                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "oscillatoraudiodriver.h"
#include "signalgen.h"
#include "oscillatorfunctions.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <sys/soundcard.h>


#include <qstring.h>
#include <qapplication.h>

#include <cmath>


OscillatorAudioDriver::OscillatorAudioDriver(SignalGen *p) : QThread() {
    parent = p;
    validAudio = false;
    audio = 0;
}

OscillatorAudioDriver::~OscillatorAudioDriver() {
    if(audio != 0) {
        close(audio);
    }
}

void OscillatorAudioDriver::closeDevice() {
    if(audio != 0) {
        close(audio);
    }
}

void OscillatorAudioDriver::openDevice() {
    closeDevice();
    // test open using non-blocking option
    audio = open(parent->devicename.c_str(), O_WRONLY|O_NONBLOCK,0);
    if(audio != -1) {
        close(audio);
        audio = open(parent->devicename.c_str(), O_WRONLY,0);
    }
    if(audio != -1) {
        setOptions(audio);
        validAudio = true;
    }
}

void OscillatorAudioDriver::run() {
    oscillator();
}

// an envelope function to prevent clicks in audio output
// as it starts and stops

double OscillatorAudioDriver::envelope(double a,double b,double t,double tc) {
    return ((b - t) * (-a + t))/((b - t + tc) * (-a + t + tc));
}

void OscillatorAudioDriver::oscillator() {
    if(audio <= 0) {
        openDevice();
    }
    if(audio != -1) {

        double t = 0.0;
        double rampTC = .003;
        double upRampv = .1;
        double downRampv = 0.0;
        double step = 1.0 / (double) parent->sampleRate;
        bool downramp = true;
        while(parent->isRunning || downramp || t-downRampv < 0) {
	//step = 1.0 / (double) parent->sampleRate;
            double r = 1.0;
            if(t < upRampv) {
                r = envelope(0,1000,t,rampTC);
            } else if(!parent->isRunning && downramp) {
                downRampv = t+upRampv;
                downramp = false;
            } else if(downRampv > 0.0) {
                r = envelope(0,downRampv,t,rampTC);
            }
            // must change waveforms on the fly in response to user input
            double z = OscillatorFunctions::getFunct(parent->waveform)(parent->freqn,t);
            int v = (int) (r * z * 32767.0 * parent->level);
            write(audio,&v,2);
            t += step;
        }
    }
}



void OscillatorAudioDriver::setOptions(int audio) {
    int format = AFMT_S16_LE;
    ioctl(audio, SNDCTL_DSP_SETFMT, &format);
    int v = parent->sampleRate;
    ioctl(audio, SNDCTL_DSP_SPEED, &v);
    if(v != parent->sampleRate) {
        parent->reviseSampleRate(v);
    }
    int val = (int) (( 2 << 16 ) + log2(512));
    ioctl(audio, SNDCTL_DSP_SETFRAGMENT, &val);

}
