/***************************************************************************
 *   Copyright (C) 2004 by Paul Lutus                                      *
 *   lutusp@arachnoid.com                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/


#include "slideprojector.h"
#include <cstdio>
#include <sys/stat.h>
#include <sys/types.h>
#include <qapplication.h>
#include <qnamespace.h>
#include <qevent.h>
#include <qwidget.h>
#include <qwidgetstack.h>
#include <qlayout.h>
#include <qtabwidget.h>
#include <kwin.h>
#include <qvariant.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qtooltip.h>
#include <qwhatsthis.h>
#include <qimage.h>
#include <qpixmap.h>
#include <qevent.h>
#include <qcombobox.h>
#include <qmessagebox.h>
#include <qcheckbox.h>
#include <qpushbutton.h>
#include <qradiobutton.h>
#include <qbuttongroup.h>
#include "PieGraph.xpm"
#include "iconeditor.h"
#include "myiconview.h"

const string SlideProjector::programName = "SlideProjector";
const string SlideProjector::programVersion = "5.4";
const string SlideProjector::thumbDirName = "sp_thumbnails";
const string SlideProjector::thumbFilePrefix = "thumb_";
const string SlideProjector::dummyShowName = "-- none --";
const string SlideProjector::extensionsScriptName = "extensions.sh";

// this invocation order avoids a warning message
SlideProjector::SlideProjector(QWidget* parent, const char* name, WFlags fl) : DCOPObject("SlideProjectorIface"),SlideProjectorMainFrame(parent,name,fl)
{

  autoShowTimer = NULL;
  zoom = false;
  borderFrame = false;
  active = false;
  editPreviewImage = NULL;
  // default, user-editable values
  editIconSize = 120;
  editSlideDuration = 10;
  autoShowSlideDuration = 20;
  rightMouseZooms = true;
  rightMouseButtonGroup->setRadioButtonExclusive(true);

  setCaption(programName + " " + programVersion);
  imghandler = new ImageFileHandler();
  usrPath = getenv("HOME");
  // default value
  lastImageDirectory = usrPath;
  progPath = usrPath + "/." + programName;
  //printf("user home path: %s\n",usrPath.c_str());
  //fflush(stdout);
  initHandler = new InitFileHandler(usrPath,programName);
  iconEditor = new IconEditor(tabWidget,this,"IconEditor");
  //QGridLayout *ieLayout = new QGridLayout( iconEditor, 1, 1, 0, 0, "ieLayout");
  setupList();

  toggleImage = -1;
  setIcon(QPixmap(PieGraph));
  screenMode = 0;

  layout7->remove
  (whatsThisDummyButton);
  delete whatsThisDummyButton;

  //whatsThisButton = new QPushButton( groupBox1, "whatsThisButton" );

  QPushButton *wt = (QPushButton *) QWhatsThis::whatsThisButton(groupBox1);

  layout7->addWidget(wt, 0, 5 );

  myql_norm = new MyQLabel(this,tab_2, "Normal" );
  tabLayout_2->addWidget(myql_norm, 0, 0);
  helpwidget = new HelpWidget(this);


  tabWidget->insertTab(helpwidget,"Help [F1]",0);


  tabWidget->insertTab(iconEditor,"Editing [F7]");

  tabWidget->setFocusPolicy(QWidget::NoFocus);


  ShowComboBox->setFocusPolicy(QWidget::NoFocus);

  dupWidget = new QWidget( this, "dupWidget" );
  dupLayout = new QGridLayout( dupWidget, 1, 1, 11, 6, "tabLayout");

  myql_full = new MyQLabel(this,dupWidget, "FullScreen" );
  //myql_full->setCursor(QCursor(Qt::BlankCursor));
  dupLayout->addWidget(myql_full, 0, 0);
  dupLayout->setMargin(0);
  dupWidget->setShown(false);
  tabWidget->setCurrentPage(1);
  readConfigFile();
  mouseZoomRadioButton->setChecked(rightMouseZooms);
  keySlideRadioButton->setChecked(!rightMouseZooms);
  ostringstream ss;
  ss << editSlideDuration;
  SlideDuration->setText(ss.str());
  ss.str("");
  ss.clear();
  ss << editIconSize;
  IconSize->setText(ss.str());
  ss.str("");
  ss.clear();
  ss << autoShowSlideDuration;
  autoShowTime->setText(ss.str());
  repeatCheckBox->setChecked(autoShowRepeat);
  borderframeCheckBox->setChecked(borderFrame);
  active = true;
  int saveTab = tabWidget->currentPageIndex();
  showChosen();
  tabWidget->setCurrentPage(saveTab);
  iconEditor->saveEditButton->setEnabled(false);
  if(imghandler->listSize() == 0)
  {
    tabWidget->setCurrentPage(1); // configuration
  }
}

SlideProjector::~SlideProjector()
{
  delete(myql_norm);
  delete(dupWidget);
  delete(myql_full);
  delete(dupLayout);
  delete(imghandler);
  delete(helpwidget);
  delete(iconEditor);
  delete(initHandler);
  delete editPreviewImage;
  if(autoShowTimer != NULL)
  {
    autoShowTimer->stop();
    delete autoShowTimer;
  }
}

/*$SPECIALIZATION$*/

void SlideProjector::setupList()
{
  vector <string> list = imghandler->getProgramList(progPath);
  string oldShow = ShowComboBox->currentText();
  string oldEdit = iconEditor->loadEditComboBox->currentText();
  bool oldActive = active;
  active = false;
  iconEditor->active = false;
  ShowComboBox->clear();
  iconEditor->loadEditComboBox->clear();
  iconEditor->loadEditComboBox->insertItem(dummyShowName);
  for(int i = 0;i < (int) list.size();i++)
  {
    ShowComboBox->insertItem(list[i]);
    iconEditor->loadEditComboBox->insertItem(list[i]);
  }
  if(oldShow.length() > 0)
  {
    ShowComboBox->setCurrentText(oldShow);
  }
  if(oldEdit.length() > 0)
  {
    iconEditor->loadEditComboBox->setCurrentText(oldEdit);
  }
  iconEditor->active = true;
  active = oldActive;
}

QString SlideProjector::getShowList()
{
  vector <string> list = imghandler->getProgramList(progPath);
  QString s;
  for(unsigned int i = 0;i < list.size();i++)
  {
    s += list[i] + " ";
  }
  return s;
}

void SlideProjector::showChosen()
{
  if(active)
  {
    string name = ShowComboBox->currentText();
    selectShow(name);
  }
}

void SlideProjector::selectShow(QString name)
{
  selectShow((string) (const char*) name);
}

void SlideProjector::selectShow(string name)
{
  if(name.length() > 0)
  {
    imghandler->newImageSet(progPath,name);
    ShowComboBox->setCurrentText(name);
    ostringstream ss;
    ss << imghandler->listSize() << " slides";
    SlidesLabel->setText(ss.str());
    setFirstImage();
  }
}

void SlideProjector::testRefreshAfterSave(string name)
{
  if(((string) (const char*)ShowComboBox->currentText()) == name)
  {
    showChosen();
  }
}

void SlideProjector::setDisplayMode(int mode)
{
  //printf("display mode %d\n",mode);
  //fflush(stdout);
  if(screenMode == 0 || screenMode == 1)
  {
    normalSize = size();
    normalPos = pos();
  }
  screenMode = mode;
  tabWidget->showPage((screenMode == 0)?tab:tab_2);
  //printf("mode: %d\n",screenMode);
  //fflush(stdout);
  if(screenMode != 3)
  {
    SlideProjectorMainFrameLayout->remove
    (dupWidget);
    SlideProjectorMainFrameLayout->addWidget( tabWidget, 0, 0 );
    tabWidget->setShown(true);
    dupWidget->setShown(false);
  }
  else
  {
    SlideProjectorMainFrameLayout->remove
    (tabWidget);
    SlideProjectorMainFrameLayout->addWidget( dupWidget, 0, 0 );
    dupWidget->setShown(true);
    tabWidget->setShown(false);
  }
  switch(screenMode)
  {
  case 0:
  case 1:
    showNormal();
    resize(normalSize);
    move(normalPos);
    break;
  case 2:
    showNormal();
    showMaximized();
    break;
  case 3:
    showFullScreen();
    break;
  }

}

void SlideProjector::keyPressEvent(QKeyEvent *e)
{
  bool accept = true;
  //printf("key event!\n");
  //fflush(stdout);
  switch (e->key())
  {
  case Qt::Key_F1:
    stopAutoSlideShow(false);
    launchHelp();
    break;
  case Qt::Key_F2:
    stopAutoSlideShow();
    setDisplayMode(0);
    break;
  case Qt::Key_F3:
    setDisplayMode(1);
    break;
  case Qt::Key_F4:
    setDisplayMode(2);
    break;
  case Qt::Key_F5:
    setDisplayMode(3);
    break;
  case Qt::Key_F6:
    calibrateAction();
    break;
  case Qt::Key_F7:
    stopAutoSlideShow(false);
    showEditor();
    break;
  case Qt::Key_F8:
    startAutoShow();
    break;
  case Qt::Key_F9:
    if(autoShowTimer == NULL)
    {
      toggleZoom();
    }
    break;
  case Qt::Key_Left:
  case Qt::Key_Prior:
    setDeltaImage(-1);
    break;
  case Qt::Key_Right:
  case Qt::Key_Next:
  case Qt::Key_Space:
    setDeltaImage(1);
    break;
  case Qt::Key_Home:
    setFirstImage();
    break;
  case Qt::Key_End:
    setLastImage();
    break;
  case Qt::Key_Escape:
    rightMouseAction();
    break;
  default:
    accept = false;

  }
  if(accept)
  {
    e->accept();
  }
}

void SlideProjector::rightMouseAction()
{
  if(autoShowTimer != NULL)
  {
    stopAutoSlideShow();
  }
  else if(rightMouseZooms)
  {
    toggleZoom();
  }
  else
  {
    toggleKeyImage();
  }
}

void SlideProjector::toggleZoom()
{
  // don't zoom if the right-click is meant to exit
  if(myql_full->fromEditor || myql_norm->fromEditor)
    return;
  zoom = !zoom;
  myql_full->refreshImage(screenMode == 3);
  myql_norm->refreshImage(screenMode != 3);
}

void SlideProjector::toggleKeyImage()
{
  setImageDisplay();
  if(toggleImage == -1)
  {
    toggleImage = imghandler->getCurrentImage();
    setFirstImage();
  }
  else
  {
    setImage(toggleImage);
    toggleImage = -1;
  }
}

void SlideProjector::zoomRadioButton(int)
{
  rightMouseZooms = mouseZoomRadioButton->isChecked();
  //printf("radio button: %d\n",rightMouseZooms);
  //fflush(stdout);
}

// the slide projector remote control
// produces a right mouse press for the key
// marked "Esc"

void SlideProjector::contextMenuEvent( QContextMenuEvent * e)
{
  rightMouseAction();
  e->accept();
  e->consume();
}

void SlideProjector::setImageDisplay()
{
  int index = tabWidget->currentPageIndex();
  int mode = (index < 2)?0:screenMode;
  if(mode < 1)
  {
    setDisplayMode(1);
  }
}

void SlideProjector::launchHelp()
{
  if(screenMode == 3)
  {
    setDisplayMode(2);
  }
  tabWidget->showPage(helpwidget);
  helpwidget->SearchString->setFocus();
}

void SlideProjector::showEditor()
{
  if(screenMode == 3)
  {
    setDisplayMode(2);
  }
  tabWidget->showPage(iconEditor);
}

void SlideProjector::calibrateAction()
{
  setImageDisplay();
  myql_full->toggleAlignmentFrame(screenMode == 3);
  myql_norm->toggleAlignmentFrame(screenMode != 3);
}

int SlideProjector::getImageCount()
{
  return imghandler->listSize();
}

int SlideProjector::getCurrentImage()
{
  return imghandler->getCurrentImage();
}

void SlideProjector::setDeltaImage(int i)
{
  setImageDisplay();
  QImage *im = imghandler->retrieveDeltaImage(i);
  updateImages(im,false);
}

void SlideProjector::setImage(int i)
{
  setImageDisplay();
  QImage *im = imghandler->retrieveImage(i);
  updateImages(im,false);
}

void SlideProjector::setLastImage()
{
  setImageDisplay();
  QImage *im = imghandler->retrieveLastImage();
  updateImages(im,false);
}

void SlideProjector::setFirstImage()
{
  setImageDisplay();
  QImage *im = imghandler->retrieveFirstImage();
  updateImages(im,false);
}

void SlideProjector::updateImages(QImage *im,bool fromEditor)
{
  delete editPreviewImage;
  editPreviewImage = NULL;
  updateImages2(im,fromEditor);
}

void SlideProjector::updateImages2(QImage *im, bool fromEditor)
{
  myql_full->setImage(im,screenMode == 3,fromEditor);
  myql_norm->setImage(im,screenMode != 3,fromEditor);
  manageExtensions(im,fromEditor);
}

void SlideProjector::manageExtensions(QImage *im,bool fromEditor)
{
  string p = progPath + "/" + extensionsScriptName;
  // printf("screenMode: %d. looking for %s\n",screenMode,p.c_str());
  if(QFile::exists(p))
  {
    if(!fromEditor && screenMode >= 1)
    {
      string path = imghandler->retrieveCurrentImagePath();
      system((p + " " + path).c_str());
      // printf("executed %s\n",p.c_str());
    }
  }
  fflush(stdout);
}

void SlideProjector::showEditPreviewImage(string path)
{
  delete editPreviewImage;
  editPreviewImage = new QImage(path);
  updateImages2(editPreviewImage,true);
  setDisplayMode(1);
}

void SlideProjector::closeEvent ( QCloseEvent * e )
{
  if(iconEditor != NULL && iconEditor->checkChanged())
  {
    writeConfigFile();
    e->accept();
  }
}

void SlideProjector::slideDurationChanged()
{
  if(active)
  {
    istringstream ss(SlideDuration->text());
    ss >> editSlideDuration;
  }
}

void SlideProjector::newAutoSlideDuration()
{
  if(active)
  {
    istringstream ss(autoShowTime->text());
    ss >> autoShowSlideDuration;
  }
}

void SlideProjector::newAutoShowRepeat()
{
  if(active)
  {
    autoShowRepeat = repeatCheckBox->isChecked();
  }
}

void SlideProjector::setAutoShowRepeat(bool r)
{
  repeatCheckBox->setChecked(r);
  newAutoShowRepeat();
}

void SlideProjector::setAutoShowSlideDuration(int d)
{
  ostringstream ss;
  ss << d;
  autoShowTime->setText(ss.str());
  newAutoSlideDuration();
}

void SlideProjector::startAutoShow()
{
  if(active)
  {
    if(autoShowTimer == NULL)
    {
      setDisplayMode(3);
      setFirstImage();
      autoShowTimer = new QTimer( this );
      connect( autoShowTimer, SIGNAL(timeout()),this, SLOT(autoShowNextSlide()) );
      autoShowTimer->start( autoShowSlideDuration * 1000, false );
    }
  }
}

// wrapper function for DCOP

void SlideProjector::startAutoSlideShow()
{
  startAutoShow();
}

void SlideProjector::autoShowNextSlide()
{
  //printf("%d of %d\n",getCurrentImage(),getImageCount());
  //fflush(stdout);
  if(getCurrentImage() >= getImageCount()-1)
  {
    if(autoShowRepeat)
    {
      setFirstImage();
    }
    else
    {
      stopAutoSlideShow();
    }
  }
  else
  {
    setDeltaImage(1);
  }
}

void SlideProjector::stopAutoSlideShow(bool setMode0)
{
  if(autoShowTimer != NULL)
  {
    autoShowTimer->stop();
    delete autoShowTimer;
    autoShowTimer = NULL;
    if(setMode0)
    {
      setDisplayMode(0);
    }
  }
}

void SlideProjector::stopAutoSlideShow()
{
  stopAutoSlideShow(true);
}

void SlideProjector::iconSizeChanged()
{
  if(active)
  {
    istringstream ss(IconSize->text());
    ss >> editIconSize;
    //printf("icon size: %d\n",editIconSize);
    //fflush(stdout);
    string message = (string)"After changing thumbnail size,\n"
                     + "for best editing appearance you may\n"
                     + "want to recreate your thumbnail directories.\n"
                     + "Do this by selecting \"Replace thumbnails\" from the context menu\n"
                     + "of your directory image views.";
    QMessageBox::information(this,programName  + ": Thumbnail Size Changed",
                             message.c_str(),QMessageBox::Ok);
  }
}

void SlideProjector::helpButton()
{
  launchHelp();
}
void SlideProjector::screen0Button()
{
  setDisplayMode(0);
}
void SlideProjector::screen1Button()
{
  setDisplayMode(1);
}
void SlideProjector::screen2Button()
{
  setDisplayMode(2);
}
void SlideProjector::screen3Button()
{
  setDisplayMode(3);
}
void SlideProjector::calibrateButton()
{
  calibrateAction();
}
void SlideProjector::EditButton()
{
  showEditor();
}
void SlideProjector::quitButton()
{
  close();
}
void SlideProjector::borderframeCheckBox_clicked()
{
  borderFrame = !borderFrame;
}


void SlideProjector::readConfigFile()
{

  initHandler->readConfigFile();

  // set some defaults
  int xv = x();
  int yv = y();
  initHandler->readValue("PosX",xv);
  initHandler->readValue("PosY",yv);
  move(xv,yv);

  xv = width();
  yv = height();
  initHandler->readValue("SizeX",xv);
  initHandler->readValue("SizeY",yv);
  resize(xv,yv);

  initHandler->readValue("CurrentShow",xv);
  int max = ShowComboBox->count();
  xv = (xv >= max)?max-1:xv;
  ShowComboBox->setCurrentItem(xv);

  xv = tabWidget->currentPageIndex();
  initHandler->readValue("CurrentTab",xv);
  tabWidget->setCurrentPage(xv);
  initHandler->readValue("EditSlideDuration",editSlideDuration);
  initHandler->readValue("AutoShowSlideDuration",autoShowSlideDuration);
  initHandler->readValue("AutoShowRepeat",autoShowRepeat);
  initHandler->readValue("Borderframe",borderFrame);
  initHandler->readValue("EditIconSize",editIconSize);
  initHandler->readValue("LastImageDirectory",lastImageDirectory);
  initHandler->readValue("RightMouseZooms",rightMouseZooms);
}

void SlideProjector::writeConfigFile()
{
  initHandler->writeValue("PosX",x());
  initHandler->writeValue("PosY",y());
  initHandler->writeValue("SizeX",width());
  initHandler->writeValue("SizeY",height());
  initHandler->writeValue("CurrentShow",ShowComboBox->currentItem());
  initHandler->writeValue("CurrentTab",tabWidget->currentPageIndex());
  initHandler->writeValue("EditSlideDuration",editSlideDuration);
  initHandler->writeValue("AutoShowSlideDuration",autoShowSlideDuration);
  initHandler->writeValue("AutoShowRepeat",autoShowRepeat);
  initHandler->writeValue("Borderframe",borderFrame);
  initHandler->writeValue("EditIconSize",editIconSize);
  initHandler->writeValue("LastImageDirectory",lastImageDirectory);
  initHandler->writeValue("RightMouseZooms",rightMouseZooms);
  initHandler->writeConfigFile();
}

#include "slideprojector.moc"

