/***************************************************************************
 *   Copyright (C) 2019 by Paul Lutus                                      *
 *   lutusp@arachnoid.com                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

package tankflow;

import java.awt.Color;
import java.awt.Font;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.RenderingHints;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;
import java.awt.image.BufferedImage;

import javax.swing.JPanel;

final public class GraphicPane extends JPanel implements MouseListener, MouseMotionListener {
	/**
	 * 
	 */
	private static final long serialVersionUID = -2734553019461908656L;
	TankFlow parent;
	double width, height;
	double dwidth, dheight;
	int dotsize = 5;
	int sd2 = dotsize / 2;
	double margin = 10;
	double wmargin, hmargin;
	double xl, xh, yl, yh;
	double xstart, xend;
	double ystart, yend;
	double MARGINDIV = 8;
	double gs = 10.0;
	double xm,ym;
	boolean mouseOver = false;
	boolean reverse;

	public GraphicPane(TankFlow p) {
		parent = p;
		addMouseListener(this);
		addMouseMotionListener(this);
	}

	@Override
	public void paintComponent(Graphics g) {
		super.paintComponent(g);
		if (parent != null && g != null && parent.terms != null && parent.terms.size() > 0 && parent.results != null
				&& parent.results.size() > 0) { 
			drawChart(g,getWidth(),getHeight());
		}
	}
	
	// size-independent drawing
	
	protected void drawChart(Graphics g,int w, int h) {
		width = w;
		height = h;
		Graphics2D g2 = (Graphics2D) g;
		g2.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
		wmargin = width / margin;
		hmargin = height / margin;
		dwidth = width - wmargin;
		dheight = height - hmargin;
		g2.setColor(Color.WHITE);
		g2.fillRect(0, 0, (int) width, (int) height);
		size_graphic();
		draw_grid(g2);
		draw_chart(g2);
		draw_text_labels(g2);
	}
	
	protected BufferedImage grabScreen(int w, int h) {
		BufferedImage bi = new BufferedImage(
	            w,h, BufferedImage.TYPE_INT_ARGB);
	        drawChart(bi.getGraphics(),w,h);
	        return bi;
	}

	protected void size_graphic() {
		xl = 1e9;
		xh = -1e9;
		yl = 1e9;
		yh = -1e9;
		// find the graphic boundaries
		// based on numeric values
		for (Pair p : parent.arg) {
			xl = Math.min(xl, p.x);
			xh = Math.max(xh, p.x);
			yl = Math.min(yl, p.y);
			yh = Math.max(yh, p.y);
		}
		// xm and ym are margins
		xm = (xh - xl) / MARGINDIV;
		ym = (yh - yl) / MARGINDIV;
		xstart = xl;
		xend = xh;
		ystart = yl;
		yend = yh;
		// create margins around
		// main graphic area
		xl -= xm;
		xh += xm;
		yl -= ym;
		yh += ym;

	}

	protected void draw_grid(Graphics2D g2) {
		int xp, yp;
		g2.setColor(Color.GREEN);
		for (int xi = 0; xi <= gs; xi++) {
			xp = (int) ntrp(xi, 0, gs, wmargin, dwidth);
			g2.drawLine(xp, (int) (height - hmargin), xp, (int) (height - dheight));
		}
		g2.setColor(Color.GREEN);
		for (int yi = 0; yi <= gs; yi++) {
			yp = (int) ntrp(yi, 0, gs, (int) (height - hmargin), (int) (height - dheight));
			g2.drawLine((int) wmargin, yp, (int) dwidth, yp);
		}
	}
	
	protected void draw_text_labels(Graphics2D g2) {
		int xp,yp;
		double y;
		String s;
		g2.setColor(Color.BLACK);
		// fs = font size
		int fs = (int)width/75;
		g2.setFont(new Font("Monospace", Font.PLAIN, fs)); 
		for (int xi = 0; xi <= gs; xi++) {
			xp = (int) ntrp(xi, 0, gs, wmargin, dwidth);
			y = (int) ntrp(xi, 0, gs, xl,xh);
			s = String.format("%.4g",y);
			int dx = xp - 4 * s.length();
			g2.drawString(s,dx,(int) (height - hmargin) + 16);
		}
		for (int yi = 0; yi <= gs; yi++) {
			yp = (int) ntrp(yi, 0, gs, (int) (height - hmargin), (int) (height - dheight));
			y = (int) ntrp(yi, 0, gs, yl,yh);
			s = String.format("%.4g",y);
			xp = (int) (wmargin -fs * 0.7 * s.length());
			g2.drawString(s,xp,yp);
		}
	}

	protected void draw_chart(Graphics2D g2) {
		// draw the regression line
		g2.setColor(Color.BLUE);
		int ox = -1, oy = -1;
		boolean starting = true;
		int line_segments = 500;
		for (int xi = 0; xi <= line_segments; xi++) {
			double x = ntrp(xi, 0, line_segments, xstart, xend);
			double y = MatrixFunctions.regress(x, parent.terms);
			int xp = (int) ntrp(x, xl, xh, wmargin, dwidth);
			int yp = (int) ntrp(y, yl, yh, hmargin, dheight);
			yp = (int) (height - yp);
			if (!starting) {
				g2.drawLine(ox, oy, xp, yp);
			}
			starting = false;
			ox = xp;
			oy = yp;
		}
		// now draw the data points
		g2.setColor(Color.RED);
		for (Pair p : parent.arg) {
			int xp = (int) ntrp(p.x, xl, xh, wmargin, dwidth) - sd2;
			int yp = (int) ntrp(p.y, yl, yh, hmargin, dheight) + sd2;
			g2.fillOval(xp, (int) (height - yp), dotsize, dotsize);
		}
	}

	protected double ntrp(double x, double xl, double xh, double yl, double yh) {
		return (x - xl) * (yh - yl) / (xh - xl) + yl;
	}

	protected void manageMouse(MouseEvent e) {
		if (mouseOver) {
			int px = e.getX();
			int py = e.getY();
			// parent.p("" + px + "," + py);
			double x = ntrp(px, wmargin, dwidth, xl, xh);
			double y = ntrp(py, hmargin, dheight, yh, yl);
			// parent.p("" + x + "," + y);
			String s = String.format("x=" + parent.format_number(x) + ",y=" + parent.format_number(y));
			setToolTipText(s);
		}
	}

	@Override
	public void mouseDragged(MouseEvent e) {
		// TODO Auto-generated method stub

	}

	@Override
	public void mouseMoved(MouseEvent e) {
		// TODO Auto-generated method stub
		manageMouse(e);

	}

	@Override
	public void mouseClicked(MouseEvent e) {
		// TODO Auto-generated method stub

	}

	@Override
	public void mousePressed(MouseEvent e) {
		// TODO Auto-generated method stub

	}

	@Override
	public void mouseReleased(MouseEvent e) {
		// TODO Auto-generated method stub

	}

	@Override
	public void mouseEntered(MouseEvent e) {
		// TODO Auto-generated method stub
		mouseOver = true;
	}

	@Override
	public void mouseExited(MouseEvent e) {
		// TODO Auto-generated method stub
		mouseOver = false;

	}

}
