/***************************************************************************
 *   Copyright (C) 2019 by Paul Lutus                                      *
 *   lutusp@arachnoid.com                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

package tankflow;

import java.util.ArrayList;

/**
 *
 * @author lutusp
 */
public final class MatrixFunctions {

	// don't allow this class to be instantiated
	private MatrixFunctions() {
	}

	// classic Gauss-Jordan matrix manipulation functions
	static private void divide(double[][] A, int i, int j, int m) {
		for (int q = j + 1; q < m; q++) {
			A[i][q] /= A[i][j];
		}
		A[i][j] = 1;
	}

	static private void eliminate(double[][] A, int i, int j, int n, int m) {
		for (int k = 0; k < n; k++) {
			if (k != i && A[k][j] != 0) {
				for (int q = j + 1; q < m; q++) {
					A[k][q] -= A[k][j] * A[i][q];
				}
				A[k][j] = 0;
			}
		}
	}

	static private ArrayList<Double> echelonize(double[][] mat) {
		int n = mat.length;
		int m = mat[0].length;
		int i = 0;
		int j = 0;
		int k;
		double[] swap;
		while (i < n && j < m) {
			// look for non-zero entries in col j at or below row i
			k = i;
			while (k < n && mat[k][j] == 0) {
				k++;
			}
			// if an entry is found at row k
			if (k < n) {
				// if k is not i, then swap row i with row k
				if (k != i) {
					swap = mat[i];
					mat[i] = mat[k];
					mat[k] = swap;
				}
				// if A[i][j] is != 1, divide row i by A[i][j]
				if (mat[i][j] != 1) {
					divide(mat, i, j, m);
				}
				// eliminate all other non-zero entries
				eliminate(mat, i, j, n, m);
				i++;
			}
			j++;
		}
		// recover result column
		ArrayList<Double> terms = new ArrayList<Double>();
		for (double[] mc : mat) {
			terms.add(mc[n]);
		}
		return terms;
	}

	// create regression coefficients
	// for provided data set
	static ArrayList<Double> compute_coefficients(ArrayList<Pair> data, int p) {
		p += 1;
		int n = data.size();
		int r, c;
		int rs = 2 * p - 1;
		//
		// by request: read each datum only once
		// not the most efficient processing method
		// but required if the data set is huge
		//
		// create square matrix with added RH column
		double[][] m = new double[p][p + 1];
		// create array of precalculated matrix data
		double[] mpc = new double[rs];
		mpc[0] = n;
		double x;
		for (Pair pr : data) {
			// process precalculation array
			x = pr.x;
			for (r = 1; r < rs; r++) {
				mpc[r] += x;
				x *= pr.x;
			}
			// process RH column cells
			m[0][p] += pr.y;
			x = pr.x;
			for (r = 1; r < p; r++) {
				m[r][p] += x * pr.y;
				x *= pr.x;
			}
		}
		// populate square matrix section
		for (r = 0; r < p; r++) {
			for (c = 0; c < p; c++) {
				m[r][c] = mpc[r + c];
			}
		}
		// reduce matrix
		return echelonize(m);
	}

	// correlation coefficient
	static double corr_coeff(ArrayList<Pair> data, ArrayList<Double> terms) {
		int n = data.size();
		double sx = 0, sx2 = 0, sy = 0, sy2 = 0, sxy = 0;
		double x, y;
		for (Pair pr : data) {
			x = regress(pr.x, terms);
			y = pr.y;
			sx += x;
			sy += y;
			sxy += x * y;
			sx2 += x * x;
			sy2 += y * y;
		}
		double div = Math.sqrt((sx2 - (sx * sx) / n) * (sy2 - (sy * sy) / n));
		if (div == 0) {
			return 0;
		}
		double q = (sxy - (sx * sy) / n) / div;
		return q * q;
	}

	// standard error
	static double std_error(ArrayList<Pair> data, ArrayList<Double> terms) {
		int n = data.size();
		if (n > 2) {
			double a = 0;
			for (Pair pr : data) {
				double q = regress(pr.x, terms) - pr.y;
				a += q * q;
			}
			return Math.sqrt(a / (n - 2));
		}
		return 0;
	}

	// produce a single y result for a given x
	static double regress(double x, ArrayList<Double> terms) {
		double a = 0;
		double t = 1;
		for (double term : terms) {
			a += term * t;
			t *= x;
		}
		return a;
	}
}
