/***************************************************************************
 *   Copyright (C) 2013 by Paul Lutus                                      *
 *   http://arachnoid.com/administration                                   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

package com.arachnoid.anchorsentinel;

import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.util.HashMap;

import android.Manifest;
import android.app.Activity;
import android.app.AlertDialog;
import android.app.Dialog;
import android.content.Context;
import android.content.DialogInterface;
import android.content.Intent;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.content.pm.PackageManager.NameNotFoundException;
import android.graphics.Color;
import android.media.AudioManager;
import android.os.Bundle;
import android.os.Handler;
import android.os.PowerManager;
import android.speech.tts.TextToSpeech;
import android.speech.tts.TextToSpeech.OnInitListener;
import android.text.Editable;
import android.text.TextWatcher;
import android.util.Log;
import android.view.Menu;
import android.view.MenuInflater;
import android.view.MenuItem;
import android.view.View;
import android.view.Window;
import android.view.WindowManager;
import android.webkit.WebSettings;
import android.webkit.WebView;
import android.widget.AdapterView;
import android.widget.ArrayAdapter;
import android.widget.Button;
import android.widget.CheckBox;
import android.widget.CompoundButton;
import android.widget.CompoundButton.OnCheckedChangeListener;
import android.widget.EditText;
import android.widget.Spinner;
import android.widget.TabHost;
import android.widget.TableRow;
import android.widget.TextView;

public class AnchorSentinelActivity extends Activity implements OnInitListener {
    private TextToSpeech mTts = null;
    int SPEECH_DATA_CHECK = 333;
    int pauseBetweenPromptsMSec = 5000;
    Handler handler;
    HashMap<String, String> extraHash;
    Thread speakThread = null;
    boolean speechReady = false;
    boolean speaking = false;
    boolean alarmTripped = false;
    TabHost tabHost;
    TextView latTV, lngTV, magDecTV, locationText;
    TextView speedKnotsTV, speedMPHTV, speedKPHTV, bearing_trueTV,
            bearing_magTV;
    EditText guardRadius;
    EditText anchorAlarmMessage, gpsAlarmMessage;
    // TextView appField;
    TextView guardMode, guardLatTV, guardLngTV, guardDistTV, guardBrgTrueTV,
            guardBrgMagTV;
    TextView guardArrivalTV, guardTtgTV, guardArrivalLabel, guardTtgLabel;
    TextView guardDtgLabel, guardDtgTV;
    AnchorSentinelApplication app;
    Button appButton;
    TextView gpsStatusText, guardStatusText;
    Spinner radiusUnitSpinner;
    Spinner intervalSpinner, gpsFailSpinner;
    CheckBox voiceAlarm, toneAlarm, showArrival;
    WebView helpWebView;
    boolean suppressRead = false;
    boolean visible = true;
    PowerManager.WakeLock wakeLock = null;
    private static final String[] INITIAL_PERMS = {
            Manifest.permission.ACCESS_FINE_LOCATION,
            Manifest.permission.WAKE_LOCK
    };
    private static final int INITIAL_REQUEST = 1337;

    /**
     * Called when the activity is first created.
     */
    @Override
    public void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        app = (AnchorSentinelApplication) getApplication();
        app.activity = this;
        setContentView(R.layout.anchorsentinellayout);
        handler = new Handler();
        setupTabs();
        mapControls();
        setupHelp();
        try {
            String appName = getString(R.string.app_name);
            PackageInfo pInfo;
            pInfo = getPackageManager().getPackageInfo(getPackageName(), 0);
            String appVersion = pInfo.versionName;
            appButton.setText("  " + appName);
            // appField.setText(appName + " " + appVersion);

        } catch (NameNotFoundException e) {
            e.printStackTrace();
        }
        writeControlValues();
        update();
        processAlarm();

        checkPermissions();
        app.startProcesses();

    }

    protected void checkPermissions() {
        if (!app.canAccessLocation() || !app.canWakeLock()) {
            requestPermissions(INITIAL_PERMS, INITIAL_REQUEST);
        }
    }

    @Override
    public void onRequestPermissionsResult(int requestCode, String[] permissions, int[] grantResults) {
        AlertDialog ad = new AlertDialog.Builder(this).create();
        ad.setTitle("Restart Application");
        ad.setIcon(R.drawable.app_icon);
        ad.setMessage("Please press Quit to exit,then restart this application");
        ad.setButton(Dialog.BUTTON_POSITIVE, "Quit", new DialogInterface.OnClickListener() {
            public void onClick(DialogInterface dialog, int which) {
                finish();
                System.exit(0);
            }
        });
        ad.show();
    }

    protected void mapControls() {

        appButton = (Button) findViewById(R.id.app_button);
        latTV = setupTextView(R.id.lattv);
        lngTV = setupTextView(R.id.lngtv);
        magDecTV = setupTextView(R.id.magdectv);
        speedKnotsTV = setupTextView(R.id.speed_knotstv);
        speedMPHTV = setupTextView(R.id.speed_mphtv);
        speedKPHTV = setupTextView(R.id.speed_kphtv);
        bearing_trueTV = setupTextView(R.id.bearing_truetv);
        bearing_magTV = setupTextView(R.id.bearing_magtv);
        guardRadius = setupEditText(R.id.guard_radius);
        anchorAlarmMessage = setupEditText(R.id.anchor_alarm_message);
        gpsAlarmMessage = setupEditText(R.id.gps_alarm_message);
        guardMode = setupTextView(R.id.guard_mode);
        guardLatTV = setupTextView(R.id.guard_lat);
        guardLngTV = setupTextView(R.id.guard_lng);
        guardDistTV = setupTextView(R.id.guard_dist);
        guardBrgTrueTV = setupTextView(R.id.guard_brg_true);
        guardBrgMagTV = setupTextView(R.id.guard_brg_mag);
        guardArrivalTV = setupTextView(R.id.guard_arrival);
        guardArrivalLabel = setupTextView(R.id.guard_arrival_label);
        guardTtgTV = setupTextView(R.id.guard_ttg);
        guardTtgLabel = setupTextView(R.id.guard_ttg_label);
        guardDtgTV = setupTextView(R.id.guard_dtg);
        guardDtgLabel = setupTextView(R.id.guard_dtg_label);
        // appField = setupTextView(R.id.app_field);
        gpsStatusText = (TextView) findViewById(R.id.gps_status_text);
        guardStatusText = (TextView) findViewById(R.id.guard_status_text);
        radiusUnitSpinner = setupSpinner(R.id.radius_unit_spinner,
                app.strRadiusUnits);
        intervalSpinner = setupSpinner(R.id.interval_spinner, app.intIntervals);
        gpsFailSpinner = setupSpinner(R.id.gps_fail_spinner, app.intIntervals);
        toneAlarm = setupCheckBox(R.id.check_tone_alarm);
        voiceAlarm = setupCheckBox(R.id.check_voice_alarm);
        showArrival = setupCheckBox(R.id.check_show_arrival);
        helpWebView = (WebView) findViewById(R.id.help_webview);
        setupWebView(helpWebView);
    }

    public void update() {
        latTV.setText(app.strLat);
        lngTV.setText(app.strLng);
        magDecTV.setText(app.strMagDec);
        speedKnotsTV.setText(app.strSpeedKnots);
        speedMPHTV.setText(app.strSpeedMPH);
        speedKPHTV.setText(app.strSpeedKPH);
        bearing_trueTV.setText(app.strBearingTrue);
        bearing_magTV.setText(app.strBearingMag);
        guardMode.setText(app.strGuardMode);
        guardLatTV.setText(app.strLat);
        guardLngTV.setText(app.strLng);
        guardDistTV.setText(app.strGuardDistance);
        guardBrgTrueTV.setText(app.strGuardBearingTrue);
        guardBrgMagTV.setText(app.strGuardBearingMag);
        guardArrivalTV.setText(app.strGuardArrival);
        guardTtgTV.setText(app.strGuardTtg);
        guardDtgTV.setText(app.strGuardDtg);
        updateFlags();
    }

    public void updateFlags() {
        // GPS delayed = yellow, fail = red, otherwise green
        int tc = app.gpsValid ? app.gpsDelayed ? 1 : 2 : 0;
        gpsStatusText.setTextColor(app.textColors[tc]);
        // Guard inactive = yellow, otherwise anchor alarm = red,
        // otherwise green
        int guardState = (!app.configuration.guarding) ? 1 : (app.guardAlarm) ? 0 : 2;
        guardStatusText.setTextColor(app.textColors[guardState]);
        guardMode.setTextColor(app.textColors[guardState]);
    }

    private void setupTabs() {
        tabHost = (TabHost) findViewById(R.id.tab_host);
        tabHost.setup();
        TabHost.TabSpec spec;
        spec = tabHost.newTabSpec("Enroute");
        spec.setContent(R.id.enroute_view);
        spec.setIndicator("Enroute");
        tabHost.addTab(spec);
        spec = tabHost.newTabSpec("Anchor");
        spec.setContent(R.id.watch_view);
        spec.setIndicator("Anchor");
        tabHost.addTab(spec);
        spec = tabHost.newTabSpec("Setup");
        spec.setContent(R.id.setup_view);
        spec.setIndicator("Setup");
        tabHost.addTab(spec);
        spec = tabHost.newTabSpec("Help");
        spec.setContent(R.id.help_webview);
        spec.setIndicator("Help");
        tabHost.addTab(spec);
        tabHost.setCurrentTab(app.configuration.activeTab);
    }

    private Spinner setupSpinner(int id, String[] items) {
        Spinner sp = (Spinner) findViewById(id);
        ArrayAdapter<String> aa = new ArrayAdapter<String>(this,
                android.R.layout.simple_spinner_item, items);
        aa.setDropDownViewResource(android.R.layout.simple_spinner_dropdown_item);
        sp.setAdapter(aa);
        sp.setOnItemSelectedListener(new AdapterView.OnItemSelectedListener() {
            public void onItemSelected(AdapterView<?> adapterView, View view,
                                       int i, long l) {
                readControlValues(null);
            }

            public void onNothingSelected(AdapterView<?> adapterView) {
                return;
            }
        });
        return sp;
    }

    private Spinner setupSpinner(int id, int[] items) {
        String[] array = new String[items.length];
        int i = 0;
        for (int x : items) {
            array[i++] = String.format("%d", x);
        }
        return setupSpinner(id, array);
    }

    private EditText setupEditText(int id) {
        EditText et = (EditText) findViewById(id);
        if (et != null) {
            et.addTextChangedListener(new TextWatcher() {

                public void afterTextChanged(Editable arg0) {
                    // TODO Auto-generated method stub
                    readControlValues(null);
                }

                public void beforeTextChanged(CharSequence s, int start,
                                              int count, int after) {
                    // TODO Auto-generated method stub

                }

                public void onTextChanged(CharSequence s, int start,
                                          int before, int count) {
                    // TODO Auto-generated method stub

                }
            });
        }
        return et;
    }

    private TextView setupTextView(int id) {
        return (TextView) findViewById(id);
    }

    private CheckBox setupCheckBox(int resource) {
        CheckBox cb = (CheckBox) findViewById(resource);
        // for some reason, to get action clicks,
        // this had to be set up in the layout file
        // cb.setOnCheckedChangeListener(new OnCheckedChangeListener() {
        // public void onCheckedChanged(CompoundButton buttonView,
        // boolean isChecked) {
        // readControlValues(null);
        // }
        // });
        return cb;
    }

    public void readControlValues(View v) {
        if (!suppressRead) {
            app.configuration.intervalIndex = intervalSpinner
                    .getSelectedItemPosition();
            app.configuration.unitIndex = radiusUnitSpinner
                    .getSelectedItemPosition();
            app.configuration.gpsFailLimit = gpsFailSpinner
                    .getSelectedItemPosition();
            try {
                String s = guardRadius.getText().toString();
                app.configuration.guardRadius = Double.parseDouble(s);
            } catch (Exception e) {
                app.beep();
                app.configuration.guardRadius = 0;
            }
            app.configuration.anchorAlarmMessage = anchorAlarmMessage.getText()
                    .toString();
            app.configuration.gpsAlarmMessage = gpsAlarmMessage.getText()
                    .toString();
            app.configuration.toneAlarm = toneAlarm.isChecked();
            app.configuration.voiceAlarm = voiceAlarm.isChecked();
            app.configuration.showArrival = showArrival.isChecked();
            app.configuration.activeTab = tabHost.getCurrentTab();
            int vis = (app.configuration.showArrival) ? View.VISIBLE
                    : View.GONE;
            guardArrivalTV.setVisibility(vis);
            guardArrivalLabel.setVisibility(vis);
            guardTtgTV.setVisibility(vis);
            guardTtgLabel.setVisibility(vis);
            guardDtgTV.setVisibility(vis);
            guardDtgLabel.setVisibility(vis);
            app.serialize();
        }
    }

    protected void writeControlValues() {
        suppressRead = true;
        intervalSpinner.setSelection(app.configuration.intervalIndex);
        radiusUnitSpinner.setSelection(app.configuration.unitIndex);
        gpsFailSpinner.setSelection(app.configuration.gpsFailLimit);
        guardRadius.setText(String
                .format("%.2f", app.configuration.guardRadius));
        anchorAlarmMessage.setText(app.configuration.anchorAlarmMessage);
        gpsAlarmMessage.setText(app.configuration.gpsAlarmMessage);
        toneAlarm.setChecked(app.configuration.toneAlarm);
        voiceAlarm.setChecked(app.configuration.voiceAlarm);
        showArrival.setChecked(app.configuration.showArrival);
        suppressRead = false;
    }

    protected void serialize() {
        readControlValues(null);
    }

    @Override
    public boolean onCreateOptionsMenu(Menu menu) {
        MenuInflater inflater = getMenuInflater();
        inflater.inflate(R.menu.main_menu, menu);
        return true;
    }

    @Override
    public boolean onOptionsItemSelected(MenuItem item) {
        int id = item.getItemId();
        switch (id) {

            case R.id.stop_and_quit:
                quitQuery(null);
                break;
        }
        return true;
    }

    public void quitQuery(View v) {
        if (app.configuration.guarding) {
            createDialog("Okay to quit and abandon anchor watch?", 0);
        } else {
            quit();
        }
    }

    public void resetQuery(View v) {
        createDialog("Okay to reset all values to defaults?", 1);
    }

    protected void createDialog(String msg, int index) {
        Intent intent = new Intent(this, SimpleDialogActivity.class);
        intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
        intent.putExtra("msg", msg);
        intent.putExtra("index", index);
        startActivity(intent);
    }

    protected void dialogCallback(int index) {
        switch (index) {
            case 0:
                quit();
                break;
            case 1:
                reset();
                break;
        }
    }

    public void quit() {
        readControlValues(null);
        app.stopProcesses();
        stopAlarms();
        app.activity = null;
        finish();
    }

    protected void reset() {
        app.reset();
        writeControlValues();
    }

    public void startGuard(View v) {
        if (app.configuration.guardRadius == 0) {
            createDialog("Please enter a guard radius greater than zero.", -1);
        } else {
            stopAlarms();
            app.startGuard(false);
            updateFlags();
        }
    }

    public void stopGuard(View v) {
        stopAlarms();
        app.stopGuard();
        updateFlags();
    }

    @Override
    public void onStart() {
        super.onStart();
        visible = true;

    }

    @Override
    public void onStop() {
        super.onStop();
        visible = false;
        serialize();
    }

    @Override
    public void onPause() {
        super.onPause();
        visible = false;
        serialize();
    }

    @Override
    public void onResume() {
        super.onResume();
        visible = true;
        serialize();
    }

    @Override
    public void onDestroy() {
        super.onDestroy();
        visible = false;
        app.activity = null;
        stopAlarms();
        closeSpeechProcess();
        // end of this view
        serialize();
    }

    // this rouses the device from standby on an alarm

    private void manageWakeLock(boolean activate) {
        if (wakeLock != null) {
            wakeLock.release();
            wakeLock = null;
        }
        if (activate) {
            Window win = getWindow();
            WindowManager.LayoutParams winParams = win.getAttributes();
            winParams.flags |= (WindowManager.LayoutParams.FLAG_DISMISS_KEYGUARD
                    | WindowManager.LayoutParams.FLAG_SHOW_WHEN_LOCKED | WindowManager.LayoutParams.FLAG_ALLOW_LOCK_WHILE_SCREEN_ON);
            win.setAttributes(winParams);
            PowerManager pm = (PowerManager) getSystemService(Context.POWER_SERVICE);
            wakeLock = pm.newWakeLock(PowerManager.FULL_WAKE_LOCK
                            | PowerManager.ACQUIRE_CAUSES_WAKEUP,
                    "Activate Screen on Alarm");
            wakeLock.acquire();
        }
    }

    protected void processAlarm() {
        if (app.configuration.guarding && app.guardAlarm) {
            if (!alarmTripped) {
                tabHost.setCurrentTab(1);
                manageWakeLock(true);
                alarmTripped = true;

                if (!app.configuration.voiceAlarm
                        && !app.configuration.toneAlarm) {
                    app.configuration.toneAlarm = true;
                    toneAlarm.setChecked(app.configuration.toneAlarm);
                }
                if (app.configuration.voiceAlarm) {
                    Intent checkIntent = new Intent();
                    checkIntent
                            .setAction(TextToSpeech.Engine.ACTION_CHECK_TTS_DATA);
                    startActivityForResult(checkIntent, SPEECH_DATA_CHECK);

                }
                if (app.configuration.toneAlarm && app.ringtone != null) {
                    app.ringtone.start();
                }
            }
        } else {
            alarmTripped = false;
            stopAlarms();
        }
        updateFlags();
    }

    // TTS requirement
    public void onInit(int arg0) {
        speechReady = true;
    }

    @Override
    protected void onActivityResult(int requestCode, int resultCode, Intent data) {
        super.onActivityResult(requestCode, resultCode, data);
        if (requestCode == SPEECH_DATA_CHECK) {
            if (resultCode == TextToSpeech.Engine.CHECK_VOICE_DATA_PASS) {
                // success, create the TTS instance
                if (mTts == null) {
                    mTts = new TextToSpeech(this, this);
                }
                if (mTts != null) {
                    extraHash = new HashMap<String, String>();
                    extraHash.put(TextToSpeech.Engine.KEY_PARAM_STREAM,
                            String.valueOf(AudioManager.STREAM_ALARM));
                    final String msg = app.getAlarmMessage();
                    speakThread = new Thread() {
                        public void run() {
                            try {
                                speaking = true;
                                while (speaking) {
                                    waitForSpeechReady();
                                    mTts.speak(msg, TextToSpeech.QUEUE_FLUSH,
                                            extraHash);
                                    if (speaking) {
                                        Thread.sleep(pauseBetweenPromptsMSec);
                                    }
                                }
                                stopVoiceMessage();
                            } catch (Exception e) {
                                // e.printStackTrace();
                            }
                        }
                    };
                    speakThread.start();
                }
            } else {
                app.beep();
                app.makeToast("Cannot synthesize voice");
                // missing data, install it
                Intent installIntent = new Intent();
                installIntent
                        .setAction(TextToSpeech.Engine.ACTION_INSTALL_TTS_DATA);
                startActivity(installIntent);
            }
        }
    }

    protected void waitForSpeechReady() {
        try {
            while (!speechReady || (mTts != null && mTts.isSpeaking())) {
                Thread.sleep(100);
            }
        } catch (InterruptedException e) {
            // e.printStackTrace();
        }
    }

    private void stopVoiceMessage() {
        if (mTts != null) {
            // waitForSpeechReady();
            mTts.stop();
        }
    }

    private void stopAlarms() {

        stopVoiceMessage();
        if (app.ringtone != null) {
            app.ringtone.stop();
        }
        manageWakeLock(false);
        alarmTripped = false;
        speaking = false;
    }

    private void closeSpeechProcess() {
        if (mTts != null) {
            stopVoiceMessage();
            mTts.shutdown();
            mTts = null;
        }
    }

    private void setupHelp() {
        String helpPage = loadLocalAsset("help.html");
        try {
            String v = getPackageManager().getPackageInfo(getPackageName(), 0).versionName;
            helpPage = helpPage.replaceAll("#version#", v);
        } catch (NameNotFoundException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        }
        loadWebViewData(helpPage, helpWebView);
    }

    private String loadLocalAsset(String name) {
        StringBuilder data = new StringBuilder();
        try {
            BufferedReader isr = new BufferedReader(new InputStreamReader(
                    getAssets().open(name)));
            String line;

            while ((line = isr.readLine()) != null) {
                line = line.trim();
                if (line.length() > 0) {
                    data.append(line + "\n");
                }
            }
            isr.close();
        } catch (Exception e) {
            app.showStackTrace(e);
        }
        return data.toString();
    }

    private void loadWebViewData(final String content, final WebView target) {
        handler.post(new Runnable() {
            public void run() {
                // TODO Auto-generated method stub
                //target.loadData(content, "text/html", null);
                target.loadDataWithBaseURL(null, content, "text/html", "UTF-8", null);
                target.reload();
            }
        });
    }

    private void setupWebView(WebView wv) {
        WebSettings set = wv.getSettings();
        set.setBuiltInZoomControls(true);
        set.setJavaScriptEnabled(true);
        set.setDefaultTextEncodingName("utf-16");
    }
}