/***************************************************************************
 *   Copyright (C) 2013 by Paul Lutus                                      *
 *   http://arachnoid.com/administration                                   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

package com.arachnoid.anchorsentinel;

import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.text.SimpleDateFormat;
import java.util.GregorianCalendar;
import java.util.Stack;

import android.Manifest;
import android.app.Application;
import android.content.Context;
import android.content.Intent;
import android.content.pm.PackageManager;
import android.hardware.Sensor;
import android.hardware.SensorEvent;
import android.hardware.SensorEventListener;
import android.location.Location;
import android.location.LocationListener;
import android.location.LocationManager;
import android.media.AudioManager;
import android.media.ToneGenerator;
import android.os.Handler;
import android.util.Log;
import android.widget.Toast;

final public class AnchorSentinelApplication extends Application implements
		SensorEventListener {

	// test mode only, allows bounds test without moving
	// set testing = false to disable
	boolean testing = false;
	int counter = 0;

	// GPS test setup for emulator, normally set gpsTest = false
	boolean gpsTest = false;
	double testLat = 58.1234;
	double testLng = -146.4321;
	double testVelocity = .00005;

	private static AnchorSentinelApplication singleton = null;
	public SerializedData configuration;
	String serialObjectPath = "AnchorSentinel.obj";
	AnchorSentinelActivity activity = null;
	Context context = null;
	LocationManager locationManager = null;
	LocationListener locationListener = null;
	MyRingtone ringtone;

	Intent gpsMonitorService = null;

	Handler handler;
	Runnable periodicTask;
	int retryCount;

	boolean gpsValid = true;
	boolean gpsDelayed = false;

	// special error indication: latitude = -100
	// double lat = 0, lng = 0;
	long time, old_time = -1;
	MagneticDeclination magComp = null;
	double speedKnots;
	double magDec = 0;
	String strLat, strLng, strMagDec;
	String strSpeedKnots, strSpeedMPH, strSpeedKPH, strBearingTrue,
			strBearingMag;
	Stack<Position> locations;

	Position currentPos = new Position();
	Position guardPos = new Position();
	double guardDistance = 0;
	double guardBearingTrue = 0;
	boolean guardAlarm = false;
	String strGuardMode = "";
	String strGuardDistance = "";
	String strGuardBearingTrue = "";
	String strGuardBearingMag = "";
	String strGuardArrival = "";
	String strGuardTtg = "";
	String strGuardDtg = "";
	String unitString = "";
	int iColorRed = 0xffff0000;
	int iColorYellow = 0xffffff00;
	int iColorGreen = 0xff00ff00;
	int[] ledColors,textColors;

	String[] strRadiusUnits = new String[] { "Feet", "Meters",
			"Nautical miles", "Statute miles", "Kilometers" };

	double[] dblRadiusConversions = new double[] { 6076.115485564, 1852, 1,
			1.150777146, 1.852 };

	int[] intIntervals = new int[] { 1, 2, 4, 8, 16, 32, 64, 128, 256, 512,
			1024 };

	int gpsFailCount = 0;

	public AnchorSentinelApplication getInstance() {
		return singleton;
	}

	@Override
	public void onCreate() {
		super.onCreate();
		singleton = this;
		ledColors = new int[] { R.drawable.ic_led_red,
				R.drawable.ic_led_yellow, R.drawable.ic_led_green };
		textColors = new int[] {iColorRed,iColorYellow,iColorGreen};
		handler = new Handler();
		locations = new Stack<Position>();
		context = getApplicationContext();
		magComp = new MagneticDeclination();
		ringtone = new MyRingtone(this);
		configuration = new SerializedData();
		deSerialize();
		startProcesses();
		if (configuration.guarding) {
			configuration.guarding = false;
			startGuard(true);
		}
	}

	protected void startProcesses() {
		if (gpsMonitorService == null) {
			gpsMonitorService = new Intent(this, GPSMonitorService.class);
			startService(gpsMonitorService);
		}
	}

	protected void stopProcesses() {
		// stopGuard();
		stopService();
		serialize();
	}

	protected void stopService() {
		if (gpsMonitorService != null) {
			stopService(gpsMonitorService);
			gpsMonitorService = null;
		}
	}

	protected void reset() {
		configuration = new SerializedData();
		stopGuard();
	}

	public void onAccuracyChanged(Sensor sensor, int accuracy) {
		// TODO Auto-generated method stub

	}

	public void onSensorChanged(SensorEvent event) {
		// TODO Auto-generated method stub

	}

	protected void updateLocation(Location loc) {

		// only fail GPS if sequential lost fixes exceed fail count limit
		gpsDelayed = (!gpsTest && loc == null);
		gpsFailCount = (gpsDelayed) ? gpsFailCount + 1 : 0;
		gpsValid = (gpsFailCount < intIntervals[configuration.gpsFailLimit]);
		// flagged invalid by default
		currentPos = new Position();
		if (gpsTest || (gpsValid && loc != null)) {
			double lat = 0;
			double lng = 0;
			if (gpsTest) {
				testLat += testVelocity;
				testLng -= testVelocity / 3.0;
				lat = testLat;
				lng = testLng;
			} else {
				lat = loc.getLatitude();
				lng = loc.getLongitude();
			}
			// GPS time is rounded off to nearest second
			// and the GPS velocity and bearing are not very good
			// time = loc.getTime();
			time = System.currentTimeMillis();
			currentPos = new Position(lat, lng, time / 1000.0);

			magDec = magComp.computePoint(lat, lng);
			strLat = toDegMin(lat, "N", "S");
			strLng = toDegMin(lng, "E", "W");
			String strSign = (magDec > 0) ? "E" : "W";
			double amagDec = Math.abs(magDec);
			strMagDec = String.format("%05.1f°%s", amagDec, strSign);
			Position oldpos = new Position();
			int interval = intIntervals[configuration.intervalIndex];
			if (locations.size() > 0 && locations.size() <= interval) {
				oldpos = locations.get(0);
			} else {
				while (locations.size() > interval) {
					oldpos = locations.remove(0);
				}
			}
			locations.push(currentPos);
			if (oldpos.valid) {
				double delta_t = (currentPos.time - oldpos.time);
				Pair r = distBrg(currentPos, oldpos);
				speedKnots = r.x * 3600 / delta_t;
				double mph = speedKnots * 1.150799;
				double kph = speedKnots * 1.852;
				double bearing = r.y;
				strSpeedKnots = String.format("%5.1f knots", speedKnots);
				strSpeedMPH = String.format("%5.1f MPH", mph);
				strSpeedKPH = String.format("%5.1f KPH", kph);
				strBearingTrue = String.format("%05.1f°", bearing);
				bearing = (bearing - magDec + 360.0) % 360.0;
				strBearingMag = String.format("%05.1f°", bearing);
			}
		} else {
			String s = gpsValid ? "GPS delayed" : "No GPS fix";
			strLat = s;
			strLng = s;
			strMagDec = "...";
			strSpeedKnots = "...";
			strSpeedMPH = "...";
			strSpeedKPH = "...";
			strBearingTrue = "...";
			strBearingMag = "...";
		}
		processGuard();
		if (activity != null) {
			synchronized (activity) {
				activity.update();
			}
		}
	}

	protected void processGuard() {
		double normalizedDistance = 0.0;
		double ttg = 0;
		guardDistance = 0.0;
		guardBearingTrue = 0.0;
		if (configuration.guarding) {
			if (!gpsValid) {
				guardAlarm = true;
			} else {
				if (guardPos.valid && currentPos.valid) {
					Pair r = distBrg(currentPos, guardPos);
					// "convFact" converts to/from nautical miles
					double convFact = dblRadiusConversions[configuration.unitIndex];
					unitString = strRadiusUnits[configuration.unitIndex];
					// r.x is distance in nautical miles
					guardDistance = r.x * convFact;
					// bearing is in decimal degrees
					guardBearingTrue = r.y;
					normalizedDistance = (configuration.guardRadius > 0) ? guardDistance
							/ configuration.guardRadius
							: 1;

					guardAlarm = (normalizedDistance >= 1.0);

					// for testing
					// speedKnots = 10.0;

					double dtg = (1.0 - normalizedDistance)
							* configuration.guardRadius;
					
					
					strGuardDtg = String.format("%7.02f %s",dtg,unitString);

					ttg = (speedKnots == 0.0) ? 0 : dtg
							/ (speedKnots * convFact);

					int ttgi = (int) (ttg * 3600 + 0.5);

					strGuardTtg = toDHMS(ttgi);

					long arrivaltime = System.currentTimeMillis() + ttgi * 1000;

					GregorianCalendar gc = new GregorianCalendar();
					gc.setTimeInMillis(arrivaltime);
					SimpleDateFormat sdf = new SimpleDateFormat(
							"yyyy-MM-dd HH:mm:ss");
					strGuardArrival = sdf.format(gc.getTime());

					// this is for testing only
					if (testing) {
						counter += 1;
						guardAlarm = guardAlarm || counter >= 10;
					}
				}
			}
			if (guardAlarm) {
				if (activity != null) {
					if (!activity.visible) {
						activity.setVisible(true);
						startActivity(activity.getIntent());
					}

				} else {
					Intent intent = new Intent(this,
							AnchorSentinelActivity.class);
					intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
					startActivity(intent);
				}
			}
			if (activity != null) {
				activity.processAlarm();
			}
			strGuardDistance = String.format("%7.02f %s (%3.0f%%)", guardDistance,unitString,
					normalizedDistance * 100);
			strGuardBearingTrue = String.format("%05.1f°", guardBearingTrue);
			double gbm = (guardBearingTrue - magDec + 360.0) % 360.0;
			strGuardBearingMag = String.format("%05.1f°", gbm);
			strGuardMode = guardAlarm?"Anchor Alarm":"Monitoring";
		} else {
			strGuardMode = "Inactive";
			strGuardDistance = "...";
			strGuardBearingTrue = "...";
			strGuardBearingMag = "...";
			strGuardArrival = "...";
			strGuardTtg = "...";
			strGuardDtg = "...";
		}
	}

	protected String getAlarmMessage() {
		return (this.gpsValid) ? configuration.anchorAlarmMessage
				: configuration.gpsAlarmMessage;
	}

	private String toDegMin(double a, String ne, String sw) {
		String s = (a < 0) ? sw : ne;
		a = Math.abs(a);
		double d = Math.floor(a);
		double m = ((a - d) * 60);
		return String.format("%03.0f° %06.3f' %s", d, m, s);
	}

	private String toDHMS(int v) {
		int s = v % 60;
		v /= 60;
		int m = v % 60;
		v /= 60;
		int h = v % 24;
		v /= 24;
		return String.format("%5d days %02d:%02d:%02d", v, h, m, s);
	}

	private double degrees(double x) {
		return x * 180 / Math.PI;
	}

	private double radians(double x) {
		return x * Math.PI / 180;
	}

	/*
	 * input: two geographical positions in decimal degrees
	 * 
	 * output: distance in nautical miles, bearing in decimal degrees
	 */

	private Pair distBrg(Position b, Position a) {
		double dlo = radians(b.lng - a.lng);
		double alat = radians(a.lat);
		double blat = radians(b.lat);
		double cdl = Math.cos(dlo);
		double sbl = Math.sin(alat);
		double cbl = Math.cos(alat);
		double dist = Math.acos(sbl * Math.sin(blat) + cbl * Math.cos(blat)
				* cdl);
		// convert from radians to nautical miles
		dist = degrees(dist) * 60;
		double brg = Math.atan2(Math.sin(dlo), (cbl * Math.tan(blat) - sbl
				* cdl));
		// no negative angles
		if (brg < 0) {
			brg = Math.PI * 2 + brg;
		}
		return new Pair(dist, degrees(brg));
	}

	protected void deSerialize() {
		try {
			FileInputStream fis = openFileInput(serialObjectPath);
			if (fis != null && fis.available() > 0) {
				ObjectInputStream in = new ObjectInputStream(fis);
				configuration = (SerializedData) in.readObject();
				in.close();
			}
		} catch (Exception e) {
			// showStackTrace(e);
		}
	}

	protected void serialize() {
		try {
			FileOutputStream fos = openFileOutput(serialObjectPath,
					Context.MODE_PRIVATE);
			ObjectOutputStream out = new ObjectOutputStream(fos);
			out.writeObject(configuration);
			out.close();
		} catch (Exception e) {
			showStackTrace(e);
		}
	}

	protected void startGuard(boolean wait) {
		if (wait && !currentPos.valid) {
			final int delay = 1000; // milliseconds
			retryCount = 8;
			periodicTask = new Runnable() {
				public void run() {
					if (!currentPos.valid && retryCount-- > 0) {
						handler.postDelayed(periodicTask, delay);
					} else {
						startGuard2();
					}
				}
			};
			periodicTask.run();
		} else {
			startGuard2();
		}
	}

	protected void startGuard2() {
		if (currentPos.valid) {
			guardPos = new Position(currentPos);
			configuration.guarding = true;
			counter = 0;
			processGuard();
		} else {
			beep();
		}
	}

	protected void stopGuard() {
		configuration.guarding = false;
		counter = 0;
	}

	protected void showStackTrace(Exception e) {
		StringWriter sw = new StringWriter();
		e.printStackTrace(new PrintWriter(sw));
		String stacktrace = sw.toString();
		Log.e("error:", stacktrace);
	}

	public void beep() {
		android.media.ToneGenerator tg = new ToneGenerator(
				AudioManager.STREAM_ALARM, 50);
		tg.startTone(ToneGenerator.TONE_CDMA_SOFT_ERROR_LITE);
	}

	protected void makeToast(String msg) {
		Toast.makeText(context, msg, Toast.LENGTH_SHORT).show();
	}

	protected boolean canAccessLocation() {
		return(hasPermission(Manifest.permission.ACCESS_FINE_LOCATION));
	}

	protected boolean canWakeLock() {
		return(hasPermission(Manifest.permission.WAKE_LOCK));
	}

	protected boolean hasPermission(String perm) {
		return(PackageManager.PERMISSION_GRANTED==checkSelfPermission(perm));
	}
}
