/***************************************************************************
 *   Copyright (C) 2011 by Paul Lutus                                      *
 *   http://arachnoid.com/administration                                   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

package com.arachnoid.carpetempus;

import android.content.Context;
import android.os.Handler;
import android.util.AttributeSet;
import android.view.LayoutInflater;
import android.view.View;
import android.widget.LinearLayout;
import android.widget.TextView;

final public class MyTimeSetEntryField extends LinearLayout {

	Context context;
	MyTimeSetEntryField carryField = null;
	String[] ampmArray = new String[]{"am","pm"};
	boolean ampmMode = false;
	private TextView mText;
	private int currentValue = 0;
	private int minVal = 0;
	private int maxVal = 59;
	private long repeatDelayMs = 100; // milliseconds
	private boolean incrementMode;
	private boolean decrementMode;
	Handler repeatHandler;
	AttributeSet attrs = null;
	private MyTimeSetButton incrementButton;
	private MyTimeSetButton decrementButton;

	private final Runnable mRunnable = new Runnable() {
		public void run() {
			if (incrementMode) {
				changeCurrentValue(1);
				repeatHandler.postDelayed(this, repeatDelayMs);
			} else if (decrementMode) {
				changeCurrentValue(-1);
				repeatHandler.postDelayed(this, repeatDelayMs);
			}
		}
	};


	public MyTimeSetEntryField(Context context) {
		super(context);
		init(context);
	}

	public MyTimeSetEntryField(Context context, AttributeSet attrs) {
		super(context, attrs);
		this.attrs = attrs;
		init(context);
	}

	public MyTimeSetEntryField(Context context, AttributeSet attrs, int defStyle) {
		super(context, attrs, defStyle);
		this.attrs = attrs;
		init(context);
	}

	private void init(Context context) {
		this.context = context;

		String min = attrs.getAttributeValue(null, "minVal");
		String max = attrs.getAttributeValue(null, "maxVal");
		// alpha reveals ampm mode
		if(min.matches(".*\\p{Alpha}.*")) {
			ampmMode = true;
			minVal = 0;
			maxVal = 1;
		}
		else {
			minVal = Integer.parseInt(min);
			maxVal = Integer.parseInt(max);
		}

		LayoutInflater inflater =
				(LayoutInflater) context.getSystemService(Context.LAYOUT_INFLATER_SERVICE);
		inflater.inflate(R.layout.numberpickerlayout, this, true);
		mText = (TextView) findViewById(R.id.display_field);
		repeatHandler = new Handler();


		OnClickListener clickListener = new OnClickListener() {
			public void onClick(View v) {
				if (!mText.hasFocus()) mText.requestFocus();
				// now perform the increment/decrement
				if (R.id.increment == v.getId()) {
					changeCurrentValue(1);

				} else if (R.id.decrement == v.getId()) {
					changeCurrentValue(-1);
				}
			}
		};

		OnLongClickListener longClickListener = new OnLongClickListener() {

			public boolean onLongClick(View v) {

				mText.clearFocus();

				if (R.id.increment == v.getId()) {
					incrementMode = true;
					repeatHandler.post(mRunnable);
				} else if (R.id.decrement == v.getId()) {
					decrementMode = true;
					repeatHandler.post(mRunnable);
				}

				return true;
			}
		};

		incrementButton = (MyTimeSetButton) findViewById(R.id.increment);
		assert(incrementButton != null);
		incrementButton.setOnClickListener(clickListener);
		incrementButton.setOnLongClickListener(longClickListener);
		incrementButton.setNumberPicker(this);

		decrementButton = (MyTimeSetButton) findViewById(R.id.decrement);
		assert(decrementButton != null);
		decrementButton.setOnClickListener(clickListener);
		decrementButton.setOnLongClickListener(longClickListener);
		decrementButton.setNumberPicker(this);

		updateView();

	}

	public void setValue(int v) {
		currentValue = v;
		updateView();
	}

	public int getValue() {
		return currentValue;
	}

	public void setMaxValue(int mv) {
		maxVal = mv;
	}

	public void stopLongPress() {
		// stop long click sequence
		incrementMode = false;
		decrementMode = false;
	}

	protected void changeCurrentValue(int increment) {
		currentValue += increment;
		if (currentValue > maxVal) {
			currentValue = minVal;
			performCarry(1);
		} else if (currentValue < minVal) {
			currentValue = maxVal;
			performCarry(-1);
		}
		updateView();
	}

	protected void setCarryField(MyTimeSetEntryField cf) {
		carryField = cf;
	}

	private void performCarry(int carry) {
		if (carryField != null) {
			carryField.changeCurrentValue(carry);
		}
	}

	private void updateView() {
		mText.setText(formatNumber(currentValue));
	}

	private String formatNumber(int value) {
		if(ampmMode) {
			return ampmArray[value];	
		}
		else {
			// deal with am/pm hour anomaly
			int v = value;
			if(maxVal == 11) {
				v = (v == 0)?12:v;
			}
			return String.format("%02d", v);
		}

	}

}
