/***************************************************************************
 *   Copyright (C) 2012 by Paul Lutus                                      *
 *   lutusp@arachnoid.com                                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package com.arachnoid.tankcalcandroid;

import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.ColorMatrix;
import android.graphics.ColorMatrixColorFilter;
import android.graphics.Paint;
import android.graphics.Path;
import android.graphics.PorterDuff;
import android.graphics.PorterDuffXfermode;
import android.graphics.Rect;
import android.os.Handler;
import android.util.AttributeSet;
import android.util.Log;
import android.view.MotionEvent;
import android.view.View;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;

/**
 * @author lutusp
 */
final public class TankImageView extends View {

    TankProperties sharedTP;
    TankCalcAndroidActivity activity;
    TankCalcAndroidApplication app;
    TankProcessor tvi;
    ImageGenerator imageGenHi, imageGenLo;
    double touchScale = 200;
    double translateScale = 3000;
    double oldTouchX1 = 0, oldTouchY1 = 0;
    double oldTouchX2 = 0, oldTouchY2 = 0;
    int oldAction = 0;
    double oldMag = 0;
    double oldTouch = 0;
    int changeCount = 0;
    Handler timerHandler = null;
    Runnable delayTimer = null;
    int timeDelay = 250;

    int xsize = -1, ysize = -1;
    int oldxs = -1, oldys = -1;

    Bitmap imagea, imageb, imagec;

    int xCenter, yCenter;
    double paintScale;
    int graphicBackgroundColor = Color.BLACK;
    int graphicForegroundColor = Color.WHITE;

    double threeDFactor = 0.04;

    double graphicPerspectiveFactor = 40;

    double polygonTransparency = .5;

    Transform3D graphicTransform;

    boolean lowResMode = false;

    public TankImageView(Context context) {
        super(context);
        setup(context);
    }

    public TankImageView(Context context, AttributeSet attrs) {
        super(context, attrs);
        setup(context);
    }

    public TankImageView(Context context, AttributeSet attrs, int defStyle) {
        super(context, attrs, defStyle);
        setup(context);
    }

    public void setup(Context a) {
        timerHandler = new Handler();
        activity = (TankCalcAndroidActivity) a;
        app = activity.app;
        sharedTP = activity.app.sharedTP;
        tvi = app.tankProcessor;
        imageGenHi = app.imageGenHi;
        imageGenLo = app.imageGenLo;
        graphicTransform = app.graphicTransform;
        //if (true) return;
        setDefaults();
        updateSettings();
    }

    void setLowMode() {
        lowResMode = true;
        delayTimer = new Runnable() {
            public void run() {
                lowResMode = false;
                invalidate();
            }
        };
        timerHandler.postDelayed(delayTimer, timeDelay);
    }

    @Override
    public boolean performClick() {
        // Calls the super implementation, which generates an AccessibilityEvent
        // and calls the onClick() listener on the view, if any
        super.performClick();

        // Handle the action for the custom click here

        return true;
    }

    @Override
    public boolean onTouchEvent(MotionEvent me) {
        //if(true) return false;
        super.onTouchEvent(me);
        sharedTP = activity.app.sharedTP;
        double mag = 0;
        double x1 = 0, y1 = 0;
        double x2 = 0, y2 = 0;
        double dx1, dy1;
        double dx2, dy2;
        double dm;

        int action = me.getAction();

        if (action == MotionEvent.ACTION_UP) {
            performClick();
        }

        int newTouch = me.getPointerCount();

        if (newTouch > 1) {
            try {
                // must use index and id to recover multi touches
                int id = me.getPointerId(0);
                x1 = me.getX(id);
                y1 = me.getY(id);
                id = me.getPointerId(1);
                x2 = me.getX(id);
                y2 = me.getY(id);
            } catch (Exception e) {
                activity.app.showStackTrace(e);
            }

        } else {
            // single touch
            x1 = me.getX();
            y1 = me.getY();
        }

        if (action != oldAction || newTouch != oldTouch) {
            // pass up initial events to avoid confused behavior
            changeCount = 2;
        }

        if (action == MotionEvent.ACTION_MOVE
                || action == MotionEvent.ACTION_DOWN) {
            // reset old touch position
            if (changeCount > 0 || action == MotionEvent.ACTION_DOWN
                    || newTouch != oldTouch) {
                oldTouchX1 = x1;
                oldTouchY1 = y1;
                oldTouchX2 = x2;
                oldTouchY2 = y2;
                oldMag = mag;
            }
            // acquire differentials
            dx1 = (oldTouchX1 - x1) * touchScale / xsize;
            dx2 = (oldTouchX2 - x2) * touchScale / xsize;
            dy1 = (oldTouchY1 - y1) * touchScale / ysize;
            dy2 = (oldTouchY2 - y2) * touchScale / ysize;

            // multitouch = zoom and translate
            if (newTouch > 1) {
                double px = Math.pow(x1 - x2, 2);
                double py = Math.pow(y1 - y2, 2);
                mag = Math.sqrt(px + py);
                if (action == MotionEvent.ACTION_DOWN) {
                    oldMag = mag;
                }
            }
            if (changeCount == 0 && newTouch == oldTouch) {
                if (newTouch > 1) {

                    // translate: fingers in same direction
                    // take relationship between Y and Z
                    // axes into account

                    double qxx = (dx1 + dx2)
                            * Math.cos(sharedTP.modelAngleX * Constants.radian);
                    double qxz = (dx1 + dx2)
                            * Math.sin(sharedTP.modelAngleX * Constants.radian);

                    double qyx = (dy1 + dy2)
                            * Math.cos(sharedTP.modelAngleY * Constants.radian);
                    double qyz = (dy1 + dy2)
                            * Math.sin(sharedTP.modelAngleY * Constants.radian);

                    double tscx = translateScale
                            / (ysize * sharedTP.modelScale);
                    double tscy = translateScale
                            / (xsize * sharedTP.modelScale);
                    sharedTP.modelTranslateX += qxx * tscx;
                    sharedTP.modelTranslateZ += (qxz + qyz) * tscy;
                    sharedTP.modelTranslateY -= qyx * tscy;
                    setTranslations();

                    // zoom: fingers in different directions

                    dm = (mag - oldMag) * 3 * sharedTP.modelScale
                            / Math.sqrt(xsize * xsize + ysize * ysize);
                    sharedTP.modelScale += dm;

                } else { // single touch = rotate
                    sharedTP.modelAngleX += dx1;
                    sharedTP.modelAngleY += dy1;
                    setRotations();

                }
            }
            invalidate();
        }


        oldAction = action;
        oldTouchX1 = x1;
        oldTouchY1 = y1;
        oldTouchX2 = x2;
        oldTouchY2 = y2;
        oldMag = mag;
        oldTouch = newTouch;
        if (changeCount > 0) {
            changeCount--;
        }
        setLowMode();
        return true;
    }

    void p(String s) {
        System.out.println(s);
    }

    String pn(double n) {
        return String.format("%.4f", n);
    }

    @Override
    protected void onDraw(Canvas canvas) {
        if (getVisibility() == View.VISIBLE) {
            //Log.e("onDraw", "*** start " + canvas);
            // FIXME
            render(canvas);
            //Log.e("onDraw", "*** end " + canvas);
        }
    }

    public void render(Canvas canvas) {
        // must refresh this reference
        sharedTP = activity.app.sharedTP;
        setTranslations();
        setRotations();
        // use clipping bounds, not width and height
        Rect rect = canvas.getClipBounds();
        xsize = rect.right;
        ysize = rect.bottom;
        //int iw = getWidth();
        //int ih = getHeight();

        if (xsize == 0 || ysize == 0) {
            return;
        }

        // weirdBug = (xsize != iw || ysize != ih);
        graphicPerspectiveFactor = (sharedTP.modelScaleFactor * 3);
        Paint paint = new Paint();
        drawSequence(canvas, paint);
    }

    public void drawSequence(Canvas canvas, Paint paint) {
        ImageGenerator gen = (lowResMode) ? imageGenLo : imageGenHi;
        if (graphicTransform.sx * graphicTransform.cy > 0) {
            drawImage(canvas, paint, gen.leftView);
        } else {
            drawImage(canvas, paint, gen.rightView);
        }
    }

    class PlotElement {
        int color;
        Path poly;
        double avgz;

        public PlotElement(int c, Path p, double av) {
            color = c;
            poly = p;
            avgz = av;
        }
    }

    private ColorMatrix getColorMatrix() {
        return new ColorMatrix(new float[]{
                -1, 0, 0, 0, 255,
                0, -1, 0, 0, 255,
                0, 0, -1, 0, 255,
                0, 0, 0, 1, 0
        });
    }

    class PlotComparator implements Comparator<PlotElement> {
        public int compare(PlotElement p1, PlotElement p2) {
            return (p1.avgz == p2.avgz) ? 0 : (p1.avgz > p2.avgz) ? 1 : -1;
        }
    }

    private static final float[] invertMat(float c) {
        return new float[]{
                -1.0f, 0, 0, 0, c, // red
                0, -1.0f, 0, 0, c, // green
                0, 0, -1.0f, 0, c, // blue
                0, 0, 0, 1.0f, 0  // alpha
        };
    }

    public void drawImage(Canvas g, Paint paint, ArrayList<ImageArray> src) {
        if (src != null) {
            polygonTransparency = (sharedTP.graphicTransparent) ? .5 : 0;
            if (xsize > 0 && ysize > 0) {
                xCenter = xsize / 2;
                yCenter = ysize / 2;
                double ts = (xsize < ysize) ? xsize : ysize;
                paintScale = sharedTP.modelScale * ts * .001;
                int bgcolor = (sharedTP.graphicInverseMode || sharedTP.graphicAnaglyphMode) ? graphicBackgroundColor
                        : graphicForegroundColor;
                int fgcolor = (sharedTP.graphicInverseMode || sharedTP.graphicAnaglyphMode) ? graphicForegroundColor
                        : graphicBackgroundColor;
                paint.setColor(bgcolor);
                // only allocate a new image buffer
                // if the image has changed size.
                // this is crucial to fast drawing
                Canvas cga = null;
                Canvas cgb = null;
                Canvas cgc = null;
                if (xsize != oldxs || ysize != oldys) {
                    //if (sharedTP.graphicAnaglyphMode) {
                    //if (imagea == null || imageb == null) {
                    imagea = Bitmap.createBitmap(xsize, ysize,
                            Bitmap.Config.ARGB_8888);
                        imageb = Bitmap.createBitmap(xsize, ysize,
                                Bitmap.Config.ARGB_8888);
                        imagec = Bitmap.createBitmap(xsize, ysize,
                                Bitmap.Config.ARGB_8888);
                }
                oldxs = xsize;
                oldys = ysize;
                cga = new Canvas(imagea);
                imagea.eraseColor(bgcolor);
                    cgb = new Canvas(imageb);
                    imageb.eraseColor(bgcolor);
                    cgc = new Canvas(imagec);
                    imagec.eraseColor(bgcolor);

                ArrayList<PlotElement> plotLista = new ArrayList();
                ArrayList<PlotElement> plotListb = new ArrayList();
                //Iterator<ImageArray> it = src.iterator();
                for (ImageArray vec : src) {
                    //ImageArray vec = it.next();
                    //Log.e("IN LOOP","" + vec);
                    if (vec.isActive()) {
                        if (sharedTP.graphicAnaglyphMode) {
                            drawElements(plotLista, vec, threeDFactor,
                                    sharedTP.graphicAnaglyphMode,
                                    Constants.redColor);
                            drawElements(plotListb, vec, -threeDFactor,
                                    sharedTP.graphicAnaglyphMode,
                                    Constants.cyanColor);
                        } else {
                            drawElements(plotLista, vec, 0,
                                    sharedTP.graphicAnaglyphMode, null);
                        }
                    }
                }
                if (sharedTP.polygonMode) {
                    Collections.sort(plotLista, new PlotComparator());
                    if (sharedTP.graphicAnaglyphMode) {
                        for (PlotElement p : plotLista) {
                            paint.setColor(p.color);
                            cga.drawPath(p.poly, paint);
                        }

                        Collections.sort(plotListb, new PlotComparator());
                        for (PlotElement p : plotListb) {
                            paint.setColor(p.color);
                            cgb.drawPath(p.poly, paint);
                        }
                    } else {
                        for (PlotElement p : plotLista) {
                            paint.setColor(p.color);
                            cga.drawPath(p.poly, paint);
                        }
                    }
                }
                if (sharedTP.graphicAnaglyphMode) {
                    paint = new Paint();
                    cgc.drawBitmap(imagea, 0, 0, paint);
                    paint.setXfermode(new PorterDuffXfermode(PorterDuff.Mode.ADD));
                    cgc.drawBitmap(imageb, 0, 0, paint);
                    drawHints(cgc, threeDFactor, sharedTP.graphicAnaglyphMode,
                            Color.RED);
                    drawHints(cgc, -threeDFactor, sharedTP.graphicAnaglyphMode,
                            Color.CYAN);
                    //paint = new Paint();
                    paint.setXfermode(new PorterDuffXfermode(PorterDuff.Mode.SRC_OVER));
                    if (!sharedTP.graphicInverseMode) {
                        paint.setColorFilter(new ColorMatrixColorFilter(invertMat(255f)));
                    }
                    g.drawBitmap(imagec,0,0,paint);
                } else {
                    paint.setColor(bgcolor);
                    g.drawBitmap(imagea, 0, 0, paint);
                    drawHints(g, 0, sharedTP.graphicAnaglyphMode, fgcolor);
                }
            }
        } else {
            Log.e("drawImage", "src = null!");
        }
    }

    private void drawElements(ArrayList<PlotElement> list, ImageArray vec,
                              double threeDFactor, boolean anaglyphMode, IColor color) {
        drawPolygons(list, vec, threeDFactor, anaglyphMode, color);
    }

    int highlight(IColor color, int alpha, double v, int highvalue) {
        int b = (int) (color.b + v * (highvalue - color.b));
        int g = (int) (color.g + v * (highvalue - color.g));
        int r = (int) (color.r + v * (highvalue - color.r));
        return (alpha << 24 | r << 16 | g << 8 | b);
    }

    int anaglyphHighlight(IColor color, int alpha, double v) {
        int b = (int) (v * color.b);
        int g = (int) (v * color.g);
        int r = (int) (v * color.r);
        return (alpha << 24 | r << 16 | g << 8 | b);
    }

    // compAreaAdd() computes a value representing the relation
    // between the view direction and a polygon in three-space.
    // View perpendicular returns 1, on-edge returns 0
    // the "outside" value indicates whether we are viewing
    // a polygon from "outside' or "inside" an object.

    AreaResult compAreaAdd(GPolygon gpoly, Path poly, double threeDFactor) {
        poly.reset();
        //double ox = 0, oy = 0, oz = 0;
        double xya = 0, xza = 0, yza = 0;
        double avgz = 0;
        float fx, fy;
        CartesianPoint p, op = null;
        for (CartesianPoint ip : gpoly.vector) {
            p = ip.clone();
            graphicTransform.transformPoint(p);
            perspective(p, threeDFactor);
            // compute determinants for three axes
            if (op != null) {
                xya += op.x * p.y - op.y * p.x;
                yza += op.y * p.z - op.z * p.y;
                xza += op.x * p.z - op.z * p.x;
            }
            op = p;
            avgz += p.z;
            fx = scalePoint(xCenter, paintScale, p.x);
            fy = scalePoint(yCenter, -paintScale, p.y);
            poly.lineTo(fx, fy);
        }
        avgz /= gpoly.length();
        // if traversal is CCW, xya > 0
        // meaning this polygon is being viewed
        // from outside object
        boolean outside = (xya > 0);
        double xya2 = xya * xya;
        // area always 0 <= area <= 1
        double area = xya2 / (xya2 + xza * xza + yza * yza);
        return new AreaResult(area, outside, avgz);
    }

    private void drawPolygons(ArrayList<PlotElement> list, ImageArray vec,
                              double threeDFactor, boolean anaglyphMode, IColor color) {

        int anaglyphMask = ((color == Constants.cyanColor) ? 0xff00ffff
                : 0xffff0000);
        int highlightValue = 240;

        for (GPolygon[] pa : vec.gpoly) {
            CartesianPoint p = pa[0].vector[0];
            for (GPolygon pb : pa) {
                IColor cc = (anaglyphMode || p == null || p.icolor == null) ? color
                        : p.icolor;
                Path poly = new Path();
                AreaResult result = compAreaAdd(pb, poly, threeDFactor);
                double alpha = result.outside ? 1 - polygonTransparency : 1;
                int ialpha = (int) (255 * alpha);
                int hv;
                if (anaglyphMode) {
                    hv = anaglyphHighlight(cc, ialpha, 1 - result.area)
                            & anaglyphMask;
                } else {
                    hv = highlight(cc, ialpha, result.area, highlightValue);
                }
                list.add(new PlotElement(hv, poly, result.avgz));
            }
        }
    }

    private void drawHints(Canvas canvas, double threeDFactor,
                           boolean threeDMode, int color) {
        if (sharedTP.graphicHints) {
            Paint paint = new Paint();
            // paint.setAntiAlias(true);
            paint.setTextSize(18);
            if (!threeDMode) {
                paint.setColor((!sharedTP.graphicInverseMode) ? graphicBackgroundColor
                        : graphicForegroundColor);
            } else {
                paint.setColor(color);
            }
            paint.setStyle(Paint.Style.FILL);
            String[] help = {"Hints:", "  One finger: Rotate",
                    "  Two fingers apart: Zoom",
                    "  Two fingers together: Translate"};
            int lineSize = (int) paint.getFontSpacing();
            int xpos = (int) ((xsize / 75) + threeDFactor * 200);
            int ypos = ysize - lineSize * help.length;

            for (String s : help) {
                canvas.drawText(s, xpos, ypos, paint);
                ypos += lineSize;
            }
        }
    }

    void perspective(CartesianPoint v, double threeDFactor) {
        v.y *= (graphicPerspectiveFactor + v.z) / graphicPerspectiveFactor;
        if (sharedTP.graphicAnaglyphMode) {
            v.x += v.z * threeDFactor;
        }
        v.x *= (graphicPerspectiveFactor + v.z) / graphicPerspectiveFactor;
    }

    int scalePoint(double center, double scale, double x) {
        return (int) ((x * scale) + center);
    }

    void rebuildModel() {
        imageGenHi.rebuildModel(sharedTP);
        imageGenLo.rebuildModel(sharedTP);
    }

    public void setDefaults() {
        graphicTransform.resetAll();
        rebuildModel();
    }

    protected void updateTankDescription() {
        rebuildModel();
        updateSettings();
    }

    void setTranslations() {
        graphicTransform.setXtranslate(sharedTP.modelTranslateX);
        graphicTransform.setYtranslate(sharedTP.modelTranslateY);
        graphicTransform.setZtranslate(sharedTP.modelTranslateZ);
    }

    void setRotations() {
        graphicTransform.setXAngle(sharedTP.modelAngleX);
        graphicTransform.setYAngle(sharedTP.modelAngleY);
    }

    public void updateSettings() {
        setTranslations();
        setRotations();
        invalidate();
    }

}
