// ***************************************************************************
// *   Copyright (C) 2018 by Paul Lutus                                      *
// *   lutusp@arachnoid.com                                                  *
// *                                                                         *
// *   This program is free software; you can redistribute it and/or modify  *
// *   it under the terms of the GNU General Public License as published by  *
// *   the Free Software Foundation; either version 2 of the License, or     *
// *   (at your option) any later version.                                   *
// *                                                                         *
// *   This program is distributed in the hope that it will be useful,       *
// *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
// *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
// *   GNU General Public License for more details.                          *
// *                                                                         *
// *   You should have received a copy of the GNU General Public License     *
// *   along with this program; if not, write to the                         *
// *   Free Software Foundation, Inc.,                                       *
// *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
// ***************************************************************************

package Arachnophilia;

/*
 * CommandSwitchboard.java
 *
 * Created on February 8, 2002, 3:35 PM
 */
import BaseCalc.*;
import BrowserStuff.*;
import CompilerReplyPanel.*;
import ConsoleWindow.*;
import FTPService.*;
import ListTableWizards.*;
import LookAndFeel.*;
import MacroManager.*;
import RTFConverter.*;
import SpellCheck.*;
import StyleColorHandler.*;
import TipOfTheDay.*;
import java.io.*;
import java.lang.reflect.*;
import java.text.DateFormat;
import java.util.*;
import javax.swing.*;

/**
 *
 * @author Administrator
 * @version
 */
final public class ComSwitchboard {

    HashMap<String,SwitchData> array;
    public String[] sortedList;
    public String[] sortedTips;
    Arachnophilia main;
    JComponent menuItem = null;

    /**
     * Creates new CommandSwitchboard
     */
    public ComSwitchboard(Arachnophilia m) {
        main = m;
        init();
    }

    public ComSwitchboard() {
        init();
    }

    class SwitchData {

        Class cls;
        String name;

        SwitchData(Class c, String s) {
            cls = c;
            name = s;
        }
    }

    private void init() {
        array = new HashMap<String,SwitchData>();
        String prefix = "class Arachnophilia.ComSwitchboard$";
        Class[] list = getClass().getDeclaredClasses();
        ArrayList<String> v = new ArrayList<String>();
        for (int i = 0; i < list.length; i++) {
            Class[] inter = list[i].getInterfaces();
            if (inter.length > 0) { // if this is a ComInterface
                String path = list[i].toString();
                //System.out.println(path);
                if (path.substring(0, prefix.length()).equals(prefix)) {
                    String className = path.substring(prefix.length());
                    v.add(className);
                    //System.out.println(className);
                    array.put(className, new SwitchData(list[i], className));
                }
            }
        }
        sortedList = v.toArray(new String[]{});
        Arrays.sort(sortedList);
        sortedTips = new String[sortedList.length];
        for (int i = 0; i < sortedList.length; i++) {
            String tip = "???";
            if (comExists(sortedList[i])) {
                tip = createInstance(sortedList[i]).getDescription();
            }
            int p = tip.indexOf('\n');
            if (p != -1) {
                tip = "<html>" + new SearchReplace().srchRplc(tip, "\n", "<br/>") + "</html>";
            }
            sortedTips[i] = tip;
        }
    }

    public String formatCommandList() {
        StringBuilder sb = new StringBuilder();
        String heading = "NOTE: Any of these system commands may be included in your own macros.\n" + "In fact, that is how the system menus and toolbars are created.\n\n";
        sb.append(heading);
        ArrayList<String> names = new ArrayList<String>();
        ArrayList<String> comments = new ArrayList<String>();
        int maxTab = 0;
        int len;
        String temp;
        for (int i = 0; i < sortedList.length; i++) {
            //System.out.println(sortedList[i]);
            ComInterface ci = createInstance(sortedList[i]);
            String comment = ci.getDescription();
            ArrayList<String> v = ArachComp.parseDelimLine(comment, "\n");
            String[] lines = v.toArray(new String[]{});
            len = lines.length;
            if (len > 0 && lines[0].length() > 0) {
                temp = "[" + sortedList[i] + "]";
                maxTab = ArachComp.getMax(temp, maxTab);
                names.add(temp);
                comments.add(lines[0]);
            }
            if (len > 1 && lines[1].length() > 0) {
                temp = "[" + sortedList[i] + ":arg]";
                maxTab = ArachComp.getMax(temp, maxTab);
                names.add(temp);
                comments.add(lines[1]);
            }
        }
        sb.append(ArachComp.doPostTab("Macro Tag", maxTab)).append("Comment\n");
        sb.append(ArachComp.dashedLine(70)).append("\n");
        len = names.size();
        for (int i = 0; i < len; i++) {
            sb.append(ArachComp.doPostTab( names.get(i), maxTab)).append( comments.get(i)).append("\n");
        }
        return sb.toString();
    }

    public ArrayList<String> getArrayListList() {
        ArrayList<String> v = new ArrayList<String>();
        for (int i = 0; i < sortedList.length; i++) {
            v.add("[" + sortedList[i] + "]");
        }
        return v;
    }

    private void test() {
        try {
            for (int i = 0; i < sortedList.length; i++) {
                ComInterface csw = createInstance(sortedList[i]);
                if (csw != null) {
                    //System.out.println(csw.getClass() + "," +  csw.getDescription());
                }
            }
        } catch (Exception e) {
            e.printStackTrace(System.out);
        }
    }

    private ComInterface createInstance(String s) {
        ComInterface csw = null;
        try {
            Class cc = ((SwitchData) array.get(s)).cls;
            Constructor[] co = cc.getConstructors();
            if (co.length == 0) {
                //System.out.println(cc + " has no default constructor!");
            } else {
                csw = (ComInterface) co[0].newInstance(new Object[]{this});
            }
        } catch (Exception e) {
            e.printStackTrace(System.out);
        }
        return csw;
    }

    private boolean comExists(String s) {
        return array.containsKey(s);
    }

    private boolean macroExists(String s) {
        MacroTreePanel panel = main.getMacroPanel();
        if (panel != null) {
            return panel.macroExists(s);
        }
        return false;
    }

    private String getMacroContent(String s) {
        return main.getMacroPanel().getMacroContent(s);
    }

    private int scanThroughBrackets(String s, int a, int b) {
        int r = b;
        int lvl = 0;
        int len = s.length();
        while (a < len) {
            char c = s.charAt(a);
            if (c == '[') {
                lvl++;
            } else if (c == ']') {
                lvl--;
                if (lvl == -1) {
                    r = a;
                    break;
                }
            }
            a++;
        }
        return r;
    }

    /*
     * exec(String s) looks for:
     * [comName:optionalArg]
     * in passed string.
     * If comName is found, it removes the entire tag and replaces it with
     * a result or places its result in the current document
     *
     **/
    public String exec(String s, int depth, JComponent item) {
        menuItem = item;
        String result = exec(s, depth);
        menuItem = null;
        return result;
    }

    public String exec(String s, int depth) {
        //System.out.println("exec [" + s + "]");
        if (depth > 16) {
            main.beep();
            JOptionPane.showMessageDialog(main, "There has been a stack overflow\n" + "in the macro processor. This is\n" + "basically caused by a macro\n" + "referring to itself in one way or another." + "The most recent macro content is:\n\n" + "\"" + s + "\"", "Macro Stack Overflow", JOptionPane.OK_OPTION);
            return s;
        }
        StringBuilder sb = new StringBuilder();
        int len = s.length();
        int origin = 0, a, b, c = -1;
        do {
            a = s.indexOf("[", origin);
            if (a != -1 && a < len - 1) {
                //System.out.println("open bracket found: " + a + "," + origin + "," + len);
                if ((a > 0 && s.charAt(a - 1) == '\\')) {
                    // bracket is escaped
                    //System.out.println("escaped bracket, appending [" + s.substring(origin,a-1) + "]");
                    sb.append(s.substring(origin, a - 1)).append('[');
                    origin = a + 1;
                    c = s.indexOf("]", origin);
                    if (c != -1) {
                        //System.out.println("escaped bracket, found close [" + s.substring(origin,c) + "]");
                        sb.append(s.substring(origin, c));
                        sb.append(']');
                        origin = c + 1;
                    }
                } else {
                    //System.out.println("normal path [" + s.substring(origin,a) + "]");
                    sb.append(s.substring(origin, a));
                    origin = a + 1;
                    // changed 6/10/2003
                    c = s.indexOf("]", a);
                    if (c != -1) {
                        //System.out.println("close bracket found: " + a + "," + origin + "," + len);
                        String argument = "";
                        String field;
                        b = s.indexOf(":", a);
                        // if there is an argument to this tag
                        if (b != -1 && b < c) {
                            // must handle nested brackets carefully
                            c = scanThroughBrackets(s, b, c);
                            if (c != -1) {
                                argument = s.substring(b + 1, c);
                                // look for recursive definitions
                                if (argument.length() > 0) {
                                    //System.out.println("recursive exec call 1: " + argument);
                                    argument = exec(argument, depth + 1);
                                }
                            }
                            field = s.substring(a + 1, b);
                        } else {
                            field = s.substring(a + 1, c);
                        }
                        String key = "[" + field + "]";
                        // look for a defined macro with this name
                        //System.out.println("looking for " + field);
                        if (macroExists(key)) {
                            //System.out.println("key exists: " + key);
                            String value = exec(getMacroContent(key), depth + 1);
                            sb.append(value);
                            origin = c + 1;
                        } // look for system commmand
                        else if (comExists(field)) {
                            //System.out.println("com exists: " + field);
                            ComInterface ci = createInstance(field);
                            if (ci != null) {
                                //System.out.println("com call: " + field + ":" + argument);
                                String value = ci.exec(argument);
                                sb.append(value);
                            }
                            origin = c + 1;
                        } else { // command not found
                            //System.out.println("com not found: " + field);
                            //c = a;
                            sb.append(s.substring(a, c));
                            sb.append(']');
                            origin = c + 1;
                        }
                    } else { // no closing bracket found
                        //.println("close bracket not found.");
                        sb.append(s.substring(origin, len));
                        origin = len;
                    }
                }
            }
        } while (a != -1 && a < len - 1 && c != -1 && origin < len);
        if (origin < len) {
            sb.append(s.substring(origin));
        }
        return sb.toString();
    }

    // ****** BEGIN ComInterface command-wrapper classes *********
    class AntiAliasing implements ComInterface {

        public AntiAliasing() {
        }
        String desc = "Toggle anti-aliasing feature (smooth font rendering).\n" + "Enable/disable anti-aliasing feature (smooth font rendering) (arg) =\"y\"/\"n\").";

        @Override
        public String exec(String arg) {
            if (arg.length() > 0) {
                char c = Character.toLowerCase(arg.charAt(0));
                main.configValues.useAntiAliasing = (c == 'y');
            } else {
                main.configValues.useAntiAliasing = !main.configValues.useAntiAliasing;
            }
            main.setAntiAliasing();
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class AutoIndent implements ComInterface {

        public AutoIndent() {
        }
        String desc = "Toggle auto-indent feature (new lines indented to prior lines).\nEnable/disable auto-indent (new lines indented to prior lines) for (arg) =\"y\"/\"n\").";

        @Override
        public String exec(String arg) {
            if (arg.length() > 0) {
                char c = Character.toLowerCase(arg.charAt(0));
                main.configValues.autoIndent = (c == 'y');
            } else {
                main.configValues.autoIndent = !main.configValues.autoIndent;
            }
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class BlockTab implements ComInterface {

        public BlockTab() {
        }
        String desc = "Tabs text block if a block is selected, otherwise acts as normal tab";

        @Override
        public String exec(String arg) {
            new DocumentProcessingFunctions(main).blockTab();
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class BlockReverseTab implements ComInterface {

        public BlockReverseTab() {
        }
        String desc = "Reverse-tabs text block if a block is selected, otherwise does nothing.";

        @Override
        public String exec(String arg) {
            new DocumentProcessingFunctions(main).blockReverseTab();
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class ClearSearchReplaceLists implements ComInterface {

        public ClearSearchReplaceLists() {
        }
        String desc = "Clears contents of search and replace dropdown lists";

        @Override
        public String exec(String arg) {
            main.findReplacePanel.clearSearchReplaceLists();
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class MoveToPriorDocument implements ComInterface {

        public MoveToPriorDocument() {
        }
        String desc = "Changes focus to prior document in tab sequence";

        @Override
        public String exec(String arg) {
            main.fileHandler.moveToPriorDocument();
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class MoveToNextDocument implements ComInterface {

        public MoveToNextDocument() {
        }
        String desc = "Changes focus to next document in tab sequence";

        @Override
        public String exec(String arg) {
            main.fileHandler.moveToNextDocument();
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class ConvertRTFToHTML implements ComInterface {

        public ConvertRTFToHTML() {
        }
        String desc = "Converts current RTF document to HTML";

        @Override
        public String exec(String arg) {
            RTFConverter rtf = new RTFConverter();
            rtf.convertRTFToHTML(main);
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class FileSaveEOL implements ComInterface {

        public FileSaveEOL() {
        }
        String desc = "With no argument, causes the file saving EOL character sequence to revert to this platform's default, which is " + EscapeUnescapeStringHandler.escapeString(ArachConstants.SYSTEM_EOL) + "\n" + "Causes the file saving EOL character sequence to be set to (arg), example entry: \\r\\n";

        @Override
        public String exec(String arg) {
            if (arg.length() > 0) {
                main.configValues.fileSaveEOL = arg;
            } else {
                main.configValues.fileSaveEOL = EscapeUnescapeStringHandler.escapeString(ArachConstants.SYSTEM_EOL);
            }
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class FileEncoding implements ComInterface {

        public FileEncoding() {
        }
        String desc = "With no argument, causes document read/write encoding to revert to the default (" + EscapeUnescapeStringHandler.escapeString(ArachConstants.DefaultEncoding) + ")\n" + "Causes document read/write encoding to be set to (arg), example entry: UTF8";

        @Override
        public String exec(String arg) {
            if (arg.length() > 0) {
                main.configValues.fileEncoding = arg;
            } else {
                main.configValues.fileEncoding = EscapeUnescapeStringHandler.escapeString(ArachConstants.DefaultEncoding);
            }
            main.updateStatusBar("");
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class AutoSave implements ComInterface {

        public AutoSave() {
        }
        public String desc = "Launches autosave configuration dialog\nAutomatlcally saves all changed files every (arg) seconds, 0 = no autosave";

        @Override
        public String exec(String arg) {
            if (arg.length() == 0) {
                TextInputDialog ti = new TextInputDialog(main);
                arg = ti.showDialog("Configure Autosave", "Choose a time interval for automatically saving changed documents, in seconds, currently " + main.configValues.autoSaveIntervalSeconds + " seconds, minimum valid autosave interval 15 seconds, 0 = disable autosave.", "" + main.configValues.autoSaveIntervalSeconds);
            }
            int as = 0;
            try {
                as = Integer.parseInt(arg);
            } catch (Exception e) {
            }
            if (as == 0 || as >= 15) {
                main.configValues.autoSaveIntervalSeconds = as;
                main.configureAutoSave();
            } else {
                main.beep();
            }
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class ShowTips implements ComInterface {

        public ShowTips() {
        }
        public String desc = "Show the Tip of the Day dialog";

        @Override
        public String exec(String arg) {
            TipJFrame tipJFrame = new TipJFrame(main);
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class ExitApp implements ComInterface {

        public ExitApp() {
        }
        public String desc = "Tries to save changed documents, quits Arachnophilia";

        @Override
        public String exec(String arg) {
            main.quit();
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class SetLookAndFeel implements ComInterface {

        public SetLookAndFeel() {
        }
        public String desc = "Launches Look & Feel selection dialog\n sets application look & feel based on (arg)";

        @Override
        public String exec(String arg) {
            LookAndFeelDialog lookAndFeelDialog = new LookAndFeelDialog(main, arg);
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class SetFileType implements ComInterface {

        public SetFileType() {
        }
        public String desc = "Launches selection dialog to choose file type for current document\nSets current document to file type name in (arg)";

        @Override
        public String exec(String arg) {
            new DocumentProcessingFunctions(main).setFileType(arg);
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class MakeUnorderedList implements ComInterface {

        public MakeUnorderedList() {
        }
        public String desc = "Turns selected lines into unordered list\nTurns selected lines into unordered list with options specified by (arg)";

        @Override
        public String exec(String arg) {
            new DocumentProcessingFunctions(main).makeUnorderedList(arg);
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class MakeOrderedList implements ComInterface {

        public MakeOrderedList() {
        }
        public String desc = "Turns selected lines into ordered list\nTurns selected lines into ordered list with options specified by (arg)";

        @Override
        public String exec(String arg) {
            new DocumentProcessingFunctions(main).makeOrderedList(arg);
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class LaunchBaseCalculator implements ComInterface {

        public LaunchBaseCalculator() {
        }
        public String desc = "Launch a simple number base calculator";

        @Override
        public String exec(String arg) {
            BaseFrame baseFrame = new BaseFrame(main);
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class ShowMacroEditorInFrame implements ComInterface {

        public ShowMacroEditorInFrame() {
        }
        public String desc = "\nSelect macro editor as (arg=\"n\") part of main application or (arg =\"y\") in separate frame";

        @Override
        public String exec(String arg) {
            if (arg.length() > 0) {
                char c = Character.toLowerCase(arg.charAt(0));
                main.configValues.showMacroEditorInFrame = (c == 'y');

            }
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class CloseCompilerWindow implements ComInterface {

        public CloseCompilerWindow() {
        }
        public String desc = "Close the window below the editing window that provides compiler error feedback";

        @Override
        public String exec(String arg) {
            main.showCompilerPanel(false, null);
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class RunCustomClassDoc implements ComInterface {

        public RunCustomClassDoc() {
        }
        public String desc = "\n(arg) = className. Processes current document through a user class named in (arg)";

        @Override
        public String exec(String arg) {
            new DocumentProcessingFunctions(main).RunCustomClassDoc(arg);
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class RunCustomClassArg implements ComInterface {

        public RunCustomClassArg() {
        }
        public String desc = "\n(arg) = (className,argument). Processes argument through user class (className)";

        @Override
        public String exec(String arg) {
            return new DocumentProcessingFunctions(main).RunCustomClassArg(arg);
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class SetSyntaxColors implements ComInterface {

        public SetSyntaxColors() {
        }
        public String desc = "Launches a dialog used to change the syntax-coloring display's colors";

        @Override
        public String exec(String arg) {
            StyleColorDialog styleColorDialog = new StyleColorDialog(main);
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class AboutDialog implements ComInterface {

        public AboutDialog() {
        }
        public String desc = "Launches the \"about\" dialog";

        @Override
        public String exec(String arg) {
            AboutBox aboutBox = new AboutBox(main, true);
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class LaunchHelp implements ComInterface {

        public LaunchHelp() {
        }
        public String desc = "Launches the help system";

        @Override
        public String exec(String arg) {
            // deal with another weird Netscape problem
            String path = "";
            if (main.basePath.length() > 0 && main.basePath.charAt(0) != '/') {
                path += '/';
            }
            StringBuilder sb = new StringBuilder();
            sb.append("file:").append(path).append(main.basePath).append("/");
            sb.append(Arachnophilia.docDirName).append("/index.html");
            main.launchBrowser(sb.toString());
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class UserHomeDirectory implements ComInterface {

        public UserHomeDirectory() {
        }
        public String desc = "Returns the user home directory, the location where Arachnophilia puts its configuration directory";

        @Override
        public String exec(String arg) {
            return main.userHome;
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class ApplicationDirectory implements ComInterface {

        public ApplicationDirectory() {
        }
        public String desc = "Returns the application directory, the location of the running application";

        @Override
        public String exec(String arg) {
            return main.jarPath;
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class SpellCheck implements ComInterface {

        public SpellCheck() {
        }
        public String desc = "Spell check document/selection";

        @Override
        public String exec(String arg) {
            if (main.spellCheckFrame == null) {
                main.spellCheckFrame = new SpellCheckFrame(main);
            } else {
                main.spellCheckFrame.spellCheck();
            }
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class JumpWordEnd implements ComInterface {

        public JumpWordEnd() {
        }
        String desc = "Jump to the end of the current word";

        @Override
        public String exec(String arg) {
            new DocumentProcessingFunctions(main).jumpTo(1, 0);
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class JumpLineEnd implements ComInterface {

        public JumpLineEnd() {
        }
        String desc = "Jump to the end of the current line";

        @Override
        public String exec(String arg) {
            new DocumentProcessingFunctions(main).jumpTo(1, 1);
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class JumpDocEnd implements ComInterface {

        public JumpDocEnd() {
        }
        String desc = "Jump to the end of the document";

        @Override
        public String exec(String arg) {
            new DocumentProcessingFunctions(main).jumpTo(1, 2);
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class JumpWordBegin implements ComInterface {

        public JumpWordBegin() {
        }
        String desc = "Jump to the beginning of the current word";

        @Override
        public String exec(String arg) {
            new DocumentProcessingFunctions(main).jumpTo(-1, 0);
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class JumpLineBegin implements ComInterface {

        public JumpLineBegin() {
        }
        String desc = "Jump to the beginning of the current line";

        @Override
        public String exec(String arg) {
            new DocumentProcessingFunctions(main).jumpTo(-1, 1);
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class JumpDocBegin implements ComInterface {

        public JumpDocBegin() {
        }
        String desc = "Jump to the beginning of the document";

        @Override
        public String exec(String arg) {
            new DocumentProcessingFunctions(main).jumpTo(-1, 2);
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class GotoLine implements ComInterface {

        public GotoLine() {
        }
        String desc = "Launch \"Go to line\" dialog\nGoes to line (arg)";

        @Override
        public String exec(String arg) {
            new DocumentProcessingFunctions(main).gotoLine(arg);
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class BlockOutdent implements ComInterface {

        public BlockOutdent() {
        }
        String desc = "Outdent document/selection one tab (or equivalent spaces)";

        @Override
        public String exec(String arg) {
            new DocumentProcessingFunctions(main).blockInOutDent(-1);
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class BlockIndent implements ComInterface {

        public BlockIndent() {
        }
        String desc = "Indent document/selection one tab (or equivalent spaces)";

        @Override
        public String exec(String arg) {
            new DocumentProcessingFunctions(main).blockInOutDent(1);
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class CreateFileLink implements ComInterface {

        public CreateFileLink() {
        }
        String desc = "\nCreate an HTML-style link using an arbitrary file type specified in (arg)";

        @Override
        public String exec(String arg) {
            if (arg.length() == 0) {
                main.beep();
                return "";
            } else {
                return new DocumentProcessingFunctions(main).createFileLink(arg);
            }
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class TableWizard implements ComInterface {

        public TableWizard() {
        }
        String desc = "Launch the table wizard";

        @Override
        public String exec(String arg) {
            if (main.tableWizard == null) {
                main.tableWizard = new TableWizardFrame(main);
            }
            main.tableWizard.launch();
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class BeautifyHTML implements ComInterface {

        public BeautifyHTML() {
        }
        String desc = "\"Beautify\" an HTML document -- indent it like program code";

        @Override
        public String exec(String arg) {
            new DocumentProcessingFunctions(main).beautifyHTML();
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class ValidateHTML implements ComInterface {

        public ValidateHTML() {
        }
        String desc = "Validate an HTML document's structure -- find tag errors";

        @Override
        public String exec(String arg) {
            new DocumentProcessingFunctions(main).validateHTML();
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class ReplaceLiTags implements ComInterface {

        public ReplaceLiTags() {
        }
        String desc = "Replace all closing </li> tags consistent with XHTML syntax rules";

        @Override
        public String exec(String arg) {
            new DocumentProcessingFunctions(main).replaceLiTags();
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class JumpToOppositeTag implements ComInterface {

        public JumpToOppositeTag() {
        }
        String desc = "Jump to partner HTML/XHTML tag from present position.";

        @Override
        public String exec(String arg) {
            new DocumentProcessingFunctions(main).jumpToOppositeTag();
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class RunGarbageCollector implements ComInterface {

        public RunGarbageCollector() {
        }
        String desc = "Run the Java garbage collector to free memory";

        @Override
        public String exec(String arg) {
            System.gc();
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class LaunchFTPService implements ComInterface {

        public LaunchFTPService() {
        }
        String desc = "Launch the Arachnophilia FTP service";

        @Override
        public String exec(String arg) {
            FTPFrame ftpFrame = new FTPFrame(main);
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class InsertFullGraphicTag implements ComInterface {

        public InsertFullGraphicTag() {
        }
        String desc = "Insert a full graphic \"img\" tag in the present document position, prompts for the desired file";

        @Override
        public String exec(String arg) {
            return new DocumentProcessingFunctions(main).insertFullGraphicTag();
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class InsertGraphicLink implements ComInterface {

        public InsertGraphicLink() {
        }
        String desc = "Insert a link to a graphic file in the present document position, prompts for the desired file";

        @Override
        public String exec(String arg) {
            return new DocumentProcessingFunctions(main).insertGraphicLink();
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class SizeGraphics implements ComInterface {

        public SizeGraphics() {
        }
        String desc = "compute sizes for all graphic file tags in document/selection";

        @Override
        public String exec(String arg) {
            new DocumentProcessingFunctions(main).sizeGraphics();
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class GetRescueMacroSet implements ComInterface {

        public GetRescueMacroSet() {
        }
        String desc = "Read the default command set from the Arachnophilia JAR file";

        @Override
        public String exec(String arg) {
            main.getRescueMacroSet();
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class DisplayEOLMarks implements ComInterface {

        public DisplayEOLMarks() {
        }
        String desc = "Toggle EOL marks in the syntax-colored display";

        @Override
        public String exec(String arg) {
            main.toggleEOLMarks();
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class DisplayLineWrap implements ComInterface {

        public DisplayLineWrap() {
        }
        String desc = "Toggle line wrapping in the plain-text display";

        @Override
        public String exec(String arg) {
            main.toggleLineWrap();
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class DisplayLineNumberColumn implements ComInterface {

        public DisplayLineNumberColumn() {
        }
        String desc = "Toggle the line number column in the syntax-colored display";

        @Override
        public String exec(String arg) {
            main.toggleLineNumbers();
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class SetEditFont implements ComInterface {

        public SetEditFont() {
        }
        String desc = "Set the program editor font type, style and size";

        @Override
        public String exec(String arg) {
            main.setEditFont();
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class SetProgramFont implements ComInterface {

        public SetProgramFont() {
        }
        String desc = "Set font tyle, style and size for entire program except programming editor display";

        @Override
        public String exec(String arg) {
            main.setProgramFont();
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class ClearFileList implements ComInterface {

        public ClearFileList() {
        }
        String desc = "Clear the recent file list";

        @Override
        public String exec(String arg) {
            main.recentFileList.clearList();
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class CloseAllDocs implements ComInterface {

        public CloseAllDocs() {
        }
        String desc = "Close all documents";

        @Override
        public String exec(String arg) {
            main.closeAllDocs();
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class CloseDoc implements ComInterface {

        public CloseDoc() {
        }
        String desc = "Close current document";

        @Override
        public String exec(String arg) {
            main.closeCurrentDoc();
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class LocateFileList implements ComInterface {

        public LocateFileList() {
        }
        String desc = "Identify the menu and position for the recent file list. This must be the last item on the menu";

        @Override
        public String exec(String arg) {
            if (menuItem != null) {
                main.recentFileList.setTarget(menuItem);
            }
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class BuildNewOpenMenus implements ComInterface {

        public BuildNewOpenMenus() {
        }
        String desc = "Build New/Open menu trees. This command is used only during menu construction";

        @Override
        public String exec(String arg) {
            if (menuItem != null) {
                ArachComp.buildNewOpenMenus(menuItem);
            }
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class UseSpacesForTabs implements ComInterface {

        public UseSpacesForTabs() {
        }
        String desc = "\nUse spaces for tabs in program functions if (arg) = \"y\", otherwise use tabs";

        @Override
        public String exec(String arg) {
            if (arg.length() > 0) {
                char c = Character.toLowerCase(arg.charAt(0));
                main.configValues.useSpacesForTabs = (c == 'y');
            }
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class SpacesToTabs implements ComInterface {

        public SpacesToTabs() {
        }
        String desc = "Convert document/selection spaces to tabs using current tab size as a guide";

        @Override
        public String exec(String arg) {
            new DocumentProcessingFunctions(main).spacesToTabs();
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class TabsToSpaces implements ComInterface {

        public TabsToSpaces() {
        }
        String desc = "Convert document/selection tabs to spaces using current tab size as a guide";

        @Override
        public String exec(String arg) {
            new DocumentProcessingFunctions(main).tabsToSpaces();
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class ConvertDocToHTML implements ComInterface {

        public ConvertDocToHTML() {
        }
        String desc = "Convert current syntax-highlighted document into HTML equivalent, put result in a new HTML document";

        @Override
        public String exec(String arg) {
            new DocumentProcessingFunctions(main).convertDocToHTML();
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class BeautifyCode implements ComInterface {

        public BeautifyCode() {
        }
        String desc = "\"Beautify\" document/selection program source code using rules based on file type";

        @Override
        public String exec(String arg) {
            new DocumentProcessingFunctions(main).beautifyCode();
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class GetSelection implements ComInterface {

        public GetSelection() {
        }
        String desc = "Get current document selection, insert it into running macro";

        @Override
        public String exec(String arg) {
            return new DocumentProcessingFunctions(main).getSelection();
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class InsertValueColor implements ComInterface {

        public InsertValueColor() {
        }
        String desc = "Launch color selector dialog, insert result\nLaunch color selector dialog using (arg) as user prompt (or default color hex code), insert result";

        @Override
        public String exec(String arg) {
            String color = new DocumentProcessingFunctions(main).launchColorSelectorDialog(arg);
            if (color == null) {
                color = "ff0000";
            }
            return color;
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class PageActiveLinkColor implements ComInterface {

        public PageActiveLinkColor() {
        }
        String desc = "Insert current HTML page active link color\nAccept hexadecimal code for active link color as arg";

        @Override
        public String exec(String arg) {
            if (arg.length() == 0) {
                return ArachComp.colorIntToString(main.configValues.pageActiveLinkColor);
            } else {
                main.configValues.pageActiveLinkColor = ArachComp.colorStringToInt(arg);
            }
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class PageVisitedLinkColor implements ComInterface {

        public PageVisitedLinkColor() {
        }
        String desc = "Insert current HTML page visited link color\nAccept hexadecimal code for visited link color as arg";

        @Override
        public String exec(String arg) {
            if (arg.length() == 0) {
                return ArachComp.colorIntToString(main.configValues.pageVisitedLinkColor);
            } else {
                main.configValues.pageVisitedLinkColor = ArachComp.colorStringToInt(arg);
            }
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class PageLinkColor implements ComInterface {

        public PageLinkColor() {
        }
        String desc = "Insert current HTML page link color\nAccept hexadecimal code for link color as arg";

        @Override
        public String exec(String arg) {
            if (arg.length() == 0) {
                return ArachComp.colorIntToString(main.configValues.pageLinkColor);
            } else {
                main.configValues.pageLinkColor = ArachComp.colorStringToInt(arg);
            }
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class PageTextColor implements ComInterface {

        public PageTextColor() {
        }
        String desc = "Insert current HTML page text color\nAccept hexadecimal code for text color as arg";

        @Override
        public String exec(String arg) {
            if (arg.length() == 0) {
                return ArachComp.colorIntToString(main.configValues.pageTextColor);
            } else {
                main.configValues.pageTextColor = ArachComp.colorStringToInt(arg);
            }
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class PageBackgroundColor implements ComInterface {

        public PageBackgroundColor() {
        }
        String desc = "Insert current HTML page background color\nAccept hexadecimal code for background color as arg";

        @Override
        public String exec(String arg) {
            if (arg.length() == 0) {
                return ArachComp.colorIntToString(main.configValues.pageBackgroundColor);
            } else {
                main.configValues.pageBackgroundColor = ArachComp.colorStringToInt(arg);
            }
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class ArachVersion implements ComInterface {

        public ArachVersion() {
        }
        String desc = "Insert the name and version of Arachnophilia";

        @Override
        public String exec(String arg) {
            return ArachConstants.APPNAME;
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class ProcessMacros implements ComInterface {

        public ProcessMacros() {
        }
        String desc = "Process document/selection as though it were a macro -- find and execute embedded system commands";

        @Override
        public String exec(String arg) {
            new DocumentProcessingFunctions(main).processMacrosInDocument();
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class ShowToolBarPanel implements ComInterface {

        public ShowToolBarPanel() {
        }
        String desc = "Change extended toolbar panel visibility\nShow none|some|all extended toolbar panels based on (arg)=\"0|1|2\"";

        @Override
        public String exec(String arg) {
            if (arg.length() > 0) {
                main.showToolBarPanel(Integer.parseInt(arg));
            } else {
                main.configValues.toolBarPanelVisibility = (main.configValues.toolBarPanelVisibility + 1) % 3;
                main.showToolBarPanel(main.configValues.toolBarPanelVisibility);
                main.getMacroPanel().mjt.update();
            }
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class BrowserTempFileName implements ComInterface {

        public BrowserTempFileName() {
        }
        String desc = "Insert the current browser temporary file name for the current document";

        @Override
        public String exec(String arg) {
            String result = "";
            if (main.currentSelectedDocument != null) {
                return main.currentSelectedDocument.getBrowserTempFileName();

            }
            return result;
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class UserInput implements ComInterface {

        public UserInput() {
        }
        String desc = "\nLaunch a dialog to accept user input with (arg) = prompt, insert the result";

        @Override
        public String exec(String arg) {
            return JOptionPane.showInputDialog(main, arg, ArachConstants.APPNAME + " User Input", JOptionPane.QUESTION_MESSAGE);
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class UndoRedoStackSize implements ComInterface {

        public UndoRedoStackSize() {
        }
        String desc = "Insert Undo/Redo stack size\nSpecify Undo/Redo stack size in (arg)";

        @Override
        public String exec(String arg) {
            if (arg.length() > 0) {
                main.configValues.undoRedoStackSize = Integer.parseInt(arg);
                return "";
            } else {
                return "" + main.configValues.undoRedoStackSize;
            }
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class ConfigureBrowsers implements ComInterface {

        public ConfigureBrowsers() {
        }
        String desc = "Locate and select Web browsers on your system";

        @Override
        public String exec(String arg) {
            BrowserDialog browserDialog = new BrowserDialog(main);
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class LaunchBrowser implements ComInterface {

        public LaunchBrowser() {
        }
        String desc = "Launch currently selected Web browser with temporary copy of current HTML document\nLaunch registered browser named (arg)/launch default browser with URL in (arg)";

        @Override
        public String exec(String arg) {
            main.launchBrowser(arg);
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class OpenDoc implements ComInterface {

        public OpenDoc() {
        }
        String desc = "Prompt to open document of the most recently opened type\nOpen document using (arg) path";

        @Override
        public String exec(String arg) {
            if (arg.length() > 0) {
                main.openDoc(new File(arg), true, true);
            } else {
                main.openDoc();
            }
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class OpenAllDoc implements ComInterface {

        public OpenAllDoc() {
        }
        String desc = "Prompt to open document with all files listed (*.*)";

        @Override
        public String exec(String arg) {
            // set "All" file type
            main.configValues.lastOpenedFileType = 0;
            main.openDoc();
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class NewDoc implements ComInterface {

        public NewDoc() {
        }
        String desc = "Create new document of the most recently created type\nCreate new document of (arg) type";

        @Override
        public String exec(String arg) {
            if (arg.length() > 0) {
                int type = main.fileTypes.getFileTypeForName(arg);
                if (type != -1) {
                    main.newDoc(type, true);
                } else {
                    main.beep();
                    JOptionPane.showMessageDialog(main, "The file type \"" + arg + "\" was not found.\n" + "Please be sure the submitted file type matches one\n" + "in the list of file types located at\n" + "\"" + main.basePath + "/ArachConf/FileData.txt.\"\n" + "These file types are case sensitive.", "New file by type", JOptionPane.OK_OPTION);
                }
            } else {
                main.newDoc();
            }
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class SystemConsole implements ComInterface {

        public SystemConsole() {
        }
        String desc = "\nExecute (arg) on local system with a console window. (arg) may include Arachnophilia system commands";

        @Override
        public String exec(String arg) {
            Process p;
            File f = null;
            if (arg.length() > 0) {
                if (main.currentSelectedDocument != null) {
                    f = new File(main.currentSelectedDocument.getPath());
                }
                try {
                    //System.out.println(arg);
                    p = Runtime.getRuntime().exec(arg, null, f);
                    ConsoleWindow consoleWindow = new ConsoleWindow(main, p);
                } catch (Exception e) {
                    sysErrorMessage(arg, e.getMessage());
                }
            }
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class SystemExec implements ComInterface {

        public SystemExec() {
        }
        String desc = "\nExecute (arg) on local system with a console window. Mostly used for compilers. (arg) may include Arachnophilia system commands";

        @Override
        public String exec(String arg) {
            Process p;
            File f = null;
            if (arg.length() > 0) {
                if (main.currentSelectedDocument != null) {
                    f = new File(main.currentSelectedDocument.getPath());
                }
                try {
                    //System.out.println(arg);
                    main.clearCompilerPanel();
                    p = Runtime.getRuntime().exec(arg, null, f);
                    ReplyPanel panel = new ReplyPanel(main);
                    panel.processData(p);

                } catch (IOException e) {
                    sysErrorMessage(arg, e.toString());
                }
            }
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class SystemCom implements ComInterface {

        public SystemCom() {
        }
        String desc = "\nInsert into current document the result of executing operating system commmand (arg), which may include Arachnophilia system commands";

        @Override
        public String exec(String arg) {
            new DocumentProcessingFunctions(main).execSystemCommand(arg, false);
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class SystemProcess implements ComInterface {

        public SystemProcess() {
        }
        String desc = "\nInsert into current document the result of executing operating system commmand (arg) on local system, with document/selection streamed to the process. (arg) may include Arachnophilia system commands";

        @Override
        public String exec(String arg) {
            new DocumentProcessingFunctions(main).execSystemCommand(arg, true);
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    private void sysErrorMessage(String arg, String error) {
        main.beep();
        JOptionPane.showMessageDialog(null,
                "Sorry!\n\nThe command string \"" + arg + "\"\ncaused your system to become petulant.\n" + "Here is the error message:\n\"" + error + "\"",
                "System Exec Error", JOptionPane.INFORMATION_MESSAGE);
    }

    class ListKeyMappings implements ComInterface {

        public ListKeyMappings() {
        }
        String desc = "Create a list of current keyboard mappings";

        @Override
        public String exec(String arg) {
            main.listKeyMappings();
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class ListSystemCommands implements ComInterface {

        public ListSystemCommands() {
        }
        String desc = "Create a list of all system commands";

        @Override
        public String exec(String arg) {
            main.listSystemCommands();
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class WordCount implements ComInterface {

        public WordCount() {
        }
        String desc = "Count words, lines, and characters in document/selection";

        @Override
        public String exec(String arg) {
            new DocumentProcessingFunctions(main).wordCount();
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class ToUpperCase implements ComInterface {

        public ToUpperCase() {
        }
        String desc = "Convert document/selection text to upper case";

        @Override
        public String exec(String arg) {
            new DocumentProcessingFunctions(main).toUpperCase();
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class ToLowerCase implements ComInterface {

        public ToLowerCase() {
        }
        String desc = "Convert document/selection text to lower case";

        @Override
        public String exec(String arg) {
            new DocumentProcessingFunctions(main).toLowerCase();
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    // A somewhat hackish way to set the focus to the Find input dialog
    class FindSetFocus implements ComInterface {

        public FindSetFocus() {
        }
        String desc =
                "Set system focus to find entry dialog";

        @Override
        public String exec(String arg) {
            new DocumentProcessingFunctions(main).setFindFocus();
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class FindReplaceAllAuto implements ComInterface {

        public FindReplaceAllAuto() {
        }
        String desc =
                "Find (using regular expressions if selected) and replace all cases using existing arguments\n" + "Find (using regular expressions if selected) and replace all cases using arguments provided as /find/replace/ (any delimiter)";

        @Override
        public String exec(String arg) {
            //main.findReplacePanel.setRegExp(false);
            if (arg.length() > 0) {
                char c = arg.charAt(0);
                ArrayList<String> v = ArachComp.parseDelimLine(arg, "" + c);
                if (v.size() > 2) {
                    main.findReplacePanel.getFindList().getEditor().setItem( v.get(1));
                    main.findReplacePanel.getReplaceList().getEditor().setItem( v.get(2));
                    main.findReplacePanel.doAllReplace();
                } else {
                    main.beep();
                }
            } else {
                main.findReplacePanel.doAllReplace();
            }
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class RegExpFindReplaceAll implements ComInterface {

        public RegExpFindReplaceAll() {
        }
        String desc =
                "Find (using regular expressions) and replace all cases using existing arguments\n" + "Find (using regular expressions) and replace all cases using arguments provided as /find/replace/ (any delimiter)";

        @Override
        public String exec(String arg) {
            main.findReplacePanel.setRegExp(true);
            if (arg.length() > 0) {
                char c = arg.charAt(0);
                ArrayList<String> v = ArachComp.parseDelimLine(arg, "" + c);
                if (v.size() > 2) {
                    main.findReplacePanel.getFindList().getEditor().setItem( v.get(1));
                    main.findReplacePanel.getReplaceList().getEditor().setItem( v.get(2));
                    main.findReplacePanel.doAllReplace();
                } else {
                    main.beep();
                }
            } else {
                main.findReplacePanel.doAllReplace();
            }
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class FindReplaceAll implements ComInterface {

        public FindReplaceAll() {
        }
        String desc =
                "Find (no regular expressions) and replace all cases using existing arguments\n" + "Find (no regular expressions) and replace all cases using arguments provided as /find/replace/ (any delimiter)";

        @Override
        public String exec(String arg) {
            main.findReplacePanel.setRegExp(false);
            if (arg.length() > 0) {
                char c = arg.charAt(0);
                ArrayList<String> v = ArachComp.parseDelimLine(arg, "" + c);
                if (v.size() > 2) {
                    main.findReplacePanel.getFindList().getEditor().setItem( v.get(1));
                    main.findReplacePanel.getReplaceList().getEditor().setItem( v.get(2));
                    main.findReplacePanel.doAllReplace();
                } else {
                    main.beep();
                }
            } else {
                main.findReplacePanel.doAllReplace();
            }
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class FindNextAuto implements ComInterface {

        public FindNextAuto() {
        }
        String desc =
                "Find (using regular expressions if selected) next case of existing search text\n" + "Find (using regular expressions if selected) next case of argument text";

        @Override
        public String exec(String arg) {
            //main.findReplacePanel.setRegExp(true);
            if (arg.length() > 0) {
                main.findNext(arg);
            } else {
                main.findNext(null);
            }
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class RegExpFindNext implements ComInterface {

        public RegExpFindNext() {
        }
        String desc =
                "Find (using regular expressions) next case of existing search text\n" + "Find (using regular expressions) next case of argument text";

        @Override
        public String exec(String arg) {
            main.findReplacePanel.setRegExp(true);
            if (arg.length() > 0) {
                main.findNext(arg);
            } else {
                main.findNext(null);
            }
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class FindNext implements ComInterface {

        public FindNext() {
        }
        String desc =
                "Find (no regular expressions) next case of existing search text\n" + "Find (no regular expressions) next case of argument text";

        @Override
        public String exec(String arg) {
            main.findReplacePanel.setRegExp(false);
            if (arg.length() > 0) {
                main.findNext(arg);
            } else {
                main.findNext(null);
            }
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class FindFirstAuto implements ComInterface {

        public FindFirstAuto() {
        }
        String desc =
                "Find (using regular expressions if selected) first case of existing search text\n" + "Find (using regular expressions if selected) first case of argument text";

        @Override
        public String exec(String arg) {
            //main.findReplacePanel.setRegExp(true);
            if (arg.length() > 0) {
                main.findFirst(arg);
            } else {
                main.findFirst(null);
            }
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class RegExpFindFirst implements ComInterface {

        public RegExpFindFirst() {
        }
        String desc =
                "Find (using regular expressions) first case of existing search text\n" + "Find (using regular expressions) first case of argument text";

        @Override
        public String exec(String arg) {
            main.findReplacePanel.setRegExp(true);
            if (arg.length() > 0) {
                main.findFirst(arg);
            } else {
                main.findFirst(null);
            }
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class FindFirst implements ComInterface {

        public FindFirst() {
        }
        String desc =
                "Find (no regular expressions) first case of existing search text\n" + "Find (no regular expressions) first case of argument text";

        @Override
        public String exec(String arg) {
            main.findReplacePanel.setRegExp(false);
            if (arg.length() > 0) {
                main.findFirst(arg);
            } else {
                main.findFirst(null);
            }
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class ReplaceAuto implements ComInterface {

        public ReplaceAuto() {
        }
        String desc =
                "Replace (using regular expressions if selected) next case of existing search text using existing replace text\n" + "Replace (using regular expressions if selected) next case of existing search text with argument";

        @Override
        public String exec(String arg) {
            //main.findReplacePanel.setRegExp(true);
            if (arg.length() > 0) {
                main.replace(arg);
            } else {
                main.replace(null);
            }
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class RegExpReplace implements ComInterface {

        public RegExpReplace() {
        }
        String desc =
                "Replace (using regular expressions) next case of existing search text using existing replace text\n" + "Replace (using regular expressions) next case of existing search text with argument";

        @Override
        public String exec(String arg) {
            main.findReplacePanel.setRegExp(true);
            if (arg.length() > 0) {
                main.replace(arg);
            } else {
                main.replace(null);
            }
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class Replace implements ComInterface {

        public Replace() {
        }
        String desc =
                "Replace (no regular expressions) next case of existing search text using existing replace text\n" + "Replace (no regular expressions) next case of existing search text with argument";

        @Override
        public String exec(String arg) {
            main.findReplacePanel.setRegExp(false);
            if (arg.length() > 0) {
                main.replace(arg);
            } else {
                main.replace(null);
            }
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class SaveAllChangedDocs implements ComInterface {

        public SaveAllChangedDocs() {
        }
        String desc = "Save all changed open documents";

        @Override
        public String exec(String arg) {
            main.saveAllChangedDocs();
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class SaveAs implements ComInterface {

        public SaveAs() {
        }
        String desc = "Save current document using dialog\nSave current document using provided path";

        @Override
        public String exec(String arg) {
            if (arg.length() > 0 && main.currentSelectedDocument != null) {
                main.currentSelectedDocument.saveFile(arg, true, false, false);
            } else {
                main.saveActiveDoc(true, false);
            }
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class Save implements ComInterface {

        public Save() {
        }
        String desc = "Save current document";

        @Override
        public String exec(String arg) {
            main.saveActiveDoc(false, false);
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class SetTabSize implements ComInterface {

        public SetTabSize() {
        }
        String desc = "Launch set tab size dialog\nSet tab size to argument";

        @Override
        public String exec(String arg) {
            if (arg.length() > 0) {
                main.setTabSize(Integer.parseInt(arg));
            } else {
                main.setTabSize(0);
            }
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class EditUndo implements ComInterface {

        public EditUndo() {
        }
        String desc = "Undo recent editing action";

        @Override
        public String exec(String arg) {
            main.undo();
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class EditRedo implements ComInterface {

        public EditRedo() {
        }
        String desc = "Redo recent undone editing action";

        @Override
        public String exec(String arg) {
            main.redo();
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class EditPaste implements ComInterface {

        public EditPaste() {
        }
        String desc = "Paste clipboard into current document";

        @Override
        public String exec(String arg) {
            main.editPaste();
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class EditCopy implements ComInterface {

        public EditCopy() {
        }
        String desc = "Copy selection in current document to clipboard";

        @Override
        public String exec(String arg) {
            main.editCopy();
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class EditCut implements ComInterface {

        public EditCut() {
        }
        String desc = "Cut selection in current document, put on clipboard";

        @Override
        public String exec(String arg) {
            main.editCut();
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class EditSelectAll implements ComInterface {

        public EditSelectAll() {
        }
        String desc = "Select all text in current document";

        @Override
        public String exec(String arg) {
            main.editSelectAll();
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class TextToHTML implements ComInterface {

        public TextToHTML() {
        }
        String desc = "Convert document/selection plain text to HTML (adds minimal appropriate tags)";

        @Override
        public String exec(String arg) {
            new DocumentProcessingFunctions(main).convertTextToHTML();
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class HTMLToText implements ComInterface {

        public HTMLToText() {
        }
        String desc = "Convert document/selection HTML to plain text (strips all tags, adds linefeeds as required)";

        @Override
        public String exec(String arg) {
            new DocumentProcessingFunctions(main).convertHTMLToText();
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class TagDelimsEscape implements ComInterface {

        public TagDelimsEscape() {
        }
        String desc = "Convert document/selection HTML tag symbols \"&lt;,&gt;\" to \"<,>\"";

        @Override
        public String exec(String arg) {
            new DocumentProcessingFunctions(main).tagDelimsEscape();
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class TagDelimsUnescape implements ComInterface {

        public TagDelimsUnescape() {
        }
        String desc = "Convert document/selection HTML tag symbols \"<,>\" to \"&lt;,&gt;\"";

        @Override
        public String exec(String arg) {
            new DocumentProcessingFunctions(main).tagDelimsUnescape();
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class LeftFlushLines implements ComInterface {

        public LeftFlushLines() {
        }
        String desc = "Remove document/selection leading tabs and spaces";

        @Override
        public String exec(String arg) {
            new DocumentProcessingFunctions(main).leftFlushLines();
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class CompressText implements ComInterface {

        public CompressText() {
        }
        String desc = "Convert document/selection individual lines into a paragraph";

        @Override
        public String exec(String arg) {
            new DocumentProcessingFunctions(main).compressText();
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class TagsUppercase implements ComInterface {

        public TagsUppercase() {
        }
        String desc = "Convert document/selection HTML tags to uppercase";

        @Override
        public String exec(String arg) {
            new DocumentProcessingFunctions(main).changeTagCase(true);
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class TagsLowercase implements ComInterface {

        public TagsLowercase() {
        }
        String desc = "Convert document/selection HTML tags to lowercase";

        @Override
        public String exec(String arg) {
            new DocumentProcessingFunctions(main).changeTagCase(false);
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class CvtExtChars implements ComInterface {

        public CvtExtChars() {
        }
        String desc = "Convert document/selection extended characters to HTML entities";

        @Override
        public String exec(String arg) {
            new DocumentProcessingFunctions(main).entityToString();
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class CvtHTMLChars implements ComInterface {

        public CvtHTMLChars() {
        }
        String desc = "Convert document/selection HTML entities to extended characters";

        @Override
        public String exec(String arg) {
            new DocumentProcessingFunctions(main).entityToChar();
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class EscapeText implements ComInterface {

        public EscapeText() {
        }
        String desc = "Escape document/selection text \" -> \\\" etc.";

        @Override
        public String exec(String arg) {
            new DocumentProcessingFunctions(main).escapeText();
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class UnescapeText implements ComInterface {

        public UnescapeText() {
        }
        String desc = "Unescape document/selection text \\\" -> \" etc.";

        @Override
        public String exec(String arg) {
            new DocumentProcessingFunctions(main).unescapeText();
            return "";
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class FullPath implements ComInterface {

        public FullPath() {
        }
        String desc = "Insert the full path and name of the current document";

        @Override
        public String exec(String arg) {
            String result = "";
            if (main.currentSelectedDocument != null) {
                result = main.currentSelectedDocument.getFullPath();
            }
            return result;
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class Pathname implements ComInterface {

        public Pathname() {
        }
        String desc = "Insert the directory path of the current document";

        @Override
        public String exec(String arg) {
            String result = "";
            if (main.currentSelectedDocument != null) {
                result = main.currentSelectedDocument.getPath();
            }
            return result;
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class Classname implements ComInterface {

        public Classname() {
        }
        String desc = "Insert the Java \"classname\" (the filename stripped of its suffix) into the current document";

        @Override
        public String exec(String arg) {
            String result = "";
            if (main.currentSelectedDocument != null) {
                result = main.currentSelectedDocument.name;
                int p = result.lastIndexOf('.');
                if (p != -1) {
                    result = result.substring(0, p);
                }
            }
            return result;
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class Filename implements ComInterface {

        public Filename() {
        }
        String desc = "Insert the filename of the current document";

        @Override
        public String exec(String arg) {
            String result = "";
            if (main.currentSelectedDocument != null) {
                result = main.currentSelectedDocument.name;
            }
            return result;
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    // this class name causes some confusion, but it is
    // the "right" name for the class, so be it
    class Date implements ComInterface {

        public Date() {
        }
        String desc = "Insert the current date using the default locale";

        @Override
        public String exec(String arg) {
            return DateFormat.getDateInstance().format(new java.util.Date());
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class Time implements ComInterface {

        public Time() {
        }
        String desc = "Insert the current time using the default locale";

        @Override
        public String exec(String arg) {
            return DateFormat.getTimeInstance().format(new java.util.Date());
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    class DateTime implements ComInterface {

        public DateTime() {
        }
        String desc = "Insert the current date and time using the default locale";

        @Override
        public String exec(String arg) {
            return DateFormat.getDateTimeInstance().format(new java.util.Date());
        }

        @Override
        public String getDescription() {
            return desc;
        }
    }

    // ********** END ComInterface classes **********
    // the ComInterface defintion
    public interface ComInterface {

        public String exec(String arg);

        public String getDescription();
    }
    /*
     public static void main(String args[]) {
     ComSwitchboard cs = new ComSwitchboard();
     cs.init();
     }*/
}
