// ***************************************************************************
// *   Copyright (C) 2018 by Paul Lutus                                      *
// *   lutusp@arachnoid.com                                                  *
// *                                                                         *
// *   This program is free software; you can redistribute it and/or modify  *
// *   it under the terms of the GNU General Public License as published by  *
// *   the Free Software Foundation; either version 2 of the License, or     *
// *   (at your option) any later version.                                   *
// *                                                                         *
// *   This program is distributed in the hope that it will be useful,       *
// *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
// *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
// *   GNU General Public License for more details.                          *
// *                                                                         *
// *   You should have received a copy of the GNU General Public License     *
// *   along with this program; if not, write to the                         *
// *   Free Software Foundation, Inc.,                                       *
// *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
// ***************************************************************************

package Arachnophilia;

/*
 * MacroKeyHandler.java
 *
 * Created on February 12, 2002, 11:48 AM
 */
import MacroManager.*;
import java.awt.event.*;
import java.util.*;
import javax.swing.*;
import javax.swing.tree.*;

/**
 *
 * @author  Administrator
 * @version
 */
final public class MacroKeyHandler {

    Arachnophilia main;
    HashMap<String,TreePath> keyMappings;
    ArrayList<String> errorList;
    static public final int POLICY_DO_NOTHING = 0;
    static public final int POLICY_ASK_USER = 1;
    static public final int POLICY_DELETE_FIRST = 2;
    static public final int POLICY_DELETE_SECOND = 3;
    int policy = POLICY_ASK_USER;

    /** Creates new MacroKeyHandler */
    public MacroKeyHandler(Arachnophilia m) {
        main = m;
    }

    public boolean execute(KeyEvent evt) {
        String k = KeyEvent.getKeyText(evt.getKeyCode());
        //System.out.println("MacroKeyHandler: " + k);

        boolean accept = false;
        String macroCode = createCode(evt);
        int id = evt.getID();
        if (id == KeyEvent.KEY_PRESSED) {
            if (keyMappings != null) {
                if (keyMappings.containsKey(macroCode)) {
                    evt.consume();
                    TreePath dest = getValue(macroCode);
                    executeMacro(dest);
                    //System.out.println("key macro: [" + macroCode + "]");
                    accept = true;
                }
            }
            if (!accept) {
                ArachDocument doc;
                if ((doc = main.currentSelectedDocument) != null) {
                    // allow "Undo" for some delete and backspace operations
                    int kcode = evt.getKeyCode();
                    String key = KeyEvent.getKeyText(kcode);
                   //System.out.println("key code: [" + key + "]");
                    if (kcode == KeyEvent.VK_BACK_SPACE || kcode == KeyEvent.VK_DELETE) {
                        String mod = KeyEvent.getKeyModifiersText(evt.getModifiers());
                        int kmod = evt.getModifiers();
                        int a = doc.textComp.getSelectionStart();
                        int b = doc.textComp.getSelectionEnd();
                        // if text is selected or Ctrl+Backspace or Ctrl+Delete
                        if ((a != b) || ((kmod & InputEvent.CTRL_MASK) != 0)) {
                            doc.undoPush();
                        }
                    }
                }
            }
        }
        return accept;
    }

    public String checkDefinition(
            String s) {
        String def = null;
        if (keyMappings.containsKey(s)) {
            def = getValue(s).toString();
        }

        return def;
    }

    public void removeKeyMacro(String path) {
        removeKeyMacro(getValue(path));
    }

    public void removeKeyMacro(TreePath path) {
        MacroTreeNodeData data = ArachComp.getDataForPath(path);
        if (data != null) {
            keyMappings.remove(data.keyboardHook);
            data.keyboardHook = null;
        }

    }

    private void executeMacro(TreePath t) {
        MacroTreeNodeData data = ArachComp.getDataForPath(t);
        if (data != null) {
            main.macroHandler.executeCommand(data.content);
        }

    }

    private TreePath getValue(String key) {
        if (keyMappings != null) {
            return keyMappings.get(key);
        } else {
            return null;
        }

    }

    public DefaultMutableTreeNode getNodeForString(
            String s) {
        return ArachComp.getNodeForPath(getValue(s));
    }

    public boolean readKeymacrosFromTree(DefaultMutableTreeNode node) {

        // throw away thwe old set
        keyMappings = new HashMap<String,TreePath>();
        errorList = new ArrayList<String>();
        policy =
                POLICY_ASK_USER;
        //System.out.println("read key macros from tree: " + policy);
        boolean isChanged = findDefs(node, false);
        // listDefs();
        return isChanged;
    }

    private boolean findDefs(DefaultMutableTreeNode node, boolean isChanged) {
        MacroTreeNodeData data = (MacroTreeNodeData) node.getUserObject();
        if (data != null && !data.isHidden) {
            if (data.keyboardHook != null) {
                if (data.keyboardHook.isValid()) {
                    //System.out.println("[" + data.keyboardHook + "]");
                    TreePath path = new TreePath(node.getPath());
                    String error;
// look for duplicate key mapping

                    if (keyMappings.containsKey(data.keyboardHook.toString())) {
                        //System.out.println("found dup: " + data.keyboardHook + "," + policy);
                        TreePath prior = getValue(data.keyboardHook.toString());
                        error =
                                ArachComp.doPostTab(data.keyboardHook.getDescription(), 8) + " mapped to " + ArachComp.doPostTab(prior.toString(), 16) + " and to " + path;
                        //System.out.println(error);
                        if (policy == POLICY_DO_NOTHING) {
                            errorList.add(error);
                            keyMappings.put(data.keyboardHook.toString(), path);
                        } else if (policy == POLICY_DELETE_FIRST) {
                            removeKeyMacro(prior);
                            keyMappings.put(data.keyboardHook.toString(), path);
                            isChanged =
                                    true;
                        } else if (policy == POLICY_DELETE_SECOND) {
                            removeKeyMacro(path);
                            isChanged =
                                    true;
                        } else if (policy == POLICY_ASK_USER) {
                            error =
                                    "This key code:    [" + data.keyboardHook.getDescription() + "]\n" + "is mapped both to " + prior.toString() + "\n" + "and to            " + path;
                            error =
                                    "There is a duplicate key mapping.\n\n" + error + "\n\nDo you want to delete the lower key mapping?\n\n\"Yes\" deletes the lower key mapping.\n\"No\" deletes the higher key mapping.\n\"Cancel\" ends interactive error checking\nand changes neither.\n\n";
                            int reply = JOptionPane.showConfirmDialog(main, error, "Key mapping scan -- duplicate definition", JOptionPane.YES_NO_CANCEL_OPTION);
                            if (reply == JOptionPane.YES_OPTION) {
                                removeKeyMacro(path);
                                isChanged =
                                        true;
                            } else if (reply == JOptionPane.NO_OPTION) {
                                removeKeyMacro(prior);
                                keyMappings.put(data.keyboardHook.toString(), path);
                                isChanged =
                                        true;
                            } else if (reply == JOptionPane.CANCEL_OPTION) {
                                policy = POLICY_DO_NOTHING;
                                keyMappings.put(data.keyboardHook.toString(), path);
                            }

                        }
                    } else {
                        keyMappings.put(data.keyboardHook.toString(), path);
                    }

                }
            }
            int len = node.getChildCount();
            for (int i = 0; i
                    < len; i++) {
                isChanged = findDefs((DefaultMutableTreeNode) node.getChildAt(i), isChanged);
            }

        }
        return isChanged;
    }

    public String listDefs() {
        ArrayList<TreePath> v = new ArrayList<>(keyMappings.values());
        ArrayList<MacroTreeNodeData> md = new ArrayList<>();
        for (int i = 0; i
                < v.size(); i++) {
            TreePath path = v.get(i);
            MacroTreeNodeData data = ArachComp.getDataForPath(path);
            md.add(data);
        }

        MacroTreeNodeData[] dataSet = null;
        dataSet = md.toArray(new MacroTreeNodeData[]{});
        //System.out.println("a");
        Arrays.sort(dataSet, new CompareMacroData());

        int tab = 0;
        // scan for longest string
        for (int i = 0; i
                < dataSet.length; i++) {
            tab = ArachComp.getMax(dataSet[i].keyboardHook.getDescription(), tab);
        }

        StringBuilder sb = new StringBuilder();
        String prologue =
                "NOTE: Any key mappings that are located within hidden subtrees\n" + "or nodes will not appear on this list, or be in force. To enable\n" + "a key mapping, make sure its node and all its parents are not hidden.\n\n";
        sb.append(prologue);
        String title = ArachComp.doPostTab("Key", tab) + " = Macro\n";
        sb.append(title);
        sb.append(ArachComp.dashedLine(70)).append("\n");
        for (int i = 0; i
                < dataSet.length; i++) {
                    sb.append(ArachComp.doPostTab(dataSet[i].keyboardHook.getDescription(), tab)).append(" = ").append(dataSet[i].name).append("\n");
        }

        return sb.toString();
    }

    public String createCode(
            KeyDescription d) {
        return d.toString();
    }

    public String createCode(
            KeyEvent evt) {
        return new KeyDescription(evt).toString();
    }

    class CompareMacroData implements Comparator<MacroTreeNodeData> {

        @Override
        public int compare(MacroTreeNodeData a, MacroTreeNodeData b) {
            MacroTreeNodeData da = (MacroTreeNodeData) a;
            MacroTreeNodeData db = (MacroTreeNodeData) b;
            return da.keyboardHook.getDescription().compareTo(db.keyboardHook.getDescription());
        }

        public boolean equals(java.lang.Object a, java.lang.Object b) {
            MacroTreeNodeData da = (MacroTreeNodeData) a;
            MacroTreeNodeData db = (MacroTreeNodeData) b;
            return da.keyboardHook.getDescription().equals(db.keyboardHook.getDescription());
        }
    }
}
