// ***************************************************************************
// *   Copyright (C) 2018 by Paul Lutus                                      *
// *   lutusp@arachnoid.com                                                  *
// *                                                                         *
// *   This program is free software; you can redistribute it and/or modify  *
// *   it under the terms of the GNU General Public License as published by  *
// *   the Free Software Foundation; either version 2 of the License, or     *
// *   (at your option) any later version.                                   *
// *                                                                         *
// *   This program is distributed in the hope that it will be useful,       *
// *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
// *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
// *   GNU General Public License for more details.                          *
// *                                                                         *
// *   You should have received a copy of the GNU General Public License     *
// *   along with this program; if not, write to the                         *
// *   Free Software Foundation, Inc.,                                       *
// *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
// ***************************************************************************

/*
 * MyPlainTextArea.java
 *
 * Created on December 15, 2001, 9:55 AM
 */
package Arachnophilia;

//import ArachComp;
//import MyJTextArea;
import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.plaf.basic.BasicBorders.MarginBorder;
import javax.swing.text.*;
import org.fife.ui.rsyntaxtextarea.*;
import org.fife.ui.rtextarea.*;

/**
 *
 * @author Administrator
 * @version
 */
final public class MySyntaxTextArea extends RSyntaxTextArea {

    Arachnophilia main;
    FontMetrics fm;
    int visibleLines;
    int viewportHeight;
    int lineHeight;
    ArachDocument theDoc;
    boolean keyCaptured = false;
    int needInit = 0;
    // EDIT
    RTextScrollPane scrollPane;

    public MySyntaxTextArea(Arachnophilia m, String s, ArachDocument d) {
        super(s);
        main = m;
        readNewColorValues();

        setBackground(SystemColor.text);
        setForeground(SystemColor.textText);
        theDoc = d;
        scrollPane = new RTextScrollPane();
        fm = getFontMetrics(getFont());

        MarginBorder mb = new MarginBorder() {
            @Override
            public Insets getBorderInsets(Component c, Insets in) {
                return (new Insets(0, 4, 0, 4));
            }
        ;
        };
        setBorder(mb);

        scrollPane.setBorder(null);

        Caret car = getCaret();
        if (car != null) {
            car.setSelectionVisible(true);
        }

        recalculateVisibleLines();

        scrollPane.addComponentListener(new java.awt.event.ComponentAdapter() {
            @Override
            public void componentResized(java.awt.event.ComponentEvent evt) {
                //System.out.println("component resized: " + getSize());
                recalculateVisibleLines();
                try {
                    scrollRectToVisible(modelToView(getCaretPosition()));
                } catch (Exception e) {
                }
            }
        });
        addKeyListener(new java.awt.event.KeyAdapter() {
            @Override
            public void keyPressed(java.awt.event.KeyEvent evt) {
                handlePressedKey(evt);
            }
        });
        addKeyListener(new java.awt.event.KeyAdapter() {
            @Override
            public void keyTyped(java.awt.event.KeyEvent evt) {
                handleTypedKey(evt);
            }
        });

        addKeyListener(new java.awt.event.KeyAdapter() {
            @Override
            public void keyReleased(java.awt.event.KeyEvent evt) {
                handleReleasedKey(evt);
            }
        });
        getDocument().addDocumentListener(new DocumentHandler());

        addFocusListener(new java.awt.event.FocusAdapter() {
            @Override
            public void focusGained(java.awt.event.FocusEvent evt) {
                theDoc.testLastModifiedTime();
                theDoc.setFocusable(true);
            }
        });

        addMouseWheelListener(
                new MouseWheelListener() {
                    @Override
                    public void mouseWheelMoved(MouseWheelEvent e) {
                        if (ctrlKey(e)) {
                            double zoom = (e.getUnitsToScroll() > 0) ? .90 : 1.10;
                            main.zoomEditFont(zoom);
                        } else {
                            e.getComponent().getParent().dispatchEvent(e);
                        }
                    }

                    public boolean ctrlKey(MouseWheelEvent e) {
                        return (e.getModifiersEx() & MouseWheelEvent.CTRL_DOWN_MASK) != 0;
                    }
                });
    }

    public void readNewColorValues() {
        getSyntaxScheme().styles = updateSyntaxColors();
        this.setSyntaxScheme(getSyntaxScheme());
    }

    public org.fife.ui.rsyntaxtextarea.Style[] updateSyntaxColors() {
        org.fife.ui.rsyntaxtextarea.Style[] styles = new org.fife.ui.rsyntaxtextarea.Style[Token.NUM_TOKEN_TYPES];
        styles[Token.COMMENT_EOL] = new MyStyle(main.configValues.STYLE_COMMENT);
        styles[Token.COMMENT_MULTILINE] = new MyStyle(main.configValues.STYLE_COMMENT);
        styles[Token.COMMENT_DOCUMENTATION] = new MyStyle(main.configValues.STYLE_DOC_COMMENT);
        styles[Token.RESERVED_WORD] = new MyStyle(main.configValues.STYLE_KEYWORD);
        styles[Token.FUNCTION] = new MyStyle(main.configValues.STYLE_FUNCTION);
        styles[Token.LITERAL_BOOLEAN] = new MyStyle(main.configValues.STYLE_LITERAL_NUMBER);
        styles[Token.LITERAL_NUMBER_DECIMAL_INT] = new MyStyle(main.configValues.STYLE_LITERAL_NUMBER);
        styles[Token.LITERAL_NUMBER_FLOAT] = new MyStyle(main.configValues.STYLE_LITERAL_NUMBER);
        styles[Token.LITERAL_NUMBER_HEXADECIMAL] = new MyStyle(main.configValues.STYLE_LITERAL_NUMBER);
        styles[Token.LITERAL_STRING_DOUBLE_QUOTE] = new MyStyle(main.configValues.STYLE_LITERAL_STRING);
        styles[Token.LITERAL_CHAR] = new MyStyle(main.configValues.STYLE_LITERAL_STRING);
        styles[Token.LITERAL_BACKQUOTE] = new MyStyle(main.configValues.STYLE_LITERAL_STRING);
        styles[Token.DATA_TYPE] = new MyStyle(main.configValues.STYLE_DATA);
        styles[Token.VARIABLE] = new MyStyle(main.configValues.STYLE_VARIABLE);
        styles[Token.IDENTIFIER] = new MyStyle(main.configValues.STYLE_IDENTIFIER);
        styles[Token.WHITESPACE] = new MyStyle(main.configValues.STYLE_WHITESPACE);
        styles[Token.SEPARATOR] = new MyStyle(main.configValues.STYLE_SEPARATOR);
        styles[Token.OPERATOR] = new MyStyle(main.configValues.STYLE_OPERATOR);
        styles[Token.PREPROCESSOR] = new MyStyle(main.configValues.STYLE_PREPROCESSOR);
        styles[Token.MARKUP_TAG_DELIMITER] = new MyStyle(main.configValues.STYLE_MARKUP_DELIM);
        styles[Token.MARKUP_TAG_NAME] = new MyStyle(main.configValues.STYLE_MARKUP_NAME);
        styles[Token.MARKUP_TAG_ATTRIBUTE] = new MyStyle(main.configValues.STYLE_MARKUP_ATTRIB);
        styles[Token.ERROR_IDENTIFIER] = new MyStyle(main.configValues.STYLE_ERROR);
        styles[Token.ERROR_NUMBER_FORMAT] = new MyStyle(main.configValues.STYLE_ERROR);
        styles[Token.ERROR_STRING_DOUBLE] = new MyStyle(main.configValues.STYLE_ERROR);
        styles[Token.ERROR_CHAR] = new MyStyle(main.configValues.STYLE_ERROR);
        return styles;
    }

    class MyStyle extends org.fife.ui.rsyntaxtextarea.Style {

        public MyStyle(SyntaxStyleData ssd) {
            StyleContext sc = StyleContext.getDefaultStyleContext();
            Font baseFont = getFont();
            int fs = ((ssd.italic) ? Font.ITALIC : 0) | ((ssd.bold) ? Font.BOLD : 0);
            font = sc.getFont(baseFont.getFamily(), fs,
                    baseFont.getSize());
            foreground = new Color(ssd.color);
        }
    }

    public void rfiw() {
            requestFocusInWindow();
    }

    // hook to key macro handler
    private void handlePressedKey(KeyEvent evt) {
        String key = KeyEvent.getKeyText(evt.getKeyCode());
        //System.out.println("MySyntaxTextArea: " + key + "," + keyCaptured);
        keyCaptured = false;
        if (!theDoc.handleLineFeed(evt)) {
            if (theDoc.main.macroKeyHandler.execute(evt)) {
                evt.consume();
                //System.out.println("MySyntaxTextArea: captured: " + key + "," + keyCaptured);
                keyCaptured = true;
            }
        }
        //String key = KeyEvent.getKeyText(evt.getKeyCode());
        //System.out.println("MySyntaxTextArea: " + key + "," + keyCaptured);
        //System.out.println("Mods: " + evt.getModifiers());
    }

    private void handleTypedKey(KeyEvent evt) {
        if (keyCaptured) {
            evt.consume();
        }
    }

    private void handleReleasedKey(KeyEvent evt) {

        if (keyCaptured) {
            evt.consume();
        }
        keyCaptured = false;
    }

    @Override
    public void setFont(Font f) {
        super.setFont(f);
        fm = getFontMetrics(f);
        recalculateVisibleLines();
    }

    public final void recalculateVisibleLines() {
        //System.out.println("recalculate visible lines");
        if (scrollPane != null) {
            JScrollBar vs = scrollPane.getVerticalScrollBar();
            viewportHeight = scrollPane.getViewport().getSize().height;
            lineHeight = getRowHeight();
            vs.setMaximum(getLineCount() * lineHeight);
            if (lineHeight != 0) {
                visibleLines = viewportHeight / lineHeight;
                if (visibleLines != 0) {
                    vs.setVisibleAmount(visibleLines * lineHeight);
                }
            }
        }
    }

    @Override
    public void paint(Graphics g) {
        Graphics2D g2 = (Graphics2D) g;
        g2.addRenderingHints(main.renderHints);
        /*Rectangle clipRect = g2.getClipBounds();
         g2.setColor(getBackground());
         g2.fillRect(clipRect.x,clipRect.y,horiz_inset,clipRect.height);
         clipRect.x += horiz_inset;
         clipRect.width -= horiz_inset;
         g2.translate(horiz_inset,0);
         //g2.setClip(clipRect); */
        super.paint(g2);
    }

    @Override
    public void paintComponent(Graphics g) {

        //System.out.println("plain-text painter repainting " + Math.random());
        super.paintComponent(g);
        if (needInit < 2) {
            centerCaretInView();
            needInit++;
            repaint();
        }

        theDoc.computeStatusString();
    }

    @Override
    public String getText(int offset, int length) {
        String s = "";
        try {
            s = getDocument().getText(offset, length);
        } catch (BadLocationException e) {
            e.printStackTrace(System.out);
        }
        return s;
    }

    @Override
    public void select(int a, int b) {
        if (theDoc != null) {
            if (theDoc.centerCaret) {
                centerCaretInView(a);
                theDoc.centerCaret = false;
            }
            super.select(a, b);
        }
    }

    public int getLength() {
        return getDocument().getLength();
    }

    public final int getVisibleLines() {
        return visibleLines;
    }

    @Override
    public final int getLineCount() {
        return getDocument().getDefaultRootElement().getElementCount();
    }

    public int getFirstLine() {
        //int p = getLineOfOffset(getCaretPosition());

        if (lineHeight != 0) {
            JScrollBar vs = scrollPane.getVerticalScrollBar();
            return scrollPane.getVerticalScrollBar().getValue() / lineHeight;
        } else {
            return 0;
        }
    }

    public void makeCaretVisible() {
    }

    public void setTokenMaker(String s) {
        setSyntaxEditingStyle(s);
    }

    public void centerCaretInView() {
        //System.out.println("center caret()");
        JScrollBar vs = scrollPane.getVerticalScrollBar();
        if (lineHeight == 0) {
            //System.out.println("recomputing");
            recalculateVisibleLines();
        }
        JScrollBar hs = scrollPane.getHorizontalScrollBar();
        try {
            int p = getCaretPosition();
            Rectangle r = modelToView(p);
            Dimension sz = scrollPane.getViewport().getSize();
            if (r != null && sz != null) {
                vs.setValue(r.y - (sz.height / 2));
                if (getLineWrap()) {
                    hs.setValue(0);
                } else {
                    hs.setValue(r.x - (sz.width / 2));
                }
            }
        } catch (Exception e) {
            e.printStackTrace(System.out);
        }
    }

    // position for future caret position
    public void centerCaretInView(int p) {
        //System.out.println("center caret(int)");
        if (scrollPane != null) {
            JScrollBar vs = scrollPane.getVerticalScrollBar();
            if (lineHeight == 0) {
                //System.out.println("recomputing");
                recalculateVisibleLines();
            }
            JScrollBar hs = scrollPane.getHorizontalScrollBar();
            try {
                Rectangle r = modelToView(p);
                Dimension sz = scrollPane.getViewport().getSize();
                if (r != null && sz != null) {
                    vs.setValue(r.y - (sz.height / 2));
                    if (getLineWrap()) {
                        hs.setValue(0);
                    } else {
                        hs.setValue(r.x - (sz.width / 2));
                    }
                }
            } catch (Exception e) {
                e.printStackTrace(System.out);
            }
        }
    }

    public void setFirstLine(int v) {

        //System.out.println("set first line:" + v);
        JScrollBar vs = scrollPane.getVerticalScrollBar();
        vs.setMaximum(getLineCount() * lineHeight);
        vs.setValue(v * lineHeight);
    }

    public final int getDocumentLength() {
        return getDocument().getLength();
    }

    public void centerEditPos() {
        Rectangle r;
        try {
            r = modelToView(getCaretPosition());
            if (r != null) {
                int gy = r.y - viewportHeight / 2;
                gy = (gy < 0) ? 0 : gy;
                scrollPane.getVerticalScrollBar().setValue(gy);
            }
        } catch (Exception e) {
            e.printStackTrace(System.out);
        }
    }

    @Override
    public final int getLineOfOffset(int offset) {
        return getDocument().getDefaultRootElement().getElementIndex(offset);
    }

    public final int getCaretLine() {
        return getLineOfOffset(getCaretPosition());
    }

    @Override
    public final int getLineStartOffset(int line) {
        int offset = -1;
        try {
            offset = super.getLineStartOffset(line);
        } catch (Exception e) {
            e.printStackTrace(System.out);
        }
        return offset;
    }

    public void addMouseListener(MouseAdapter m) {
        super.addMouseListener(m);
    }

    public int getLineLength(int line) {
        Element lineElement = getDocument().getDefaultRootElement().getElement(line);
        if (lineElement == null) {
            return -1;
        } else {
            return lineElement.getEndOffset()
                    - lineElement.getStartOffset() - 1;
        }
    }

    protected void documentChanged(DocumentEvent evt) {
        //System.out.println("document changed in mysyntaxtextarea");
        if (!theDoc.suppressFirst) {
            boolean oldChanged = theDoc.docChanged;
            theDoc.docChanged = true;
            if (oldChanged == false) {
                theDoc.updateTitle();
            }
        }
    }

    @Override
    public int viewToModel(Point p) {
        return super.viewToModel(p);
    }

    @Override
    public void setText(String s) {
        //System.out.println(s);
        try {
            //getDocument().beginCompoundEdit();
            getDocument().remove(0, getDocument().getLength());
            getDocument().insertString(0, s, null);
        } catch (BadLocationException bl) {
            bl.printStackTrace(System.out);
        } finally {
            //getDocument().endCompoundEdit();
        }
    }

    public String getLineText(int pos) {
        String result = "";
        try {
            int a = getLineStartOffset(pos);
            int b = getLineEndOffset(pos);
            result = getText(a, b - a - 1);
        } catch (Exception e) {
            e.printStackTrace(System.out);
        }
        return result;
    }

    /**
     * Deletes the selected text from the text area and places it into the
     * clipboard.
     */
    @Override
    public void cut() {
        copy();
        replaceSelection("");
    }

    /**
     * Places the selected text into the clipboard.
     */
    @Override
    public void copy() {
        if (getSelectionStart() != getSelectionEnd()) {
            String selection = getSelectedText();
            ClipboardSender clipboardSender = new ClipboardSender(selection);
        }
    }

    @Override
    public void paste() {
        ClipboardReceiver cr = new ClipboardReceiver();
        String selection = cr.getData();
        replaceSelection(selection);
    }

    class DocumentHandler implements DocumentListener {

        @Override
        public void insertUpdate(javax.swing.event.DocumentEvent evt) {
            documentChanged(evt);
        }

        @Override
        public void removeUpdate(javax.swing.event.DocumentEvent evt) {
            documentChanged(evt);
        }

        @Override
        public void changedUpdate(javax.swing.event.DocumentEvent evt) {
            documentChanged(evt);
        }
    }
}
