// ***************************************************************************
// *   Copyright (C) 2018 by Paul Lutus                                      *
// *   lutusp@arachnoid.com                                                  *
// *                                                                         *
// *   This program is free software; you can redistribute it and/or modify  *
// *   it under the terms of the GNU General Public License as published by  *
// *   the Free Software Foundation; either version 2 of the License, or     *
// *   (at your option) any later version.                                   *
// *                                                                         *
// *   This program is distributed in the hope that it will be useful,       *
// *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
// *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
// *   GNU General Public License for more details.                          *
// *                                                                         *
// *   You should have received a copy of the GNU General Public License     *
// *   along with this program; if not, write to the                         *
// *   Free Software Foundation, Inc.,                                       *
// *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
// ***************************************************************************

 /*
 * MacroFileReadWrite.java
 *
 * Created on February 5, 2002, 2:10 PM
 */
package MacroManager;

import Arachnophilia.*;
import java.util.*;
import javax.swing.*;
import javax.swing.tree.*;

/**
 *
 * @author  Administrator
 * @version
 */
final public class MacroFileReadWrite {

    Arachnophilia main;

    /** Creates new MacroFileReadWrite */
    public MacroFileReadWrite(Arachnophilia m) {
        main = m;
    }

    public boolean readLegacyMacroFile(String path, DefaultMutableTreeNode node) {
        boolean isChanged = false;
        LegacyToolbarConverter leg = new LegacyToolbarConverter();
        String s = leg.convertFile(path);
        if (s != null) {
            ArrayList<String> data = ArachComp.parseDelimLine(s, "\n", false);
            processReadXMLData(1, data, node);
            isChanged = true;
        }
        return isChanged;
    }

    public boolean readMacroFile(String path, DefaultMutableTreeNode node) {
        String data = ArachComp.readProgramFile(path, true);
        if (data != null) {
            if (data.indexOf("_DocPopup") == -1) {
                JOptionPane.showMessageDialog(main, "This system has an outdated program configuration. Please:\n\n   Exit Arachnophilia\n   Delete the directory located at " + main.basePath + "\n   Run Arachnophilia again\n\nThank you.", "Outdated Program Configuration", JOptionPane.OK_OPTION);
                System.exit(0);
            }
            return readMacroString(path, data, node);
        }
        return false;
    }

    public boolean readMacroString(String path, String dataString, DefaultMutableTreeNode node) {
        boolean isChanged = false;
        if (dataString != null) {
            ArrayList<String> data = ArachComp.parseDelimLine(dataString, "\n", true);
            if (data.size() > 0) {
                String s = ( data.get(0)).toLowerCase();
                if (s.indexOf("<?xml") != -1) {
                    isChanged = processReadXMLData(1, data, node);
                } else {
                    main.beep();
                    JOptionPane.showMessageDialog(null,
                            "File \"" + path + "\"\nis not a valid macro file.\n" + "If you are trying to read legacy Arachnophilia\n" + "(pre-4.0) toolbar files, use the \"Read legacy file\"\n" + "menu option.",
                            "Not Valid Macro File", JOptionPane.INFORMATION_MESSAGE);
                }
            }
        }
        return isChanged;
    }

    private boolean processReadXMLData(int i, ArrayList<String> data, DefaultMutableTreeNode node) {
        execReadXMLData(i, data, node);
        return true;
    }

    private int execReadXMLData(int i, ArrayList<String> data, DefaultMutableTreeNode node) {
        MacroTreeNodeData field = (MacroTreeNodeData) node.getUserObject();
        int len = data.size();
        while (i < len) {
            int type = readTag(field, data.get(i));
            i++;
            if (type == 1) {
                MacroTreeNodeData childField = new MacroTreeNodeData();
                DefaultMutableTreeNode child = new DefaultMutableTreeNode(childField);
                node.add(child);
                i = execReadXMLData(i, data, child);
                childField.name = ArachComp.getStringForNode(child);
            } else if (type == -1) {
                break;
            }
        }
        return i;
    }

    private int readTag(MacroTreeNodeData field, String s) {
        String tag = "";
        // tagTypes: -1=closing, 0=data, 1=opening
        if (s.indexOf("<?") != -1) {
            return 0;
        }
        int a = s.indexOf("<");
        int b = s.indexOf(">", a);
        int c = s.indexOf("</");
        if (a == -1) {
            return 0;
        }
        if (c != -1 && c < b) { // closing tag
            //tag = s.substring(c + 1, b);
            return -1;
        }
        if (b != -1 && a < b) {
            tag = s.substring(a + 1, b);
        }
        if (c != -1 && b < c) { // data
            if (field != null) {
                String val = s.substring(b + 1, c);
                //System.out.println("esc content: " + val);
                val = ArachComp.unescapeXMLContent(val);
                //System.out.println("unesc content: " + val);
                if (tag.equals("title")) {
                    // must trim these titles
                    field.title = val.trim();
                } else if (tag.equals("tooltip")) {
                    field.toolTip = val;
                } else if (tag.equals("keyboardhook")) {
                    field.keyboardHook = new KeyDescription(val);
                    if (!field.keyboardHook.isValid()) {
                        KeyStroke k = translateToAccelerator(val);
                        field.keyboardHook = new KeyDescription(k);
                    }
                } else if (tag.equals("content")) {
                    field.content = val;

                } else if (tag.equals("icon")) {
                    field.icon = val;
                    field.refreshIcon();
                } else if (tag.equals("toolbar")) {
                    field.isToolBar = true;
                    field.refreshIcon();
                } else if (tag.equals("menu")) {
                    field.isMenu = true;
                    field.refreshIcon();
                } else if (tag.equals("hidden")) {
                    field.isHidden = true;
                    field.refreshIcon();
                } else if (tag.equals("separator")) {
                    field.isSeparator = true;
                    field.refreshIcon();
                } else if (tag.equals("showtitleandicon")) {
                    field.showTitleAndIcon = true;
                    field.refreshIcon();
                }

            }
            return 0;
        }
        return 1; // opening tag
    }

    private KeyStroke translateToAccelerator(String q) {
        SearchReplace srch = new SearchReplace();
        q = srch.srchRplc(q, "Ctrl", "control");
        q = srch.srchRplc(q, "Alt", "alt");
        q = srch.srchRplc(q, "Shift", "shift");
        q = srch.srchRplc(q, "-", " ");
        q = srch.srchRplc(q, "+", " ");
        q = srch.srchRplc(q, "Enter", "ENTER");
        q = srch.srchRplc(q, "Space", "SPACE");
        return KeyStroke.getKeyStroke(q);
    }

    // writes entire tree
    public void writeRootMacroFile(String path, DefaultMutableTreeNode node) {
        if (node.getUserObject() != null) {
            writeMacroFileCore(path, node, 0);
        } else {
            main.beep();
        }
    }

    // writes only children of node
    public void writeMacroFile(String path, DefaultMutableTreeNode node) {
        writeMacroFileCore(path, node, -1);
    }

    // if level = -1, doesn't include root node
    private void writeMacroFileCore(String path, DefaultMutableTreeNode node, int level) {
        StringBuilder output = new StringBuilder();
        output.append(ArachConstants.xmlDecl + "\n");
        scanJTree(level, output, node);
        ArachComp.writeProgramFile(path, output.toString(), true);
    }

    private String makeTabs(int i) {
        StringBuilder s = new StringBuilder();
        while (i-- > 0) {
            s.append("  ");
        }
        return s.toString();
    }

    private void scanJTree(int level, StringBuilder output, DefaultMutableTreeNode node) {
        MacroTreeNodeData data = (MacroTreeNodeData) node.getUserObject();
        int count;
        String s;
        if ((count = node.getChildCount()) > 0) { // is a node
            if (level >= 0) {
                s = makeTagGroup(data, "node", level, false);
                output.append(s);
            }
            for (int i = 0; i < count; i++) {
                DefaultMutableTreeNode tn = (DefaultMutableTreeNode) node.getChildAt(i);
                scanJTree(level + 1, output, tn);
            }
            if (level >= 0) {
                s = makeTag(level, "node", false) + "\n";
                output.append(s);
            }
        } else if (level >= 0) { // is a leaf
            s = makeTagGroup(data, "leaf", level, true);
            output.append(s);
        }
    }

    private String makeTagGroup(MacroTreeNodeData data, String tag, int level, boolean closingTag) {
        StringBuilder sb = new StringBuilder();
        sb.append(makeTag(level, tag, true)).append("\n");
        sb.append(makeDataTag(level + 1, "title", data.title));
        sb.append(makeDataTag(level + 1, "tooltip", data.toolTip));
        String hook = "";
        if (data.keyboardHook != null && data.keyboardHook.isValid()) {
            hook = data.keyboardHook.toString();
        }
        sb.append(makeDataTag(level + 1, "keyboardhook", hook));
        sb.append(makeDataTag(level + 1, "icon", data.icon));
        sb.append(makeDataTag(level + 1, "content", data.content));
        sb.append(makeDataTag(level + 1, "toolbar", data.isToolBar));
        sb.append(makeDataTag(level + 1, "menu", data.isMenu));
        sb.append(makeDataTag(level + 1, "hidden", data.isHidden));
        sb.append(makeDataTag(level + 1, "separator", data.isSeparator));
        sb.append(makeDataTag(level + 1, "showtitleandicon", data.showTitleAndIcon));
        if (closingTag) {
            sb.append(makeTag(level, tag, false)).append("\n");
        }
        return sb.toString();
    }

    private String makeTag(int level, String tag, boolean opening) {
        return makeTabs(level) + "<" + ((opening) ? "" : "/") + tag + ">";
    }

    private String makeDataTag(int level, String tag, String value) {
        String result = "";
        if (value.length() > 0) {
            result = makeTabs(level) + "<" + tag + ">" + ArachComp.escapeXMLContent(value) + "</" + tag + ">\n";
        }
        return result;
    }

    private String makeDataTag(int level, String tag, boolean value) {
        String result = "";
        if (value) {
            result = makeDataTag(level, tag, "true");
        }
        return result;
    }
}
