// ***************************************************************************
// *   Copyright (C) 2018 by Paul Lutus                                      *
// *   lutusp@arachnoid.com                                                  *
// *                                                                         *
// *   This program is free software; you can redistribute it and/or modify  *
// *   it under the terms of the GNU General Public License as published by  *
// *   the Free Software Foundation; either version 2 of the License, or     *
// *   (at your option) any later version.                                   *
// *                                                                         *
// *   This program is distributed in the hope that it will be useful,       *
// *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
// *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
// *   GNU General Public License for more details.                          *
// *                                                                         *
// *   You should have received a copy of the GNU General Public License     *
// *   along with this program; if not, write to the                         *
// *   Free Software Foundation, Inc.,                                       *
// *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
// ***************************************************************************

 /*
 * MyJTree.java
 *
 * Created on February 5, 2002, 3:31 PM
 */

package MacroManager;

import Arachnophilia.*;
import FilePicker.*;
import FileTypes.*;
import java.awt.*;
import java.awt.datatransfer.*;
import java.awt.dnd.*;
import java.awt.event.*;
import java.io.*;
import java.util.*;
import javax.swing.*;
import javax.swing.tree.*;


/**
 *
 * @author  Administrator
 * @version
 */
final public class MyJTree extends JTree implements DropTargetListener {
    
    Arachnophilia main = null;
    Frame frame = null;
    MacroFileReadWrite macro_file_read_write;
    ToolBarManager toolBarManager;
    MenuManager menuManager;
    public TreeMap<String,DefaultMutableTreeNode> macroMap;
    
    MyRenderer renderer;
    private DefaultMutableTreeNode rootNode;
    private DefaultTreeModel localTreeModel;
    TreePath copyCutSelection = null;
    TreePath currentSelection = null;
    boolean isCut = false;
    public boolean isChanged = false;
    public boolean editMode = false;
    int tabSize;
    
    /** Creates new MyJTree */
    public MyJTree(Arachnophilia m,int t) {
        main = m;
        init(m,t);
    }
    /*
    public MyJTree(Frame f,int t) {
        init(f,t);
    }*/
    
    private void init(Frame f,int t) {
        frame = f;
        tabSize = t;
        setBorder(null);
        getSelectionModel().setSelectionMode
        (TreeSelectionModel.SINGLE_TREE_SELECTION);
        //setRootVisible(false);
        //setShowsRootHandles(true);
        // this must be set!
        setToolTipText("");
        addKeyListener(new java.awt.event.KeyAdapter() {
            @Override
            public void keyTyped(java.awt.event.KeyEvent evt) {
                handleKeys(evt);
            }
            @Override
            public void keyPressed(java.awt.event.KeyEvent evt) {
                handleKeys(evt);
            }
            @Override
            public void keyReleased(java.awt.event.KeyEvent evt) {
                handleKeys(evt);
            }
            
        });
        addMouseListener(new java.awt.event.MouseAdapter() {
            @Override
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                handleMouse(evt);
            }
        });
        addTreeSelectionListener(new javax.swing.event.TreeSelectionListener() {
            @Override
            public void valueChanged(javax.swing.event.TreeSelectionEvent evt) {
                handleTreeValueChanged(evt);
            }
        });
        DropTarget dropTarget = new DropTarget(this,
                                DnDConstants.ACTION_COPY_OR_MOVE,
                                this);
        
        macro_file_read_write = new MacroFileReadWrite(main);
        renderer = new MyRenderer();
        setupIcons(renderer);
        setCellRenderer(renderer);
        isChanged |= readRootMacroFile(main.macroPath);
        
        localTreeModel = (DefaultTreeModel)getModel();
        rootNode = (DefaultMutableTreeNode)localTreeModel.getRoot();
        toolBarManager = new ToolBarManager(main);
        menuManager = new MenuManager(main);
        update();
    }
    
    public void update() {
        updateAfterChange(rootNode,false,false);
    }
    
    @Override
     public void paint(Graphics g) {
        Graphics2D g2 = (Graphics2D) g;
        g2.addRenderingHints( main.renderHints );
        super.paint(g2);
    }
    
    // drag & drop code section
    
    @Override
    public void drop(java.awt.dnd.DropTargetDropEvent dtde) {
        Transferable trans = dtde.getTransferable();
        
        if( dtde.isDataFlavorSupported(DataFlavor.javaFileListFlavor) ) {
            try {
                dtde.acceptDrop(DnDConstants.ACTION_COPY);
                java.util.List<File> files =
                (java.util.List<File>)trans.getTransferData(DataFlavor.javaFileListFlavor);
                Iterator<File> iter = files.iterator();
                ArrayList<File> fl = new ArrayList<File>();
                while( iter.hasNext() ) {
                    File f = iter.next();
                    fl.add(f);
                    
                }
                dtde.dropComplete(true);
                openFileArray((File[])fl.toArray(new File[]{}));
            }
            catch( IOException e ) {
                //System.out.println(e);
            }
            catch( UnsupportedFlavorException e ) {
                //System.out.println(e);
            }
        }
    }
    
    public ArrayList<String> getMacroList() {
        return new ArrayList<String>(macroMap.keySet());
    }
    
    private void openFileArray(File[] array) {
        MacroTreeNodeData d = new MacroTreeNodeData("","New Macro Folder","Macro file(s) dropped onto Macro Window");
        d.isHidden = true;
        DefaultMutableTreeNode node = new DefaultMutableTreeNode(d);
        d.name = ArachComp.getStringForNode(node);
        rootNode.add(node);
        for(int i = 0;i < array.length;i++) {
            readMacroFile(array[i].getPath(),node,false);
        }
        updateAfterChange(rootNode,true,true);
    }
    
    // required additional methods
    
    @Override
    public void dragEnter(java.awt.dnd.DropTargetDragEvent e) { }
    @Override
    public void dragExit(java.awt.dnd.DropTargetEvent e) { }
    @Override
    public void dragOver(java.awt.dnd.DropTargetDragEvent e) {  }
    @Override
    public void dropActionChanged(java.awt.dnd.DropTargetDragEvent e) { }
    
    
    
    private void updateAfterChange(DefaultMutableTreeNode node,boolean refocus,boolean goToEditMode) {
        updateAfterChange(node,node,refocus,goToEditMode);
    }
    
    private void updateAfterChange(DefaultMutableTreeNode node,DefaultMutableTreeNode focusNode,boolean refocus,boolean goToEditMode) {
        main.macroKeyHandler.readKeymacrosFromTree(rootNode);
        toolBarManager.buildToolBars(rootNode);
        macroMap = menuManager.buildMenus(rootNode);
        localTreeModel.nodeStructureChanged(node);
        if(refocus && focusNode != null) {
            refocusNode(focusNode);
            if(goToEditMode) {
                editNode(frame,tabSize);
            }
        }
    }
    
    private void testSaveChanges() {
        if(main.macroEditor != null) {
            main.macroEditor.testSaveChanges();
        }
    }
    
    public void setEditMode(boolean mode,boolean force) {
        editMode = mode;
        //System.out.println("1:" + main.macroEditor.changed);
        if(mode) {
            if(main.macroEditor != null) {
                main.macroEditor.changed = false;
            }
            editNode(frame,tabSize);
            //System.out.println("2:" + main.macroEditor.changed);
        }
        else {
            testSaveChanges();
        }
        main.showMacroEditPanel(mode,force);
    }
    
    private void handleTreeValueChanged(javax.swing.event.TreeSelectionEvent evt) {
        if(editMode) {
            editNode(frame,tabSize);
        }
    }
    
    public void saveOnExit() {
        if(isChanged) {
            writeMacroFile(main.macroPath);
        }
    }
    
    private void handleMouse(java.awt.event.MouseEvent evt) {
        //System.out.println("MyJTree mouse event: " + evt);
        if(!evt.isPopupTrigger() && evt.getClickCount() == 2 || (main != null && main.configValues.oneClickMacroMode)) {
            TreePath selPath = getPathForLocation(evt.getX(), evt.getY());
            if(selPath != null) {
                setSelectionPath(selPath);
                //getCurrentSelection(false);
                DefaultMutableTreeNode node = (DefaultMutableTreeNode) selPath.getLastPathComponent();
                if(main != null) {
                    // don't execute node commands
                    if(node.getChildCount()== 0) {
                        String command = ((MacroTreeNodeData)node.getUserObject()).content;
                        main.macroHandler.executeCommand(command);
                    }
                }
            }
        }
    }
    
    private void handleKeys(java.awt.event.KeyEvent evt) {
        
        //evt.consume(); // always
        /*if(evt.getID() == KeyEvent.KEY_TYPED) {
            char c = evt.getKeyChar();
            if(c == '+' || c == '=') {
                moveDown();
                evt.consume();
            }
            else if(c == '-' || c == '_') {
                moveUp();
                evt.consume();
            }
         
        }*/
        if(evt.getID() == KeyEvent.KEY_PRESSED) {
            int k = evt.getKeyCode();
            char c = evt.getKeyChar();
            if(c == '+' || c == '=') {
                moveDown();
                evt.consume();
            }
            else if(c == '-' || c == '_') {
                moveUp();
                evt.consume();
            }
            else if(c == 'x' || c == 'X') {
                showHide();
                evt.consume();
            }
            else if(k == KeyEvent.VK_X) { // cut
                cutNode();
                evt.consume();
            }
            else if(k == KeyEvent.VK_C) { // copy
                copyNode();
                evt.consume();
            }
            else if(k == KeyEvent.VK_V) { // paste
                pasteNode();
                evt.consume();
            }
            else if(k == KeyEvent.VK_DELETE) { // delete
                deleteNode();
                evt.consume();
            }
            // this is now mapped to NewDoc
            
            /*else if(k == KeyEvent.VK_N) { // new
                newNode();
                evt.consume();
            }*/
            else {
                main.macroKeyHandler.execute(evt);
            }
        }
    }
    
    private void setupIcons(MyRenderer r) {
        Icon closed = new ImageIcon(getClass().getResource("/Icons/Folder.gif"));
        r.setClosedIcon(closed);
        Icon open = new ImageIcon(getClass().getResource("/Icons/Open.gif"));
        r.setOpenIcon(open);
        Icon leaf = new ImageIcon(getClass().getResource("/Icons/Document.gif"));
        r.setLeafIcon(leaf);
        setRowHeight(closed.getIconHeight());
    }
    
    public boolean readRootMacroFile(String path) {
        boolean localIsChanged;
        String toolTip = "Located on this system at " + main.macroPath;
        MacroTreeNodeData d = new MacroTreeNodeData("","Macros",toolTip);
        DefaultMutableTreeNode dm = new DefaultMutableTreeNode(d);
        d.name = ArachComp.getStringForNode(dm);
        localIsChanged = macro_file_read_write.readMacroFile(path,dm);
        setModel(new DefaultTreeModel(dm));
        return localIsChanged;
    }
    
    public void promptReadMacroFile(Arachnophilia m,boolean legacy) {
        DefaultMutableTreeNode node;
        if(getCurrentSelection()) {
            node = (DefaultMutableTreeNode) currentSelection.getLastPathComponent();
            FileTypes ft = new FileTypes(m.fileTypePath);
            ft.setFileType((legacy)?"Text":"XML");
            int type = ft.getFileType();
            String path = m.configValues.fileTypePaths[type];
            final PickerDialog fc;
            fc = new PickerDialog(m,new File(path),ft,PickerPanel.FILES_ONLY);
            //fc.setFileSelectionMode(PickerPanel.FILES_ONLY);
            fc.setDialogTitle("Read " + ft.fileTypeNames[type] + " macro file");
            int returnVal = fc.showOpenDialog();
            if (returnVal == PickerPanel.ACCEPT) {
                File[] flist = fc.getSelectedFiles();
                if(flist != null) {
                    for(int i = 0;i < flist.length;i++) {
                        if(readMacroFile(flist[i].getPath(),node,legacy)) {
                            isChanged = true;
                        }
                        m.configValues.fileTypePaths[type] = ArachComp.pathFromFullPath(flist[i].getPath());
                    }
                }
            }
        }
    }
    
    public void promptWriteMacroFile(Arachnophilia m) {
        DefaultMutableTreeNode node;
        if(getCurrentSelection()) {
            node = (DefaultMutableTreeNode) currentSelection.getLastPathComponent();
            boolean isChild = canBeSource(node,false);
            FileTypes ft = new FileTypes(m.fileTypePath);
            ft.setFileType("XML");
            int type = ft.getFileType();
            String path = m.configValues.fileTypePaths[type];
            final PickerDialog fc;
            fc = new PickerDialog(m,new File(path),ft,PickerPanel.FILES_ONLY);
            //fc.setFileSelectionMode(PickerPanel.FILES_ONLY);
            fc.setDialogTitle("Write " + ft.fileTypeNames[type] + " macro file");
            int returnVal = fc.showSaveDialog();
            if (returnVal == PickerPanel.ACCEPT) {
                File[] flist = fc.getSelectedFiles();
                if(flist != null) {
                    for(int i = 0;i < flist.length;i++) {
                        int reply = JOptionPane.OK_OPTION;
                        if(flist[i].exists()) {
                            reply = JOptionPane.showConfirmDialog(m,
                            "The file \n\"" + flist[i].getPath() + "\"\n exists.\nOkay to overwrite?",
                            "File Exists",JOptionPane.YES_NO_OPTION);
                        }
                        if(reply == JOptionPane.OK_OPTION) {
                            if(isChild) {
                                writeRootMacroFile(flist[i].getPath(),node);
                            }
                            else {
                                writeMacroFile(flist[i].getPath(),node);
                            }
                            m.configValues.fileTypePaths[type] = ArachComp.pathFromFullPath(flist[i].getPath());
                        }
                    }
                }
            }
        }
    }
    
    public boolean readMacroFile(String path,DefaultMutableTreeNode node, boolean legacy) {
        boolean localIsChanged;
        if(legacy) {
            localIsChanged = macro_file_read_write.readLegacyMacroFile(path,node);
        }
        else {
            localIsChanged = macro_file_read_write.readMacroFile(path,node);
        }
        localTreeModel.nodeStructureChanged(node);
        refocusNode(node);
        return localIsChanged;
    }
    
    public void readRescueMacroString(String path, String data) {
        MacroTreeNodeData v = new MacroTreeNodeData("Rescue Macros","A set of recovery macros","This set of macros can be used to repair a macro set that has been scrambled beyond repair.\n\nYou may replace your entire macro tree with this one, or you may choose to replace individual items or subtrees.");
        v.isHidden = true;
        DefaultMutableTreeNode node = new DefaultMutableTreeNode();
        rootNode.add(node);
        macro_file_read_write.readMacroString(path,data,node);
        node.setUserObject(v);
        isChanged = true;
        updateAfterChange(rootNode,true,true);
    }
    
    public void writeMacroFile(String path) {
        writeMacroFile(path,(DefaultMutableTreeNode) localTreeModel.getRoot());
    }
    
    public void writeMacroFile(String path,DefaultMutableTreeNode dm) {
        macro_file_read_write.writeMacroFile(path,dm);
    }
    
    public void writeRootMacroFile(String path,DefaultMutableTreeNode dm) {
        macro_file_read_write.writeRootMacroFile(path,dm);
    }
    
    class MyRenderer extends DefaultTreeCellRenderer {
        
        @Override
        public Component getTreeCellRendererComponent(
        JTree tree,
        Object value,
        boolean sel,
        boolean expanded,
        boolean leaf,
        int row,
        boolean hasFocus) {
            
            super.getTreeCellRendererComponent(
            tree, value, sel,
            expanded, leaf, row,
            hasFocus);
            DefaultMutableTreeNode tn = (DefaultMutableTreeNode)value;
            Object o = tn.getUserObject();
            if(o instanceof MacroTreeNodeData) {
                
                MacroTreeNodeData v = (MacroTreeNodeData) o;
                if(v.isHidden) {
                    setText("(" + v.title + ")");
                }
                else {
                    setText(v.title);
                }
                if(v.toolTip.length() > 0) {
                    setToolTipText(v.toolTip);
                }
                else {
                    if(v.keyboardHook != null) {
                        setToolTipText(v.keyboardHook.getDescription());
                    }
                }
                if(leaf) {
                    if(v.imageIcon != null) {
                        setIcon(v.imageIcon);
                    }
                    else {
                        setIcon(getLeafIcon());
                    }
                }
            }
            return this;
        }
    }
    
    public void selectOnRightClick(java.awt.event.MouseEvent evt) {
        //System.out.println(x + "," + y);
        TreePath path = getPathForLocation(evt.getX(),evt.getY());
        if (path != null) {
            setSelectionPath(path);
        }
    }
    
    
    
    // ***************** editing functions
    
    public void refocusNode(DefaultMutableTreeNode node) {
        if(node != null) {
            TreeNode[] tn = node.getPath();
            TreePath tp = new TreePath(tn);
            //expandPath(tp);
            setSelectionPath(tp);
        }
    }
    
    private boolean getCurrentSelection() {
        return getCurrentSelection(true);
    }
    
    private boolean canBeSource(DefaultMutableTreeNode node, boolean complain) {
        boolean accept = (node != null && node.getParent() != null);
        if(!accept && complain) {
            main.beep();
        }
        return accept;
    }
    
    private boolean getCurrentSelection(boolean complain) {
        currentSelection = getSelectionPath();
        if(currentSelection == null && complain) {
            main.beep();
        }
        return currentSelection != null;
    }
    
    public void cutNode() {
        if(getCurrentSelection()) {
            DefaultMutableTreeNode node = (DefaultMutableTreeNode) currentSelection.getLastPathComponent();
            if(canBeSource(node,true)) {
                copyCutSelection = currentSelection;
                isCut = true;
            }
        }
    }
    
    public void copyNode() {
        if(getCurrentSelection()) {
            DefaultMutableTreeNode node = (DefaultMutableTreeNode) currentSelection.getLastPathComponent();
            if(canBeSource(node,true)) {
                copyCutSelection = currentSelection;
                isCut = false;
            }
        }
    }
    
    public void pasteNode() {
        if(getCurrentSelection()) {
            if(copyCutSelection != null) {
                DefaultMutableTreeNode source = (DefaultMutableTreeNode) copyCutSelection.getLastPathComponent();
                DefaultMutableTreeNode dest = (DefaultMutableTreeNode) currentSelection.getLastPathComponent();
                if(source != null && dest != null) {
                    DefaultMutableTreeNode clone = cloneNode(source);
                    if(clone != null) {
                        dest.add(clone);
                        
                        if(isCut) {
                            removeFromParent(source);
                        }
                        isChanged = true;
                        localTreeModel.nodeStructureChanged(source);
                        updateAfterChange(dest,true,true);
                        
                    }
                }
            }
            else {
                main.beep();
            }
        }
        copyCutSelection = null;
    }
    
    private DefaultMutableTreeNode cloneNode(DefaultMutableTreeNode node) {
        DefaultMutableTreeNode newNode = null;
        MacroTreeNodeData md = (MacroTreeNodeData)node.getUserObject();
        if(md != null) {
            newNode =
            new DefaultMutableTreeNode(md.clone());
            int len = node.getChildCount();
            for(int i=0;i<len; i++) {
                newNode.add(cloneNode(
                (DefaultMutableTreeNode)node.getChildAt(i) ) );
            }
        }
        return newNode;
    }
    
    
    private DefaultMutableTreeNode removeFromParent(DefaultMutableTreeNode node) {
        DefaultMutableTreeNode sibling = null;
        DefaultMutableTreeNode parent = (DefaultMutableTreeNode) node.getParent();
        if(parent != null) {
            
            parent.remove(node);
            if(parent.getChildCount() > 0) {
                sibling = (DefaultMutableTreeNode) parent.getFirstChild();
            }
            else {
                sibling = parent;
            }
            //System.out.println(sibling);
            isChanged = true;
            localTreeModel.nodeChanged(parent);
        }
        else {
            main.beep();
        }
        return sibling;
    }
    
    public void deleteNode() {
        if(getCurrentSelection()) {
            DefaultMutableTreeNode source = (DefaultMutableTreeNode) currentSelection.getLastPathComponent();
            if(canBeSource(source,true)) {
                main.beep();
                Container c = ArachComp.getFrameParent(this);
                TreePath path = new TreePath(source.getPath());
                int reply = JOptionPane.showConfirmDialog(c,
                "Warning: you are about to delete\n\"" + path + "\"\nand any children it may have.\nThis action cannot be undone.\nProceed?",
                "Macro Tree deletion",JOptionPane.YES_NO_OPTION);
                if(reply == JOptionPane.OK_OPTION) {
                    DefaultMutableTreeNode sibling = removeFromParent(source);
                    //System.out.println(sibling);
                    isChanged = true;
                    updateAfterChange(source,sibling,true,true);
                    //main.macroKeyHandler.readKeymacrosFromTree(rootNode);
                }
            }
            else {
                main.beep();
            }
        }
    }
    
    public void moveUp() {
        swapNodes(-1);
    }
    
    public void moveDown() {
        swapNodes(1);
    }
    
    private void swapNodes(int offset) {
        if(getCurrentSelection()) {
            DefaultMutableTreeNode node = (DefaultMutableTreeNode) currentSelection.getLastPathComponent();
            DefaultMutableTreeNode parent = (DefaultMutableTreeNode) node.getParent();
            if(parent != null) {
                int count = parent.getChildCount();
                int i = parent.getIndex(node);
                // allow wrap-around but no negative numbers
                int j = (i + offset + count) % count;
                DefaultMutableTreeNode sibling = (DefaultMutableTreeNode) parent.getChildAt(j);
                parent.insert(sibling,i);
                parent.insert(node,j);
                isChanged = true;
                updateAfterChange(parent,false,false);
                refocusNode(node);
            }
        }
    }
    
    public void editNode() {
        editNode(frame,tabSize);
    }
    
    public void editNode(Frame f,int tab) {
        if(getCurrentSelection(false)) {
            DefaultMutableTreeNode node = (DefaultMutableTreeNode) currentSelection.getLastPathComponent();
            MacroTreeNodeData data = (MacroTreeNodeData) node.getUserObject();
            //System.out.println(data.debugString());
            //MacroEditorDialog me = new MacroEditorDialog(main,tab,true,node,data);
            //me.show();
            //if(me.editor.validData) {
            //MacroEditorDialog me = new MacroEditorDialog(main,tab,true,node,data);
            main.showMacroEditPanel(true,false);
            //main.showMacroEditor();
            main.macroEditor.edit(this,node);
            isChanged = true;
            /*if(main.macroEditor.validData) {
                //System.out.println(me.data.debugString());
                node.setUserObject(me.editor.data);
                treeModel.nodeChanged(node);
                isChanged = true;
                updateAfterChange(node,true);
                //main.macroKeyHandler.readKeymacrosFromTree(rootNode);
            }*/
        }
    }
    
    public void acceptEdit(DefaultMutableTreeNode target) {
        localTreeModel.nodeChanged(target);
        isChanged = true;
        updateAfterChange(target,false,false);
    }
    
    public boolean itemVisible() {
        if(getCurrentSelection()) {
            DefaultMutableTreeNode node = (DefaultMutableTreeNode) currentSelection.getLastPathComponent();
            MacroTreeNodeData data = (MacroTreeNodeData) node.getUserObject();
            return !data.isHidden;
        }
        return false;
    }
    
    public void showHide() {
        if(getCurrentSelection()) {
            DefaultMutableTreeNode node = (DefaultMutableTreeNode) currentSelection.getLastPathComponent();
            if(node.getParent() != null) {
                MacroTreeNodeData data = (MacroTreeNodeData) node.getUserObject();
                data.isHidden = !data.isHidden;
                localTreeModel.nodeChanged(node);
                isChanged = true;
                updateAfterChange(node,true,false);
            }
        }
    }
    
    public void newNode() {
        if(getCurrentSelection()) {
            MacroTreeNodeData mt = new MacroTreeNodeData("","New Item","");
            DefaultMutableTreeNode node = new DefaultMutableTreeNode(mt);
            DefaultMutableTreeNode parent = (DefaultMutableTreeNode) currentSelection.getLastPathComponent();
            parent.add(node);
            mt.name = ArachComp.getStringForNode(node);
            isChanged = true;
            localTreeModel.nodeStructureChanged(parent);
            refocusNode(node);
            main.getMacroPanel().setEditMode(true,true);
        }
    }
    
    public void sortNormal() {
        sort(false);
    }
    
    public void sortReverse() {
        sort(true);
    }
    
    private void sort(boolean reverse) {
        int reply = JOptionPane.showConfirmDialog(main,
        "Sorting a long list of alphabetic commands may make\n"
        +"sense, but you cannot undo a sort, and if you sort\n"
        + "a toolbar, you may soon wish you hadn't.\n"
        + "Okay to continue?",
        "Sort Macro List",JOptionPane.YES_NO_OPTION);
        
        if(reply == JOptionPane.OK_OPTION) {
            if(getCurrentSelection()) {
                DefaultMutableTreeNode node = (DefaultMutableTreeNode) currentSelection.getLastPathComponent();
                
                int len = node.getChildCount();
                ArrayList<DefaultMutableTreeNode> v = new ArrayList<DefaultMutableTreeNode>();
                
                for(int i = 0;i < len;i++) {
                    v.add((DefaultMutableTreeNode)node.getChildAt(i));
                }
                Collections.sort(v,new MyComp(reverse));
                node.removeAllChildren();
                for(int j = 0;j < len;j++) {
                    node.add(v.get(j));
                }
                localTreeModel.nodeStructureChanged(node);
                isChanged = true;
                updateAfterChange(node,true,true);
            }
        }
    }
    
    
    
    class MyComp implements Comparator<DefaultMutableTreeNode> {
        boolean sortReverse;
        public MyComp(boolean r) {
            sortReverse = r;
        }
        @Override
        public int compare(DefaultMutableTreeNode oa, DefaultMutableTreeNode ob) {
            MacroTreeNodeData a = (MacroTreeNodeData) oa.getUserObject();
            MacroTreeNodeData b = (MacroTreeNodeData) ob.getUserObject();
            return (sortReverse)?b.title.compareTo(a.title):a.title.compareTo(b.title);
        }
        
        
    }
}