/***************************************************************************
 *   Copyright (C) 2016 by Paul Lutus                                      *
 *   http://arachnoid.com/administration                                   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

/*
 * Purpose: to formalize the reading and writing
 * of data files consisting of flat tables
 * of various data types including primitives
 *
 * on one side there is a tab-delimited file
 * containing a header with field names
 * and records of data
 * on the other side is a vector of data objects
 *
 * The goal is to have a way to create a data file from any
 * object consisting of simple variables including primitives
 * or to read a data file and populate a vector with
 * initialized instantiations of the desired object
 *
 *
 */

import java.lang.reflect.*;
import java.io.*;
import java.util.*;
import java.awt.*;
import javax.swing.*;


final public class DataFileManager {
    
    /** Creates a new instance of DataFileManager */
    final static String lineSep = System.getProperty("line.separator");
    final static private String delim = "\t";
    //public DataFileManager() {
    //}
    
    static public void writeObjectToFile(String path, Object ob,String fieldPrefix) {
        
        try {
            FileWriter fw = new FileWriter(path);
            if(!testFileAccess(new File(path))) return;
            Field[] publicFields = ob.getClass().getFields();
            String[] fieldNames = getFieldNames(publicFields,fieldPrefix);
            String[] headerNames = getHeaderNames(fieldNames,fieldPrefix);
            for(int i = 0;i < fieldNames.length;i++) {
                Field f = ob.getClass().getDeclaredField(fieldNames[i]);
                String row = headerNames[i] + "=" + getFieldValueAsString(f,ob);
                fw.write(row);
                fw.write(lineSep);
            }
            fw.close();
        }
        catch(Exception e) {
            e.printStackTrace();
        }
    }
    
    static public boolean testFileAccess(File f) {
        boolean ok = false;
        if(!f.exists()) f = new File(f.getParent());
        boolean read = f.canRead();
        boolean write = f.canWrite();
        ok = read && write;
        if(!ok) {
            String error = "";
            if(!read && !write) {
                error = "read or write";
            }
            else if (!read) {
                error = "read";
            }
            else {
                error = "write";
            }
            String message = "This program cannot " + error + " one or more of its essential configuration files.\n\n"
            + "Please correct this file permission problem and try again.";
            JOptionPane.showMessageDialog(null,message,"File/Directory Access error.",JOptionPane.WARNING_MESSAGE);
                
        }
        return ok;
    }
    
    static public Vector readObjectFromFile(String path,Object ob,String fieldPrefix) {
        Vector data = new Vector();
        try {
            File f = new File(path);
            if(!testFileAccess(f)) return data;
            if(f.exists()) {
                BufferedReader br = new BufferedReader(new FileReader(path));
                String line;
                while((line = br.readLine()) != null) {
                    int p = line.indexOf("=");
                    if(p != -1) {
                        String name = fieldPrefix + line.substring(0,p);
                        String value = line.substring(p+1);
                        parseSingleValue(ob,name,value);
                    }
                }
                br.close();
            }
        }
        catch (Exception e) {
            e.printStackTrace();
        }
        return data;
    }
    
    // get header data from first object
    
    static public void writeDataTable(String path, Vector data,Object obj,String fieldPrefix) {
        try {
            if(!testFileAccess(new File(path))) return;
            FileWriter fw = new FileWriter(path);
            //Field[] publicFields = data.get(0).getClass().getFields();
            Field[] publicFields = obj.getClass().getFields();
            String[] fieldNames = getFieldNames(publicFields,fieldPrefix);
            String[] headerNames = getHeaderNames(fieldNames,fieldPrefix);
            fw.write(join(headerNames,delim)); // header
            fw.write(lineSep);
            for(int i = 0;i < data.size();i++) {
                Object ob = data.get(i);
                String[] record = createRecord(ob,fieldNames);
                fw.write(join(record,delim));
                fw.write(lineSep);
            }
            fw.close();
        }
        catch(Exception e) {
            e.printStackTrace();
        }
    }
    
    static public void writeDataTable(String path, TreeMap data,Object obj,String fieldPrefix) {
        try {
            String[] keys = (String[]) data.keySet().toArray(new String[]{});
            if(!testFileAccess(new File(path))) return;
            FileWriter fw = new FileWriter(path);
            Field[] publicFields = obj.getClass().getFields();
            String[] fieldNames = getFieldNames(publicFields,fieldPrefix);
            String[] headerNames = getHeaderNames(fieldNames,fieldPrefix);
            fw.write(join(headerNames,delim)); // header
            fw.write(lineSep);
            for(int i = 0;i < keys.length;i++) {
                Object ob = data.get(keys[i]);
                String[] record = createRecord(ob,fieldNames);
                fw.write(join(record,delim));
                fw.write(lineSep);
            }
            fw.close();
        }
        catch(Exception e) {
            e.printStackTrace();
        }
    }
    
    static private String[] createRecord(Object ob,String[] fieldNames) {
        String[] result = new String[fieldNames.length];
        try {
            for(int i = 0;i < fieldNames.length;i++) {
                Field f = ob.getClass().getDeclaredField(fieldNames[i]);
                result[i] = getFieldValueAsString(f,ob);
            }
        }
        catch (Exception e) {
            e.printStackTrace();
        }
        return result;
    }
    
    // this routine only translates
    // "true" = "1" and
    // "false" = "0"
    // to prevent excessive file length
    // if it were not for this,
    // String.valueOf(f.get(ob))
    // would be acceptable in all cases
    
    static private String getFieldValueAsString(Field f,Object ob) {
        String result = "";
        try {
            if(f.getType().isAssignableFrom(boolean.class)) {
                result = (((Boolean)f.get(ob)).booleanValue())?"1":"0";
            }
            else if(f.getType().isAssignableFrom(Boolean.class)) {
                result = (((Boolean)f.get(ob)).booleanValue())?"1":"0";
            }
            else {
                result = String.valueOf(f.get(ob));
            }
        }
        catch (Exception e) {
            e.printStackTrace();
        }
        return result;
    }
    
    // get header data from first file line
    
    static public boolean readDataTable(String path,Vector data,Object o,String fieldPrefix) {
        boolean wasRead = false;
        try {
            File f = new File(path);
            if(!testFileAccess(f)) return false;
            if(f.exists()) {
                wasRead = true;
                BufferedReader br = new BufferedReader(new FileReader(path));
                String line = br.readLine(); // toss header
                String[] headerNames = split(line,delim);
                String[] fieldNames = makeFieldNames(headerNames,fieldPrefix);
                while((line = br.readLine()) != null) {
                    String[] values = split(line,delim);
                    Object ob = o.getClass().newInstance();
                    parseRecordValues(ob,fieldNames,values);
                    data.add(ob);
                }
                br.close();
            }
        }
        catch (Exception e) {
            e.printStackTrace();
        }
        return wasRead;
    }
    
    static public void readDataTable(String path,TreeMap data,Object obj,String fieldPrefix) {
        try {
            File f = new File(path);
            if(!testFileAccess(f)) return;
            if(f.exists()) {
                BufferedReader br = new BufferedReader(new FileReader(path));
                String line = br.readLine(); // toss header
                String[] headerNames = split(line,delim);
                String[] fieldNames = makeFieldNames(headerNames,fieldPrefix);
                while((line = br.readLine()) != null) {
                    String[] values = split(line,delim);
                    Object ob = obj.getClass().newInstance();
                    parseRecordValues(ob,fieldNames,values);
                    data.put(values[0],ob);
                }
                br.close();
            }
        }
        catch (Exception e) {
            e.printStackTrace();
        }
    }
    
    static private String[] makeFieldNames(String[] headerNames, String fieldPrefix) {
        if(headerNames == null) return null;
        String[] result = new String[headerNames.length];
        for(int i = 0;i < headerNames.length;i++) {
            result[i] = fieldPrefix + headerNames[i];
        }
        return result;
    }
    
    static private void parseRecordValues(Object ob, String[] fieldNames, String[] values) {
        for(int i = 0;i < fieldNames.length;i++) {
            String name = fieldNames[i];
            String value = values[i];
            parseSingleValue(ob,name,value);
        }
    }
    
    static void parseSingleValue(Object ob, String name, String value) {
        try {
            Field f = ob.getClass().getDeclaredField(name);
            Class fieldClass = f.getType();
            if(fieldClass.isAssignableFrom(String.class)) {
                f.set(ob,value);
            }
            else if(fieldClass.isAssignableFrom(Boolean.class)) {
                f.set(ob,new Boolean(value.equals("1")));
            }
            else if(fieldClass.isAssignableFrom(boolean.class)) {
                f.set(ob,new Boolean(value.equals("1")));
                //System.out.println("boolean: " + f.get(ob));
            }
            else if(fieldClass.isAssignableFrom(Integer.class)) {
                f.set(ob,new Integer(value));
                //System.out.println("int: " + f.get(ob));
            }
            else if(fieldClass.isAssignableFrom(int.class)) {
                f.set(ob,new Integer(value));
                //System.out.println("int: " + f.get(ob));
            }
            else if(fieldClass.isAssignableFrom(Long.class)) {
                f.set(ob,new Long(value));
                //System.out.println("long: " + f.get(ob));
            }
            else if(fieldClass.isAssignableFrom(long.class)) {
                f.set(ob,new Long(value));
                //System.out.println("long: " + f.get(ob));
            }
            else if(fieldClass.isAssignableFrom(Double.class)) {
                f.set(ob,new Double(value));
                //System.out.println("double: " + f.get(ob));
            }
            else if(fieldClass.isAssignableFrom(double.class)) {
                f.set(ob,new Double(value));
                //System.out.println("double: " + f.get(ob));
            }
            else if(fieldClass.isAssignableFrom(Color.class)) {
                // format 3 integer values into tabbed row
                // "\D" = non-numerical digit
                // "\d" = numerical digit
                String s = value.replaceFirst(
                "\\D+(\\d+)\\D+(\\d+)\\D+(\\d+)\\D+","$1\t$2\t$3");
                // split them into array
                String[] val = s.split("\t");
                // use them to create color
                f.set(ob,new Color(
                Integer.parseInt(val[0]),
                Integer.parseInt(val[1]),
                Integer.parseInt(val[2])
                ));
            }
            else if(fieldClass.isAssignableFrom(Dimension.class)) {
                // format 3 integer values into tabbed row
                // "\D" = non-numerical digit
                // "\d" = numerical digit
                String s = value.replaceFirst(
                "\\D+(\\d+)\\D+(\\d+)\\D+","$1\t$2");
                // split them into array
                String[] val = s.split("\t");
                // use them to create color
                f.set(ob,new Dimension(
                Integer.parseInt(val[0]),
                Integer.parseInt(val[1])
                ));
            }
            else if(fieldClass.isAssignableFrom(Point.class)) {
                // format 3 integer values into tabbed row
                // "\D" = non-numerical digit
                // "\d" = numerical digit
                String s = value.replaceFirst(
                "\\D+(\\d+)\\D+(\\d+)\\D+","$1\t$2");
                // split them into array
                String[] val = s.split("\t");
                // use them to create color
                f.set(ob,new Point(
                Integer.parseInt(val[0]),
                Integer.parseInt(val[1])
                ));
            }
            else if(fieldClass.isAssignableFrom(Rectangle.class)) {
                // format 3 integer values into tabbed row
                // "\D" = non-numerical digit
                // "\d" = numerical digit
                String s = value.replaceFirst(
                "\\D+(\\d+)\\D+(\\d+)\\D+(\\d+)\\D+(\\d+)\\D+","$1\t$2\t$3\t$4");
                // split them into array
                String[] val = s.split("\t");
                // use them to create color
                f.set(ob,new Rectangle(
                Integer.parseInt(val[0]),
                Integer.parseInt(val[1]),
                Integer.parseInt(val[2]),
                Integer.parseInt(val[3])
                ));
            }
            else {
                //System.out.println("cannot parse field [" + name + "], value [" + value + "], class [" + fieldClass + "]");
            }
        }
        catch(Exception e) {
            //e.printStackTrace();
        }
    }
    
    static public String[] getFieldNames(Field[] fields,String fieldPrefix) {
        Vector result = new Vector();
        for(int i = 0;i < fields.length;i++) {
            String name = fields[i].getName();
            if(name.indexOf(fieldPrefix) == 0) {
                result.add(name);
            }
        }
        return (String[]) result.toArray(new String[result.size()]);
    }
    
    static public String[] getHeaderNames(String[] fieldNames, String fieldPrefix) {
        String[] result = new String[fieldNames.length];
        for(int i = 0;i < fieldNames.length;i++) {
            result[i] = fieldNames[i].substring(fieldPrefix.length());
        }
        return result;
    }
    
    static public String join(String[] data, String delim) {
        StringBuffer sb = new StringBuffer();
        for(int i = 0;i < data.length;i++) {
            sb.append(data[i]);
            if(i < data.length-1) {
                sb.append(delim);
            }
        }
        return sb.toString();
    }
    
    // essential for records with empty fields
    
    static public String[] split(String data,String token) {
        if(data == null) return null;
        Vector v = new Vector();
        int a = 0,b;
        int tlen = token.length();
        while((b = data.indexOf(token,a)) != -1) {
            v.add(data.substring(a,b));
            a = b + tlen;
        }
        v.add(data.substring(a));
        return (String[]) v.toArray(new String[v.size()]);
    }
    
}
