/***************************************************************************
 *   Copyright (C) 2016 by Paul Lutus                                      *
 *   http://arachnoid.com/administration                                   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

import java.io.*;
import java.util.*;
import java.net.*;
import javax.swing.*;

final public class LocationHandler {

    SatFinder parent;
    String zipPath = "ZipDivisions";
    String statePath = "States";
    String keyFileName = "keys.txt";
    TreeMap states = null;
    TreeMap zipDivs = null;

    /** Creates a new instance of LocationHandler */
    public LocationHandler(SatFinder p) {
        parent = p;
        states = new TreeMap();
        zipDivs = new TreeMap();
    }

    public void buildDataFiles() {
        parent.showProgressBar(true);
        
        final File f = new File(parent.zipPath);
        Thread t1 = new Thread() {

            public void run() {
                readZipCodesThread(f);
            }
        };
        t1.start();
    }

    private void readZipCodesThread(File path) {
        TreeMap t_states = new TreeMap();
        TreeMap t_zipDivs = new TreeMap();
        HashMap t_cities = new HashMap();
        BufferedReader br = null;
        try {
            //f.mkdirs();
            URL url = parent.getClass().getResource(parent.zipFileName);
            InputStream is = url.openStream();
            br = new BufferedReader(new InputStreamReader(is));
            int max = is.available();
            int len = 0;
            String line;
            while ((line = br.readLine()) != null) {
                len += line.length() + 1;
                parent.updateProgressBar("Reading ZIP data file", 0, max, len);
                String[] array = line.split("\\|");
                if (array.length == 6) {
                    parseZipRecord(line, array, t_states, t_zipDivs, t_cities);
                } else {
                    throw new Exception("Malformed record in ZIP data file \"" + url + "\": \"" + line + "\"");
                }
            }
            createDataFiles(path, t_states, t_zipDivs);
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            try {
                if (br != null) {
                    br.close();
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
        parent.updateProgressBar("", 0, 0, 0);
        parent.showProgressBar(false);
        parent.setup();
    }

    private void parseZipRecord(String line, String[] array, TreeMap t_states, TreeMap t_zipDivs, HashMap t_cities) {
        ZipRecord zr = new ZipRecord(array);
        if (!t_states.containsKey(zr.state)) {
            t_states.put(zr.state, new TreeMap());
        }
        String cskey = zr.state + "." + zr.city;
        if (!t_cities.containsKey(cskey)) {
            TreeMap state = (TreeMap) t_states.get(zr.state);
            state.put(zr.zip, line);
            t_cities.put(cskey, true);
        }
        int z = Integer.parseInt(zr.zip);
        String zipKey = "" + (z / 1000);
        if (!t_zipDivs.containsKey(zipKey)) {
            t_zipDivs.put(zipKey, new TreeMap());
        }
        TreeMap zd = (TreeMap) t_zipDivs.get(zipKey);
        zd.put(zr.zip, zr.toString());
    }

    private void createDataFiles(File path, TreeMap t_states, TreeMap t_zipDivs) {
        File sp = new File(path, statePath);
        sp.mkdirs();
        String[] keys = (String[]) t_states.keySet().toArray(new String[]{});
        writeKeyFile(sp, keys);
        for (int i = 0; i < keys.length; i++) {
            parent.updateProgressBar("Writing ZIP data files", 0, keys.length, i);
            writeZipFile(sp, keys[i], t_states);
        }
        File zp = new File(path, zipPath);
        zp.mkdirs();
        keys = (String[]) t_zipDivs.keySet().toArray(new String[]{});
        writeKeyFile(zp, keys);
        for (int i = 0; i < keys.length; i++) {
            parent.updateProgressBar("Writing state data files", 0, keys.length, i);
            writeZipFile(zp, keys[i], t_zipDivs);
        }
    }

    private void writeZipFile(File path, String key, TreeMap map) {

        try {
            TreeMap tm = (TreeMap) map.get(key);
            File f = new File(path, key + ".txt");
            FileWriter fw = new FileWriter(f);
            String[] keys = (String[]) tm.keySet().toArray(new String[]{});
            for (int i = 0; i < keys.length; i++) {
                String line = (String) tm.get(keys[i]);
                fw.write(line + parent.lineSep);
            }
            fw.close();
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private void writeKeyFile(File path, String[] array) {
        try {
            String keyString = join(array, "|");
            File f = new File(path, keyFileName);
            FileWriter fw = new FileWriter(f);
            fw.write(keyString + parent.lineSep);
            fw.close();
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private String join(String[] fields, String delim) {
        StringBuffer sb = new StringBuffer();
        for (int i = 0; i < fields.length; i++) {
            if (i > 0) {
                sb.append(delim);
            }
            sb.append(fields[i]);
        }
        return sb.toString();
    }

    public String[] getStateKeys() {
        String[] array = null;
        try {
            File f = new File(parent.zipPath + "/" + statePath + "/" + keyFileName);
            BufferedReader br = new BufferedReader(new FileReader(f));
            String line = br.readLine();
            br.close();
            array = line.split("\\|");
        } catch (Exception e) {
            e.printStackTrace();
        }
        return array;
    }

    public String[] getCityNames(String key) {
        return getKeys(states, key);
    }

    private void loadDataSection(TreeMap map, String key) {
        try {

            TreeMap section = new TreeMap();
            map.put(key, section);
            String sec = (map == states) ? statePath : zipPath;
            File f = new File(parent.zipPath + "/" + sec + "/" + key + ".txt");
            if (!f.exists()) {
                return;
            }
            BufferedReader br = new BufferedReader(new FileReader(f));
            String line;
            while ((line = br.readLine()) != null) {
                String[] array = line.split("\\|");
                if (array.length == 6) {
                    ZipRecord zr = new ZipRecord(array);
                    if (map == states) {
                        section.put(zr.city, zr);
                    } else {
                        section.put(zr.zip, zr);
                    }
                } else {
                    throw new Exception("Malformed record in data file \"" + f.getPath() + "\": \"" + line + "\"");
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private String[] getKeys(TreeMap map, String key) {
        if (!map.containsKey(key)) {
            loadDataSection(map, key);
        }
        String[] array = null;
        TreeMap sub = (TreeMap) map.get(key);
        if (sub != null) {
            array = (String[]) sub.keySet().toArray(new String[]{});
        }
        return array;
    }

    private ZipRecord getZipRecord(TreeMap map, String section, String key) {
        if (!map.containsKey(section)) {
            loadDataSection(map, section);
        }
        TreeMap tm = (TreeMap) map.get(section);
        ZipRecord zr = (ZipRecord) tm.get(key);
        return zr;
    }

    public ZipRecord getZipRecordFromCity(String state, String key) {
        return getZipRecord(states, state, key);
    }

    public ZipRecord getZipRecordFromZip(String key) {
        int n = Integer.parseInt(key.trim());
        String section = "" + (n / 1000);
        return getZipRecord(zipDivs, section, key);
    }
}
