/***************************************************************************
 *   Copyright (C) 2016 by Paul Lutus                                      *
 *   http://arachnoid.com/administration                                   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

/**
 *
 * @author  lutusp
 */
import javax.swing.*;
import java.io.*;
import java.net.*;
import java.text.*;
import java.awt.*;
import java.awt.event.*;
import javax.swing.UIManager.*;

final public class SatFinder extends javax.swing.JFrame {

    static final String VERSION = "5.5";
    static final int programBuild = 1422;
    static final String requiredJavaVersion = "1.7.0";
    static final public String programName = "SatFinder";
    // static final public String jarName = programName + ".jar";
    static public String copyright = "Satellite Finder is &copy; Copyright 2016, P. Lutus";
    static final public String zipDir = "ZipCodeData";
    static final public String zipFileName = "zip_database.txt";
    static final public String satListFileName = "satlist.txt";
    static final String programTitle = programName + " " + VERSION;
    public String userDir, userPath, configPath, zipPath;
    String fileSep, lineSep;
    private ProgramValues programValues;
    LocationHandler locationHandler;
    MagneticDeclination magDec;
    ZipRecord currentPosition = null;
    boolean propagate = true;

    /** Creates new form SatFinder */
    public SatFinder() {
        testJavaVersion();
        setDefaultCloseOperation(JFrame.DO_NOTHING_ON_CLOSE);

        setTitle("Satellite Finder" + " " + VERSION);

        URL url = getClass().getResource(programName + ".png");

        Image appIcon = new ImageIcon(url).getImage();
        setIconImage(appIcon);

        userDir = System.getProperty("user.home");
        fileSep = System.getProperty("file.separator");
        lineSep = System.getProperty("line.separator");
        userPath = userDir + fileSep + "." + programName;
        File f = new File(userPath);
        f.mkdirs();
        configPath = userPath + fileSep + programName + ".ini";
        programValues = new ProgramValues();
        programValues.readValues(configPath);
        zipPath = userPath + fileSep + zipDir;
        locationHandler = new LocationHandler(this);
        magDec = new MagneticDeclination();
        initComponents();
        jTabbedPane1.addTab("Help", new SatFinderHelp(this));
        showProgressBar(false);
        pack();
        setSize(programValues.db_ProgWindowSize);
        setLocation(programValues.db_ProgWindowLocation);
        setExtendedState(programValues.db_ProgWindowState);
        setVisible(true);
        f = new File(zipPath);
        if (!f.exists()) {
            locationHandler.buildDataFiles();
        } else {
            setup();
        }
    }

    public void setup() {
        stateComboBox.setModel(new DefaultComboBoxModel(locationHandler.getStateKeys()));
        if (programValues.db_directPosEntryMode) {
            setPositionFromComplex(new Complex(programValues.db_posLng, programValues.db_posLat));
        } else {
            loadPositionFromZipCode(programValues.db_zip);
        }
    }

    private void loadState(boolean resetCity) {
        programValues.db_directPosEntryMode = false;
        String key = (String) stateComboBox.getSelectedItem();
        cityComboBox.setModel(new DefaultComboBoxModel(locationHandler.getCityNames(key)));
        cityComboBox.setSelectedIndex(0);
        loadPositionFromCity();
    }

    private void loadPositionFromCity() {
        if (propagate) {
            programValues.db_directPosEntryMode = false;
            String key = (String) cityComboBox.getSelectedItem();
            String state = (String) stateComboBox.getSelectedItem();
            ZipRecord zr = locationHandler.getZipRecordFromCity(state, key);
            zipTextField.setText(zr.zip);
            setPositionFromZipRecord(zr);
        }
    }

    private void loadPositionFromZipCodeEntry(KeyEvent evt) {
        if (evt.getKeyChar() == KeyEvent.VK_ENTER) {
            programValues.db_directPosEntryMode = false;
            String zip = zipTextField.getText();
            loadPositionFromZipCode(zip);
        }
    }

    private void loadPositionFromZipCode(String zip) {
        if (zip.length() > 0) {
            zipTextField.setText(zip);
            ZipRecord zr = locationHandler.getZipRecordFromZip(zip);
            if (zr == null) {
                JOptionPane.showMessageDialog(this, "Please enter a valid ZIP code.", "Invalid ZIP Code entry: " + zip, JOptionPane.WARNING_MESSAGE);
            } else {
                propagate = false;
                stateComboBox.setSelectedItem(zr.state);
                cityComboBox.setSelectedItem(zr.city);
                propagate = true;
                setPositionFromZipRecord(zr);
            }
        }
    }

    private void setPositionFromZipRecord(ZipRecord zr) {
        currentPosition = zr;
        setPositionFromDoubles(zr.lat, zr.lng);
        ((SatList) rightPanel).computePositions();
    }

    private void setPositionFromComplex(Complex pos) {
        setPositionFromDoubles(pos.y(), pos.x());
        ((SatList) rightPanel).computePositions();
    }

    private void setPositionFromDoubles(double lat, double lng) {
        //String slat = df.format(Math.abs(lat));
        //String slng = df.format(Math.abs(lng));
        latTextField.setText(formatDegMin(lat));
        northRadioButton.setSelected(lat >= 0);
        southRadioButton.setSelected(lat < 0);
        lngTextField.setText(formatDegMin(lng));
        eastRadioButton.setSelected(lng >= 0);
        westRadioButton.setSelected(lng < 0);
        updateMagDecField(lat, lng);
    }

    public String formatDegMin(double a) {
        a = Math.abs(a);
        DecimalFormat df = new DecimalFormat("0.000");
        int deg = (int) a;
        double min = (a - deg) * 60.0;
        return deg + "° " + df.format(min) + "'";
    }

    public Complex getPositionFromForm() {
        Complex result = null;

        String sla = latTextField.getText();
        String sln = lngTextField.getText();
        if (sla.length() > 0 && sln.length() > 0) {
            double x = readDegMin(sln);
            double y = readDegMin(sla);
            if (westRadioButton.isSelected()) {
                x = -x;
            }
            if (southRadioButton.isSelected()) {
                y = -y;
            }
            result = new Complex(x, y);
            updateMagDecField(result.y(), result.x());
            setPositionFromDoubles(result.y(), result.x());
        }
        return result;
    }

    private double readDegMin(String s) {
        s = s.replaceAll("[^\\d|\\.]", " ");
        //System.out.println("[" + s + "]");
        String[] array = s.split("\\s+");
        double d = 0;
        if (array.length > 0) {
            d = Double.parseDouble(array[0]);
            if (array.length > 1) {
                d += Double.parseDouble(array[1]) / 60.0;
            }
        }
        return Math.abs(d);
    }

    private String letterSigned(double v, String neg, String pos) {
        return formatDegMin(v) + ((v < 0) ? neg : pos);
    }

    private void updateMagDecField(double lat, double lng) {
        DecimalFormat df = new DecimalFormat("#.00");
        double magDecl = magDec.computePoint(lat, lng);
        declTextField.setText(df.format(magDecl));
    }

    private void readDirectPosEntry(KeyEvent evt) {
        if (evt.getKeyChar() == KeyEvent.VK_ENTER) {
            readDirectPosEntry();
        }
    }

    private void readDirectPosEntry() {
        programValues.db_directPosEntryMode = true;
        ((SatList) rightPanel).computePositions();
    }

    private void testJavaVersion() {
        String vs = System.getProperty("java.version");
        boolean accept = convertVersionString(vs) >= convertVersionString(requiredJavaVersion);
        if (!accept) {
            JOptionPane.showMessageDialog(this, "Very Sorry!\n\n"
                    + programName + " " + VERSION + " requires a Java runtime engine (JRE)\n"
                    + "with a version of " + requiredJavaVersion + " or better.\n"
                    + "This machine has a version " + vs + " JRE.\n"
                    + "Please acquire the correct JRE at http://java.com.", "Wrong Java Runtime Version", JOptionPane.WARNING_MESSAGE);
            System.exit(0);
        }
    }

    private int convertVersionString(String version) {
        try {
            int len = version.length();
            int i = 0;
            while (i < len && !Character.isDigit(version.charAt(i))) {
                i++;
            }
            StringBuffer sb = new StringBuffer();
            char c = ' ';
            while (i < len && Character.isDigit((c = version.charAt(i))) || c == '.') {
                if (Character.isDigit(c)) {
                    sb.append(c);
                }
                i++;
            }
            int q = Integer.parseInt(sb.toString());
            while (q > 0 && q < 100) {
                q *= 10;
            }
            return q;
        } catch (Exception e) {
            return 0;
        }
    }

    public String currentLocation(boolean html) {
        String result = "";
        String lineBreak = (html)?"<br/>":"\n";
        if (!programValues.db_directPosEntryMode) {
            result = "Location: " + (String) cityComboBox.getSelectedItem()
                    + ", "
                    + currentPosition.county
                    + ", "
                    + (String) stateComboBox.getSelectedItem()
                    + " " + zipTextField.getText() + lineBreak;
        }
        Complex pos = getPositionFromForm();
        result += "Latitude " + letterSigned(pos.y(), "S", "N")
                + ", Longitude " + letterSigned(pos.x(), "W", "E");
        return result + lineBreak;
    }
    
    private void toClipboard() {
       ((SatList) rightPanel).toClipboard(); 
    }

    public void updateProgressBar(String prompt, int min, int max, int pos) {
        int v = 0;
        if (max - min > 0) {
            v = 100 * (pos - min) / (max - min);
        }
        progressBar.setString(prompt + " " + v + "%");
        progressBar.setMinimum(min);
        progressBar.setMaximum(max);
        progressBar.setValue(pos);
    }

    public void showProgressBar(boolean show) {
        progressBar.setVisible(show);
    }

    private void exitProgram() {
        programValues.db_zip = zipTextField.getText();
        Complex pos = getPositionFromForm();
        programValues.db_posLat = pos.y();
        programValues.db_posLng = pos.x();
        programValues.db_ProgWindowSize = getSize();
        programValues.db_ProgWindowLocation = getLocation();
        programValues.db_ProgWindowState = getExtendedState();
        programValues.writeValues(configPath);
        System.exit(0);
    }

    /** This method is called from within the constructor to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the Form Editor.
     */
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {
        java.awt.GridBagConstraints gridBagConstraints;

        latButtonGroup = new javax.swing.ButtonGroup();
        lngButtonGroup = new javax.swing.ButtonGroup();
        jPanel2 = new javax.swing.JPanel();
        progressBar = new javax.swing.JProgressBar();
        jTabbedPane1 = new javax.swing.JTabbedPane();
        jPanel1 = new javax.swing.JPanel();
        leftPanel = new javax.swing.JPanel();
        jPanel3 = new javax.swing.JPanel();
        jLabel1 = new javax.swing.JLabel();
        stateComboBox = new javax.swing.JComboBox();
        jLabel2 = new javax.swing.JLabel();
        cityComboBox = new javax.swing.JComboBox();
        jPanel4 = new javax.swing.JPanel();
        jLabel3 = new javax.swing.JLabel();
        zipTextField = new javax.swing.JTextField();
        jPanel5 = new javax.swing.JPanel();
        jLabel4 = new javax.swing.JLabel();
        latTextField = new javax.swing.JTextField();
        northRadioButton = new javax.swing.JRadioButton();
        southRadioButton = new javax.swing.JRadioButton();
        jLabel5 = new javax.swing.JLabel();
        lngTextField = new javax.swing.JTextField();
        westRadioButton = new javax.swing.JRadioButton();
        eastRadioButton = new javax.swing.JRadioButton();
        jLabel6 = new javax.swing.JLabel();
        declTextField = new javax.swing.JTextField();
        jPanel6 = new javax.swing.JPanel();
        copyButton = new javax.swing.JButton();
        quitButton = new javax.swing.JButton();
        rightPanel = new SatList(this);

        addWindowListener(new java.awt.event.WindowAdapter() {
            public void windowClosing(java.awt.event.WindowEvent evt) {
                exitForm(evt);
            }
        });

        jPanel2.setLayout(new java.awt.BorderLayout());

        progressBar.setStringPainted(true);
        jPanel2.add(progressBar, java.awt.BorderLayout.CENTER);

        getContentPane().add(jPanel2, java.awt.BorderLayout.SOUTH);

        jPanel1.setLayout(new java.awt.GridBagLayout());

        leftPanel.setBorder(javax.swing.BorderFactory.createTitledBorder("Choose a location:"));
        leftPanel.setLayout(new java.awt.GridBagLayout());

        jPanel3.setBorder(javax.swing.BorderFactory.createTitledBorder("By selecting a state and city:"));
        jPanel3.setToolTipText("First select a state, then a city");
        jPanel3.setLayout(new java.awt.GridBagLayout());

        jLabel1.setText("State:");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.insets = new java.awt.Insets(2, 2, 2, 2);
        jPanel3.add(jLabel1, gridBagConstraints);

        stateComboBox.setToolTipText("Click the arrow at the right to open this list");
        stateComboBox.addItemListener(new java.awt.event.ItemListener() {
            public void itemStateChanged(java.awt.event.ItemEvent evt) {
                stateComboBoxItemStateChanged(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        jPanel3.add(stateComboBox, gridBagConstraints);

        jLabel2.setText("City:");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.insets = new java.awt.Insets(2, 2, 2, 2);
        jPanel3.add(jLabel2, gridBagConstraints);

        cityComboBox.setToolTipText("Click the arrow at the right to open this list");
        cityComboBox.addItemListener(new java.awt.event.ItemListener() {
            public void itemStateChanged(java.awt.event.ItemEvent evt) {
                cityComboBoxItemStateChanged(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        jPanel3.add(cityComboBox, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTH;
        gridBagConstraints.weightx = 1.0;
        leftPanel.add(jPanel3, gridBagConstraints);

        jPanel4.setBorder(javax.swing.BorderFactory.createTitledBorder("Or by entering a ZIP code:"));
        jPanel4.setLayout(new java.awt.GridBagLayout());

        jLabel3.setText("ZIP Code:");
        jPanel4.add(jLabel3, new java.awt.GridBagConstraints());

        zipTextField.setToolTipText("Press the \"Enter\" key to accept your change");
        zipTextField.setMinimumSize(new java.awt.Dimension(4, 24));
        zipTextField.addKeyListener(new java.awt.event.KeyAdapter() {
            public void keyTyped(java.awt.event.KeyEvent evt) {
                zipTextFieldKeyTyped(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        jPanel4.add(zipTextField, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTH;
        gridBagConstraints.weightx = 1.0;
        leftPanel.add(jPanel4, gridBagConstraints);

        jPanel5.setBorder(javax.swing.BorderFactory.createTitledBorder("Or by entering a geographic position:"));
        jPanel5.setLayout(new java.awt.GridBagLayout());

        jLabel4.setText("Latitude:");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.insets = new java.awt.Insets(2, 2, 2, 2);
        jPanel5.add(jLabel4, gridBagConstraints);

        latTextField.setColumns(18);
        latTextField.setFont(new java.awt.Font("Monospaced", 0, 12));
        latTextField.setToolTipText("<html>Enter format: ddd.ddd [mm.mmm]<br>Press the \"Enter\" key to accept your entry</html>");
        latTextField.setMinimumSize(new java.awt.Dimension(112, 24));
        latTextField.addKeyListener(new java.awt.event.KeyAdapter() {
            public void keyPressed(java.awt.event.KeyEvent evt) {
                latTextFieldKeyPressed(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        jPanel5.add(latTextField, gridBagConstraints);

        latButtonGroup.add(northRadioButton);
        northRadioButton.setText("North");
        northRadioButton.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                northRadioButtonMouseClicked(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        jPanel5.add(northRadioButton, gridBagConstraints);

        latButtonGroup.add(southRadioButton);
        southRadioButton.setText("South");
        southRadioButton.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                southRadioButtonMouseClicked(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        jPanel5.add(southRadioButton, gridBagConstraints);

        jLabel5.setText("Longitude:");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTH;
        gridBagConstraints.insets = new java.awt.Insets(2, 2, 2, 2);
        jPanel5.add(jLabel5, gridBagConstraints);

        lngTextField.setColumns(18);
        lngTextField.setFont(new java.awt.Font("Monospaced", 0, 12));
        lngTextField.setToolTipText("<html>Enter format: ddd.ddd [mm.mmm]<br>Press the \"Enter\" key to accept your entry</html>");
        lngTextField.setMinimumSize(new java.awt.Dimension(112, 24));
        lngTextField.addKeyListener(new java.awt.event.KeyAdapter() {
            public void keyPressed(java.awt.event.KeyEvent evt) {
                lngTextFieldKeyPressed(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTH;
        gridBagConstraints.weightx = 1.0;
        jPanel5.add(lngTextField, gridBagConstraints);

        lngButtonGroup.add(westRadioButton);
        westRadioButton.setText("West");
        westRadioButton.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                westRadioButtonMouseClicked(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        jPanel5.add(westRadioButton, gridBagConstraints);

        lngButtonGroup.add(eastRadioButton);
        eastRadioButton.setText("East");
        eastRadioButton.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                eastRadioButtonMouseClicked(evt);
            }
        });
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 3;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        jPanel5.add(eastRadioButton, gridBagConstraints);

        jLabel6.setText("Mag. Decl.:");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTH;
        gridBagConstraints.weighty = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(2, 2, 2, 2);
        jPanel5.add(jLabel6, gridBagConstraints);

        declTextField.setBackground(java.awt.SystemColor.control);
        declTextField.setColumns(12);
        declTextField.setEditable(false);
        declTextField.setFont(new java.awt.Font("Monospaced", 0, 12));
        declTextField.setMinimumSize(new java.awt.Dimension(70, 24));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTH;
        gridBagConstraints.weightx = 1.0;
        jPanel5.add(declTextField, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTH;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.weighty = 1.0;
        leftPanel.add(jPanel5, gridBagConstraints);

        copyButton.setText("Copy to Clipboard");
        copyButton.setToolTipText("Copy data table to system clipbooard");
        copyButton.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseClicked(java.awt.event.MouseEvent evt) {
                copyButtonMouseClicked(evt);
            }
        });
        copyButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                copyButtonActionPerformed(evt);
            }
        });
        jPanel6.add(copyButton);

        quitButton.setText("Exit");
        quitButton.setToolTipText("Quit Satellite Finder");
        quitButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                quitButtonActionPerformed(evt);
            }
        });
        jPanel6.add(quitButton);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 3;
        leftPanel.add(jPanel6, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.weighty = 1.0;
        jPanel1.add(leftPanel, gridBagConstraints);

        rightPanel.setBorder(javax.swing.BorderFactory.createTitledBorder("Satellite position data"));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.weighty = 1.0;
        jPanel1.add(rightPanel, gridBagConstraints);

        jTabbedPane1.addTab("Program", jPanel1);

        getContentPane().add(jTabbedPane1, java.awt.BorderLayout.CENTER);

        pack();
    }// </editor-fold>//GEN-END:initComponents

    private void quitButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_quitButtonActionPerformed
        // Add your handling code here:
        exitProgram();
    }//GEN-LAST:event_quitButtonActionPerformed

    private void eastRadioButtonMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_eastRadioButtonMouseClicked
        // Add your handling code here:
        readDirectPosEntry();
    }//GEN-LAST:event_eastRadioButtonMouseClicked

    private void westRadioButtonMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_westRadioButtonMouseClicked
        // Add your handling code here:
        readDirectPosEntry();
    }//GEN-LAST:event_westRadioButtonMouseClicked

    private void southRadioButtonMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_southRadioButtonMouseClicked
        // Add your handling code here:
        readDirectPosEntry();
    }//GEN-LAST:event_southRadioButtonMouseClicked

    private void northRadioButtonMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_northRadioButtonMouseClicked
        // Add your handling code here:
        readDirectPosEntry();
    }//GEN-LAST:event_northRadioButtonMouseClicked

    private void lngTextFieldKeyPressed(java.awt.event.KeyEvent evt) {//GEN-FIRST:event_lngTextFieldKeyPressed
        // Add your handling code here:
        readDirectPosEntry(evt);
    }//GEN-LAST:event_lngTextFieldKeyPressed

    private void latTextFieldKeyPressed(java.awt.event.KeyEvent evt) {//GEN-FIRST:event_latTextFieldKeyPressed
        // Add your handling code here:
        readDirectPosEntry(evt);
    }//GEN-LAST:event_latTextFieldKeyPressed

    private void zipTextFieldKeyTyped(java.awt.event.KeyEvent evt) {//GEN-FIRST:event_zipTextFieldKeyTyped
        // Add your handling code here:
        loadPositionFromZipCodeEntry(evt);
    }//GEN-LAST:event_zipTextFieldKeyTyped

    private void cityComboBoxItemStateChanged(java.awt.event.ItemEvent evt) {//GEN-FIRST:event_cityComboBoxItemStateChanged
        // Add your handling code here:
        loadPositionFromCity();
    }//GEN-LAST:event_cityComboBoxItemStateChanged

    private void stateComboBoxItemStateChanged(java.awt.event.ItemEvent evt) {//GEN-FIRST:event_stateComboBoxItemStateChanged
        // Add your handling code here:
        loadState(false);
    }//GEN-LAST:event_stateComboBoxItemStateChanged

    /** Exit the Application */
    private void exitForm(java.awt.event.WindowEvent evt) {//GEN-FIRST:event_exitForm
        exitProgram();
    }//GEN-LAST:event_exitForm

    private void copyButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_copyButtonActionPerformed
        // TODO add your handling code here:
    }//GEN-LAST:event_copyButtonActionPerformed

    private void copyButtonMouseClicked(java.awt.event.MouseEvent evt) {//GEN-FIRST:event_copyButtonMouseClicked
        // TODO add your handling code here:
        toClipboard();
    }//GEN-LAST:event_copyButtonMouseClicked

    /**
     * @param args the command line arguments
     */
    public static void main(String args[]) {

        SwingUtilities.invokeLater(new Runnable() {

            public void run() {
                try {
                    for (LookAndFeelInfo info : UIManager.getInstalledLookAndFeels()) {
                        if ("Nimbus".equals(info.getName())) {
                            UIManager.setLookAndFeel(info.getClassName());
                            break;
                        }
                    }
                } catch (Exception e) {
                    // If Nimbus is not available, you can set the GUI to another look and feel.
                }
                System.setProperty("awt.useSystemAAFontSettings", "on");
                System.setProperty("swing.aatext", "true");
                new SatFinder().setVisible(true);
            }
        });

    }
    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JComboBox cityComboBox;
    protected javax.swing.JButton copyButton;
    private javax.swing.JTextField declTextField;
    private javax.swing.JRadioButton eastRadioButton;
    private javax.swing.JLabel jLabel1;
    private javax.swing.JLabel jLabel2;
    private javax.swing.JLabel jLabel3;
    private javax.swing.JLabel jLabel4;
    private javax.swing.JLabel jLabel5;
    private javax.swing.JLabel jLabel6;
    private javax.swing.JPanel jPanel1;
    private javax.swing.JPanel jPanel2;
    private javax.swing.JPanel jPanel3;
    private javax.swing.JPanel jPanel4;
    private javax.swing.JPanel jPanel5;
    private javax.swing.JPanel jPanel6;
    private javax.swing.JTabbedPane jTabbedPane1;
    private javax.swing.ButtonGroup latButtonGroup;
    private javax.swing.JTextField latTextField;
    private javax.swing.JPanel leftPanel;
    private javax.swing.ButtonGroup lngButtonGroup;
    private javax.swing.JTextField lngTextField;
    private javax.swing.JRadioButton northRadioButton;
    private javax.swing.JProgressBar progressBar;
    private javax.swing.JButton quitButton;
    private javax.swing.JPanel rightPanel;
    private javax.swing.JRadioButton southRadioButton;
    private javax.swing.JComboBox stateComboBox;
    private javax.swing.JRadioButton westRadioButton;
    private javax.swing.JTextField zipTextField;
    // End of variables declaration//GEN-END:variables
}
