
/***************************************************************************
 *   Copyright (C) 2016 by Paul Lutus                                      *
 *   http://arachnoid.com/administration                                   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

import java.io.*;
import java.net.*;
import java.util.*;
import java.text.*;
import javax.swing.*;
import java.awt.*;
import java.awt.datatransfer.*;

final public class SatList extends javax.swing.JPanel {

    SatFinder parent;
    String dataPath;
    String[] fieldNames = {
        "Name", "Longitude", "Azimuth True", "Azimuth Mag.", "Elevation", "Skew:0", "Skew:90"
    };
    ArrayList<String> sat_data;
    String css = "<style type=\"text/css\">td { text-align:right;}"
            + " .odd {background:#f8f8f8} .even{background:#f0f0f0}"
            + " .head {background:#e0ffe0;color:blue}"
            + " table {width:100%; background-color:black; border-right:1px solid black; border-bottom:1px solid black; }"
            + " td {border-left:1px solid gray; border-top: 1px solid gray; padding:2px;}"
            + " </style>";
    final double toDeg = 180.0 / Math.PI;
    final double toRad = Math.PI / 180.0;
    int fields = fieldNames.length;
    DecimalFormat df = new DecimalFormat("#0.0");
    Complex oldPos;
    StringBuilder tsvBuffer = null;

    /** Creates new form SatList */
    public SatList(SatFinder p) {
        parent = p;
        initComponents();
        sat_data = new ArrayList<String>();
        dataPath = parent.userPath + "/satlist.txt";
    }

    public void readFile() {
        try {
            unpackDataFile();
            File f = new File(dataPath);
            if (f.exists()) {
                BufferedReader br = new BufferedReader(new FileReader(f));
                String line;
                while ((line = br.readLine()) != null) {
                    sat_data.add(line);
                }
                br.close();
            } else {
                System.out.println("compute read fail");
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public void computePositions() {
        if (sat_data.size() < 10) {
            readFile();
            oldPos = null;
        }
        Complex pos = parent.getPositionFromForm();
        if (oldPos != null && pos.x() == oldPos.x() && pos.y() == oldPos.y()) {
            return;
        }
        oldPos = pos;
        // put up a temporary message
        satDisplay.setBackground(Color.white);
        satDisplay.setOpaque(true);
        satDisplay.setHorizontalAlignment(SwingConstants.CENTER);
        satDisplay.setText("<html><center><br/><b>One moment, please ...</b></center></html>");
        double decl = parent.magDec.computePoint(pos.y(), pos.x());
        ArrayList<String[]> v = new ArrayList<String[]>();
        String tableTag = "<center><table cellspacing='0' border='0'>";
        boolean inTable = false;
        int[] tabs = new int[fields];
        String[] header = fieldNames;  // new String[fields];
        try {
            for (String line : sat_data) {
                processLine(line, tabs, v, pos, decl);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        StringBuilder pageBuffer = new StringBuilder();
        tsvBuffer = new StringBuilder();
        tsvBuffer.append(parent.currentLocation(false));
        int line = 0;
        for (int i = 0; i < v.size(); i++) {

            String[] array = (String[]) v.get(i);
            if (array[0].matches(".*\\*\\*\\*.*")) {
                if (inTable) {
                    pageBuffer.append("</table></center>");
                }
                line = 0;
                String title = array[0].replaceFirst("[\\W]*", "");
                tsvBuffer.append("*** " + title + "\n");
                pageBuffer.append("<br/><center><b>" + title + "</b></center>");
                pageBuffer.append(tableTag);
                inTable = true;
                formatRecord(line, '\t',pageBuffer, tsvBuffer, header, tabs, true);
            } else if (array.length > 2) {
                formatRecord(line, '\t',pageBuffer, tsvBuffer, array, tabs, false);
            }
            line++;

        }
        if (inTable) {
            pageBuffer.append("</table>");
        }
        final StringBuilder output = new StringBuilder("<html>");
        output.append(css);
        output.append("<center>" + parent.currentLocation(true) + "</center>");
        output.append(pageBuffer);
        output.append("<br/>");
        output.append("<center>" + parent.copyright + "</center>");
        output.append("</html>");
        SwingUtilities.invokeLater(new Runnable() {

            public void run() {
                satDisplay.setText(output.toString());
                //listTextArea.setCaretPosition(0);
            }
        });

    }

    private void formatRecord(int line, char delim, StringBuilder pageBuffer, StringBuilder csvBuffer, String[] array, int[] tabs, boolean head) {
        //int tabLen = 0;
        if (head) {
            pageBuffer.append("<tr class=\"head\">");
        } else {
            if ((line % 2) == 0) {
                pageBuffer.append("<tr class=\"odd\">");
            } else {
                pageBuffer.append("<tr class=\"even\">");
            }
        }
        StringBuilder record = new StringBuilder();
        for (int i = 0; i < array.length && array[i] != null; i++) {
            StringBuilder lb = new StringBuilder();
            if (i == 0) {
                lb.append("<td width=\"140\">");
            } else {
                lb.append("<td>");
                record.append(delim);
            }
            lb.append(array[i]);
            record.append(array[i]);
            lb.append("</td>");
            pageBuffer.append(lb.toString());
        }
        csvBuffer.append(record.toString() + "\n");
        pageBuffer.append("</tr>");
        //return tabLen;
    }

    private void processLine(String line, int[] tabs, ArrayList<String[]> v, Complex pos, double decl) {
        line = line.trim();
        if (line.length() > 0) {
            String[] array = line.split(",");
            String[] record = new String[fields];

            if (array.length > 1) {
                updateRecord(record, 0, array[0]);
                double satLng = Double.parseDouble(array[1]);
                updateRecord(record, 1, letterSigned(satLng, " W", " E"));
                Complex spos = computePos(pos.y(), pos.x(), satLng);
                updateRecord(record, 2, df.format(spos.x()));
                double dd = (spos.x() - decl + 720) % 360;
                updateRecord(record, 3, df.format(dd));
                updateRecord(record, 4, df.format(spos.y()));
                double skew = computeSkew(pos.y(), pos.x(), satLng);
                updateRecord(record, 5, df.format(skew));
                updateRecord(record, 6, df.format(skew + 90));
            } else {
                record[0] = array[0];
            }
            v.add(record);
        }
    }

    private String letterSigned(double v, String neg, String pos) {
        return df.format(Math.abs(v)) + ((v < 0) ? neg : pos);
    }

    private void updateRecord(String[] record, int i, String data) {
        data = data.trim();
        record[i] = data;
    }

    private void unpackDataFile() {
        File f = new File(dataPath);
        if (!f.exists()) {
            try {
                URL url = parent.getClass().getResource(parent.satListFileName);
                BufferedReader br = new BufferedReader(new InputStreamReader(url.openStream()));
                FileWriter fw = new FileWriter(f);
                String line;
                while ((line = br.readLine()) != null) {
                    fw.write(line.trim() + parent.lineSep);
                }
                br.close();
                fw.close();
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
    }

    /*
     *
     * ComputePos(input: (earth) lat (deg), lng (deg),
     * (satellite) satlng (deg),
     * output: Complex(x = az true (deg), y = el (deg))
     *
     * az format: North = 0, East = 90, South = 180, West = 270
     * el format: Horizontal 0, Vertical 90
     *
     */

    private Complex computePos(double lat, double lng, double satLng) {
        double dlngr = (lng - satLng) * toRad;
        double az = Math.atan2(Math.sin(lat * toRad), Math.tan(dlngr)) * toDeg;
        az = (270.0 - az) / 360.0;
        az = az - Math.floor(az);
        az *= 360.0;
        double r1 = 6.6107; // ratio synchronous orbit/earth radius
        double clng = Math.cos(dlngr);
        double clat = Math.cos(lat * toRad);
        double v1 = r1 * clat * clng - 1.0;
        double v2 = r1 * Math.sqrt(1 - clat * clat * clng * clng);
        double el = Math.atan2(v1, v2) * toDeg;
        return new Complex(az, el);
    }

    private double computeSkew(double lat, double lng, double satLng) {
        double alat = Math.abs(lat);
        double dlngr = (satLng - lng) * toRad;
        double v = (Math.atan2(Math.tan(alat * toRad), Math.sin(dlngr)) * toDeg) - 90.0;
        return (lat < 0) ? -v : v;
    }

    private void handleMenu(java.awt.event.MouseEvent evt) {
        if (evt.isPopupTrigger()) {
            popupMenu.show(evt.getComponent(),
                    evt.getX(), evt.getY());
        }
    }

    public void toClipboard() {
        if (tsvBuffer != null) {
            StringSelection ss = new StringSelection(tsvBuffer.toString());
            Toolkit.getDefaultToolkit().getSystemClipboard().setContents(ss, null);
        }
    }

    /** This method is called from within the constructor to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the Form Editor.
     */
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {

        popupMenu = new javax.swing.JPopupMenu();
        copyMenuItem = new javax.swing.JMenuItem();
        jScrollPane1 = new javax.swing.JScrollPane();
        satDisplay = new javax.swing.JLabel();

        copyMenuItem.setText("Copy");
        copyMenuItem.setToolTipText("Copy list to system clipboard");
        copyMenuItem.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                copyMenuItemActionPerformed(evt);
            }
        });
        popupMenu.add(copyMenuItem);

        setLayout(new java.awt.BorderLayout());

        satDisplay.setText("jLabel1");
        satDisplay.setBorder(javax.swing.BorderFactory.createEmptyBorder(4, 4, 4, 4));
        jScrollPane1.setViewportView(satDisplay);

        add(jScrollPane1, java.awt.BorderLayout.CENTER);
    }// </editor-fold>//GEN-END:initComponents

    private void copyMenuItemActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_copyMenuItemActionPerformed
        // Add your handling code here:
        toClipboard();
    }//GEN-LAST:event_copyMenuItemActionPerformed
    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JMenuItem copyMenuItem;
    private javax.swing.JScrollPane jScrollPane1;
    private javax.swing.JPopupMenu popupMenu;
    protected javax.swing.JLabel satDisplay;
    // End of variables declaration//GEN-END:variables
}
